package gdxapp.object3d;

import java.util.ArrayList;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.util.Properties;
import java.util.UUID;

import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.utils.Disposable;

import dressing.mathutils.Direction;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Surface;
import dressing.mathutils.Vector4;
import dressing.model.Copyable;
import dressing.model.persistence.dpos.Object3D;
import geometry.CompoundShape;

public class WorldObject implements Copyable, Disposable{
	
	protected String name;					
	protected Vector4 realWorldDimension;
	protected Vector3 realWorldPosition;
	protected ObjectModel model;
	protected ObjectType type;				//to replace with subclassing
	//the model ID
	protected UUID uuid;
	protected UUID sceneID;
	protected transient boolean isSelected;		//to remove
	protected float rotation;                 // the rotation of the object along the Y axis in degrees
	protected Properties properties = new Properties();
	protected Direction selectedSurface;
	public ContainerBox containerBox;
	protected ArrayList<Surface> surfaces=new ArrayList<Surface>();
	protected boolean requireRefrech = true;
	public boolean moveable = true;
	public transient boolean drawInnerQuotations= false;
	public transient boolean drawOuterQuotation= false;
	public transient boolean hidden = false;
	public boolean staticobject;
	public boolean constraint = false;
	//
	public boolean activeWidth = false;
	private ArrayList<Vector3> vertices;

	
	public WorldObject() {	
		super();
		properties=new Properties();
		moveable = true;
	}
	

	public void calculateBounds() {
		if(containerBox == null)
			containerBox = new ContainerBox(this);
		containerBox.update();
	}
	
	public WorldObject clone() {
		WorldObject clone = null;
		if(this.model != null) {
			try {
				clone = WorldObjectFactory.getFactory().createObjectFromModel(model.clone());
			} catch (IOException e) {
				e.printStackTrace();
			}
		}else {
			clone = new WorldObject();
		}
		clone.setRealWorldDimension(getRealWorldDimension());
		clone.setRealWorldPosition(getRealWorldPosition());
		clone.setModel(this.getModel());
		clone.setUuid(getUuid());
		clone.setSceneId(getSceneId());
		return clone;
	}
	
	public String getName() {
		if(name != null) {
			return name;
		}else {
			return "undefined";
		}
	}
	public void setName(String name) {
		this.name = name;
	}
	public Vector4 getRealWorldDimension() {
		if(realWorldDimension == null)
			realWorldDimension = new Vector4();
		return realWorldDimension;
	}
	public void setRealWorldDimension(Vector4 realWorldDimension) {
		this.realWorldDimension = realWorldDimension;
	}
	
	public void setRealWorldDimension(Vector3 realWorldDimension) {
		if(this.realWorldDimension == null)
			this.realWorldDimension = new Vector4();
		this.realWorldDimension.x = realWorldDimension.x;
		this.realWorldDimension.y = realWorldDimension.y;
		this.realWorldDimension.z = realWorldDimension.z;
	}
	public Vector3 getRealWorldPosition() {
		if(realWorldPosition == null)
			realWorldPosition = new Vector3();
		return realWorldPosition;
	}
	public void setRealWorldPosition(Vector3 realWorldPosition) {
		this.realWorldPosition = realWorldPosition;
	}
	public ObjectModel getModel() {
		return model;
	}
	public void setModel(ObjectModel model) {
		this.model = model;
	}
	
	public ObjectType getType() {
		return type;
	}

	public void setType(ObjectType type) {
		this.type = type;
	}
	
	public UUID getUuid() {
		return uuid;
	}

	public void setUuid(UUID uuid) {
		this.uuid = uuid;
	}

	public boolean isSelected() {
		return isSelected;
	}

	public void setSelected(boolean isSelected) {
		this.isSelected = isSelected;
	}
	
		
	public boolean isMoveable() {
		return moveable;
	}


	public void setMoveable(boolean moveable) {
		this.moveable = moveable;
	}


	public float getRotation() {
		return rotation;
	}

	public void setRotation(float rotation) {
		float rot = (rotation + 360) %360;
		if(Math.abs(rot - this.rotation) > 0.01f) {
			this.rotation = rot;
			requireRefrech = true;
		}
	}

	public void setProperties(Properties properties) {
		this.properties = properties;
	}

	public Properties getProperties() {
		if(properties==null) {
			properties=new Properties();
		}
		return properties;
	}
	
	public Direction getSelectedSurface() {
		return selectedSurface;
	}

	public void setSelectedSurface(Direction selectedSurface) {
		this.selectedSurface = selectedSurface;
	}
	
	public void setSurfaces(ArrayList<Surface> surfaces) {
		this.surfaces = surfaces;
	}
	
	
	

	public boolean isHidden() {
		return hidden;
	}


	public void setHidden(boolean hidden) {
		this.hidden = hidden;
	}


	//thepersistable form 
	public <T extends Object3D> T toDPO() {
		return (T) new Object3D(this);
	}
	
	
	public boolean isDrawInnerQuotations() {
		return drawInnerQuotations;
	}


	public void setDrawInnerQuotations(boolean drawInnerQuotations) {
		this.drawInnerQuotations = drawInnerQuotations;
	}
	
//	

	public float getActiveWidth() {
		float width = Float.NaN;
		if(activeWidth) {
			width = Math.max(realWorldDimension.z, realWorldDimension.w);
		}else {
			width = realWorldDimension.x;
		} 
		return width;
	}
	



	public void setActiveWidth(boolean activeWidth) {
		this.activeWidth = activeWidth;
	}

	public ContainerBox getContainerBox() {
		if(containerBox == null)
			calculateBounds();
		return containerBox;
	}

	public boolean isDrawOuterQuotation() {
		return drawOuterQuotation;
	}


	public void setDrawOuterQuotation(boolean drawOuterQuotation) {
		this.drawOuterQuotation = drawOuterQuotation;
	}

	transient protected PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(
			this);

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(listener);
	}

	public void addPropertyChangeListener(String propertyName,
			PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(propertyName,
				listener);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(listener);
	}

	public void removePropertyChangeListener(String propertyName,
			PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(propertyName,
				listener);
	}

	public void firePropertyChange(String propertyName, Object oldValue,
			Object newValue) {
		
			propertyChangeSupport.firePropertyChange(propertyName, oldValue,
					newValue);
					
	}
	public PropertyChangeListener[] getListeners() {
		if(propertyChangeSupport==null) {
			return null;
		}
		return	propertyChangeSupport.getPropertyChangeListeners();
	}
	public void removeAllLsteners() {
		PropertyChangeListener[] listeners=getListeners();
		if(listeners==null) {
			return;
		}
		for(PropertyChangeListener listener:listeners) {
			propertyChangeSupport.removePropertyChangeListener(listener);
		}
	}
	
	public enum ObjectType{
		DEFINED, MODELED, POLY
	}
	
	//getters and setters
	public ArrayList<Surface> getSurfaces() {
		return surfaces;
	}

	public Surface getSurface(Direction direction) {
		Surface surface = null;
		switch(direction) {
		case TOP :
			surface = this.getContainerBox().getRight();
			break;
		case BOTTOM:
			surface = this.getContainerBox().getLeft();
			break;
		case LEFT:
			surface = this.getContainerBox().getBack();
			break;
		default:
			surface = this.getContainerBox().getFace();
			break;
		}
		return surface;
	}
	
	public WorldObject getClosest(ArrayList<WorldObject> array) {
		WorldObject closest = null;
		float distance = 10000000;
		for(WorldObject other: array) {
			if(other == this)
				continue;
			if(other.getRealWorldPosition() != null) {
				float newDistance = getRealWorldPosition().dst(other.getRealWorldPosition());
				if( newDistance < distance) {
					closest = other;
					distance = newDistance;
				}
			}
		}
		return closest;
	}
	
	public <T extends Object2D> T create2DObject(){
		return (T) new Object2D(this);
	}


	public ArrayList<Vector3> getVertices() {
		return vertices;
	}

	public boolean isStaticobject() {
		return staticobject;
	}

	public void setStaticobject(boolean staticobject) {
		this.staticobject = staticobject;
	}
	

	public UUID getSceneId() {
		return sceneID;
	}

	public void setSceneId(UUID modelId) {
		this.sceneID = modelId;
	}

	public void calculatePropertiesFromVertices() {
		this.realWorldPosition = MathUtilities.calculateCenter(vertices);
		this.realWorldDimension = MathUtilities.calculateContainerBoxDimension(vertices);
		this.rotation = 0;
	}
	
	public boolean isConstraint() {
		var isConstraint  =  properties.getOrDefault("constraint", false);
		if(isConstraint instanceof String)
			return Boolean.parseBoolean((String) isConstraint);
		return (boolean) isConstraint;
	}

	public void setConstraint(boolean constraint) {
		this.constraint = constraint;
	}

	@Override
	public String toString() {
		return "WorldObject [name=" + name + ", realWorldDimension=" + realWorldDimension + ", realWorldPosition="
				+ realWorldPosition + ", type=" + type + ", uuid=" + uuid + ", rotation=" + rotation + ", vertices="
				+ vertices + "]";
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((sceneID == null) ? 0 : sceneID.hashCode());
		result = prime * result + ((uuid == null) ? 0 : uuid.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		WorldObject other = (WorldObject) obj;
		if (sceneID == null) {
			if (other.sceneID != null)
				return false;
		} else if (!sceneID.equals(other.sceneID))
			return false;
		if (uuid == null) {
			if (other.uuid != null)
				return false;
		} else if (!uuid.equals(other.uuid))
			return false;
		return true;
	}



	//return a copy of this object dimensions
	public Vector4 getObjectDimention() {
		return realWorldDimension.cpy();
	}


	@Override
	public WorldObject copy() {
		WorldObject copy = new WorldObject();
		return copy;
	}
	




	@Override
	public void dispose() {
		
	}



	public boolean isRequireRefrech() {
		return requireRefrech;
	}



	public void setRequireRefrech(boolean requireRefrech) {
		this.requireRefrech = requireRefrech;
	}



	

}
