package dressing.controller;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import com.badlogic.gdx.math.Vector2;
import geometry.Polygon;

public class KitchenShapeBuilder {
	
	private float width = 4000;		//default value
	private float height = 4000;	//default value
	private float[] middlePoint = new float[] {0, 0, 0,			//dst to walk towards previous vertex, dst to walk towards next vertex, if != 0 include the opposite vertex
											   0, 0, 0,			
											   0, 0, 0,			
											   0, 0, 0};
	private final ArrayList<Vector2>[] corners = new ArrayList[] {new ArrayList<Vector2>(), new ArrayList<Vector2>(),
																	new ArrayList<Vector2>(), new ArrayList<Vector2>()};
	//bit order top, right, bottom, left
	private byte edges = 0;
	private PropertyChangeSupport pcs = new PropertyChangeSupport(this);
	
	private void parse() {
		Vector2 halfDims = new Vector2(this.width, this.height).scl(0.0005f);
		Vector2[] controlPoints = new Vector2[] {halfDims.cpy().scl(-1), halfDims.cpy().scl(1,-1), halfDims.cpy(), halfDims.cpy().scl(-1, 1)};
		float epsilon = 0.0001f;
		for(int i = 0; i <4; i++) {
			int offset = 3 * i;
			ArrayList<Vector2> points = corners[i];
			points.clear();
			if(middlePoint[offset] < epsilon || middlePoint[offset + 1] < epsilon) {
				points.add(controlPoints[i]);
			}else {
				int previous = (i + controlPoints.length -1) % controlPoints.length;
				previous = previous < 0 ? previous + controlPoints.length: previous;
				int next = (i+1) % controlPoints.length;
				Vector2 toPrevious = controlPoints[previous].cpy().sub(controlPoints[i]).nor().scl(middlePoint[offset]);
				Vector2 p = controlPoints[i].cpy().add(toPrevious);
				Vector2 toNext = controlPoints[next].cpy().sub(controlPoints[i]).nor().scl(middlePoint[offset + 1]);
				Vector2 n = controlPoints[i].cpy().add(toNext);
				points.add(p);
				if(middlePoint[offset + 2] > epsilon) {
					Vector2 opposite = controlPoints[i].cpy().add(toPrevious).add(toNext);
					points.add(opposite);
				}
				points.add(n);
			}
		}
	}
	
	public List<int[]> extractIndices(){
		String edgeConfigFlipped = String.format("%4s",Integer.toBinaryString(edges)).replace(' ', '0');
		String edgeConfig = "";
		for(int i = edgeConfigFlipped.length() -1; i >= 0; i--)
			edgeConfig += edgeConfigFlipped.charAt(i);
		ArrayList<int[]> wallIndices = new ArrayList<int[]>();
		do {
			ArrayList<String> tmpPolies = new ArrayList<String>();
			for(int i = 0; i < 4; i ++) {
				if(edgeConfig.charAt(i) == '0')
					continue;
				String indices = "" + i;
				int nextCharIndex = (i+1) % 4;
				boolean stop = false;
				do {
					stop = edgeConfig.charAt(nextCharIndex) == '0';
					if(!stop)
						indices += nextCharIndex;
					nextCharIndex = (nextCharIndex + 1) % 4;
				}while(nextCharIndex != i && !stop);
				tmpPolies.add(indices);
			}
			tmpPolies.sort(new Comparator<String>() {
				//descending sorting
				@Override
				public int compare(String o1, String o2) {
					return o2.length() - o1.length();
				}
			});
			int[] polygonIndices = tmpPolies.get(0).chars().map(Character::getNumericValue).toArray();
			wallIndices.add(polygonIndices);
			char[] configuration = edgeConfig.toCharArray();
			for(int c = 0; c < polygonIndices.length; c++) {
				configuration[polygonIndices[c]] = '0';
			}
			edgeConfig = new String(configuration);
		}while(!edgeConfig.equals("0000"));
		return wallIndices;
	}
	
	public ArrayList<Polygon> createWalls() {
		parse();
		ArrayList<Polygon> polies = new ArrayList<Polygon>();
		if(edges > 0) {
			List<int[]> indicesList = extractIndices();
			for(var indices: indicesList) {
				Polygon polygon = new Polygon();
				for(var index: indices) {
					polygon.getVertices().addAll(this.corners[index]);
					polygon.getVertices().addAll(this.corners[(index +1) % corners.length]);
				}
				polies.add(polygon);
			}
		}
		
		return polies;
	}
	
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		this.pcs.addPropertyChangeListener(listener);
	}
	
	public void clearListeners() {
		PropertyChangeListener[] listeners = pcs.getPropertyChangeListeners();
		for(var listener: listeners) {
			this.pcs.removePropertyChangeListener(listener);
		}
	}
	
	public float getWidth() {
		return width;
	}

	public float getHeight() {
		return height;
	}

	public void setWidth(float width) {
		float oldValue = this.width;
		this.width = width;
		pcs.firePropertyChange("width", oldValue, width);
	}
	
	public void setHeight(float height) {
		float oldValue = this.height;
		this.height = height;
		pcs.firePropertyChange("height", oldValue, height);

	}
	
	public byte getEdges() {
		return edges;
	}

	public void setEdges(byte edges) {
		byte oldValue = this.edges;
		this.edges = edges;
		pcs.firePropertyChange("edges", oldValue, edges);
	}
	
	

	public float[] getMiddlePoint() {
		return middlePoint;
	}
	
	public void setMiddlePoint(float[] points) {
		 middlePoint = points;
		 pcs.firePropertyChange(new PropertyChangeEvent(points, null, null, middlePoint));
	}

	public static void main(String...strings) {
		KitchenShapeBuilder kitchenShapeBuilder = new KitchenShapeBuilder();
		kitchenShapeBuilder.setWidth(6000);
		kitchenShapeBuilder.setHeight(4000);
		kitchenShapeBuilder.middlePoint = new float[] {100,100,0 ,
													   100,100,0,
													   100,100,0,
													   100,100,0};
		kitchenShapeBuilder.parse();
		byte edges = 0;
		edges |= 1;
		edges |= 1 << 3;
		edges |= 1 << 1;
		kitchenShapeBuilder.edges = edges;
		kitchenShapeBuilder.createWalls();
	}
	


}
