package gdxapp.fabs3d;

import java.util.ArrayList;
import java.util.Arrays;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.particles.values.MeshSpawnShapeValue.Triangle;
import com.badlogic.gdx.graphics.g3d.utils.MeshBuilder;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder.VertexInfo;
import com.badlogic.gdx.graphics.g3d.utils.shapebuilders.SphereShapeBuilder;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import gdxapp.assets.AssetsTextures;
import gdxapp.object3d.PolygonBuilder;
import gdxapp.shaders.PbrMaterial;
import geometry.ShapeBuilder;


//this class represents a volumetric bezier curve
public class BezierCurve {

	private final ArrayList<Vector3> controlPoints = new ArrayList<>();
	float segmentDensity;
	float radius;
	float step;
	
	private final ArrayList<Vector3> vertices = new ArrayList<>();
	private ArrayList<ArrayList<Vector3>> ringsList = new ArrayList<>();
	private Model model;

	public BezierCurve(float segmentDensity, Vector3... controlPoints) {
		this.segmentDensity = segmentDensity;
		this.controlPoints.addAll(Arrays.asList(controlPoints));
		//build();
	}

	private Vector3 at(float t) {
		ArrayList<Vector3> midPoints = new ArrayList<>(controlPoints);
		while (midPoints.size() > 2) {
			ArrayList<Vector3> tmpList = new ArrayList<>();
			for (int i = 0; i < midPoints.size() - 1; i++) {
				Vector3 v = midPoints.get(i + 1).cpy().sub(midPoints.get(i));
				tmpList.add(v.scl(t).add(midPoints.get(i)));
			}
			midPoints.clear();
			midPoints.addAll(tmpList);
		}
		Vector3 point = midPoints.get(1).cpy().scl(t).add(midPoints.get(0).cpy().scl((1 - t)));
		return point;
	}

	public void build() {
		calculateAxisPoints();
		//buildRings();
		//createModel();
		//createMesh();
	}

	private void createMesh() {
		int vertexNumber = this.ringsList.size() * this.ringsList.get(0).size() * 3;
		float[] vertices = new float[vertexNumber];
		int c = -1;
		for (ArrayList<Vector3> ring : ringsList) {
			for (Vector3 vertex : ring) {
				vertices[++c] = vertex.x;
				vertices[++c] = vertex.y;
				vertices[++c] = vertex.z;
			}
		}
		int ringSize = ringsList.get(0).size();
		int indicesNumber = 12 * (ringsList.size() - 1) * (ringSize);
		short[] indices = new short[indicesNumber];
		int k = -1;
		for (int i = 0; i < ringsList.size() - 1; i++) {
			int offsetFirstRing = i * ringSize;
			int offsetSecondRing = (i + 1) * ringSize;
			for (int j = 0; j < ringSize; j++) {
				int index00 = offsetFirstRing + j;
				int index01 = offsetFirstRing + ((j + 1) % ringSize);
				int index10 = offsetSecondRing + j;
				int index11 = offsetSecondRing + ((j + 1) % ringSize);
				indices[++k] = (short) index00;
				indices[++k] = (short) index01;
				indices[++k] = (short) index10;
				
				indices[++k] = (short) index00;
				indices[++k] = (short) index10;
				indices[++k] = (short) index01;
				
				
				indices[++k] = (short) index01;
				indices[++k] = (short) index11;
				indices[++k] = (short) index10;
				
				indices[++k] = (short) index01;
				indices[++k] = (short) index10;
				indices[++k] = (short) index11;
			}
		}
		PbrMaterial material =  AssetsTextures.getInstance().getMaterial("Aluminuim");//  new Material(ColorAttribute.createDiffuse(Color.GREEN));
		MeshBuilder meshBuilder = new MeshBuilder();
		meshBuilder.begin(VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal | VertexAttributes.Usage.TextureCoordinates);
		
		meshBuilder.addMesh(vertices, indices);
		Mesh mesh = meshBuilder.end();
		PolygonBuilder.getModelBuilder().begin();
		PolygonBuilder.getModelBuilder().part("torus", mesh, GL20.GL_TRIANGLES, material);
		this.model = PolygonBuilder.getModelBuilder().end();
	}
	
	public void createModel() {
		int attr = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal | VertexAttributes.Usage.TextureCoordinates;
		PbrMaterial material =  AssetsTextures.getInstance().getMaterial("Aluminuim");
		int vertexPerRing = ringsList.get(0).size(); 
		PolygonBuilder.getModelBuilder().begin();
		for(int i = 0; i < ringsList.size() -1; i++) {
			MeshPartBuilder meshPartBuilder = PolygonBuilder.getModelBuilder().part("part_" + i, GL20.GL_TRIANGLES, attr, material);
			ArrayList<Vector3> firstRing = ringsList.get(i);
			ArrayList<Vector3> secondRing = ringsList.get(i + 1);
			for(int k = 0; k < vertexPerRing; k++) {
				int i0 = k;
				int i1 = (k+1)%vertexPerRing;
				Vector3 v00 = firstRing.get(i0);
				Vector3 v01 = firstRing.get(i1);
				Vector3 v10 = secondRing.get(i0);
				Vector3 v11 = secondRing.get(i1);
				Vector3 normal = v10.cpy().sub(v00).crs(v01.cpy().sub(v00)).nor();
				float u0 = i * step;
				float u1 = (i + 1) * step;
				float v0 = (float) (Math.PI * 2 / vertexPerRing * k);
				float v1 = (float) (Math.PI * 2 / vertexPerRing * (k + 1));
				VertexInfo info0 = new VertexInfo();
				info0.setPos(v00);
				info0.setNor(normal);
				info0.setUV(u0, v0);
				VertexInfo info1 = new VertexInfo();
				info1.setPos(v01);
				info1.setNor(normal);
				info1.setUV(u0, v1);
				VertexInfo info2 = new VertexInfo();
				info2.setPos(v10);
				info2.setNor(normal);
				info2.setUV(u1, v0);
				VertexInfo info3 = new VertexInfo();
				info3.setPos(v11);
				info3.setNor(normal);
				info3.setUV(u1, v1);
				meshPartBuilder.triangle(info0, info1, info2);
				meshPartBuilder.triangle(info0, info2, info1);
				meshPartBuilder.triangle(info1, info2, info3);
				meshPartBuilder.triangle(info1, info3, info2);
			}
		}
		this.model = PolygonBuilder.getModelBuilder().end();
	}

	public void calculateAxisPoints() {
		step = Math.max(1.0f / segmentDensity, 0.001f);
		for (int i = 0; i <= segmentDensity; i++) {
			vertices.add(at(step * i));
		}
	}

	public void buildRings() {
		Vector3 normal = vertices.get(1).cpy().sub(vertices.get(0)).nor();
		ArrayList<Vector3> ring = ShapeBuilder.getCircleVertices(normal, radius, Math.min(Math.round(radius * 1800), 120));
		ArrayList<Vector3> firstRing = new ArrayList<Vector3>();
		
		for (Vector3 vertex : ring) {
			firstRing.add(vertex.cpy().add(vertices.get(0)));
		}
		ringsList.add(firstRing);
		Vector3 n;
		for (int i = 1; i < vertices.size() - 1; i++) {
			n = vertices.get(i + 1).cpy().sub(vertices.get(i)).nor();
			ArrayList<Vector3> circle = ShapeBuilder.rotateCircleVertices(ring, n);
			for (Vector3 vertex : circle) {
				vertex.add(vertices.get(i));
			}
			ringsList.add(circle);
		}
		//last ring
		n = vertices.get(vertices.size() -1).cpy().sub(vertices.get(vertices.size() - 2)).nor();
		ArrayList<Vector3> circle = ShapeBuilder.rotateCircleVertices(ring, n);
		for (Vector3 vertex : circle) {
			vertex.add(vertices.get(vertices.size() - 1));
		}
		ringsList.add(circle);
		
	}
	
	public ModelInstance[] getCurvePoints() {
		Model sphere = PolygonBuilder.getModelBuilder().createSphere(0.01f, 0.01f, 0.01f, 8, 8, new Material(), VertexAttributes.Usage.Position);
		
		ModelInstance[] points = new ModelInstance[vertices.size()];
		int c = 0;
		for(Vector3 vertex: vertices) {
			ModelInstance sphereInstance = new ModelInstance(sphere);
			sphereInstance.transform.setToTranslation(vertex);
			points[c++] = sphereInstance;
			
		}
		return points;
	}

	public ArrayList<Vector3> getVertices() {
		return vertices;
	}

	public ArrayList<ArrayList<Vector3>> getRingsList() {
		return ringsList;
	}

	public void setRingsList(ArrayList<ArrayList<Vector3>> ringsList) {
		this.ringsList = ringsList;
	}

	public Model getModel() {
		return model;
	}

}
