package gdxapp.assets;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.assets.AssetDescriptor;
import com.badlogic.gdx.assets.AssetErrorListener;
import com.badlogic.gdx.assets.AssetManager;
import com.badlogic.gdx.assets.loaders.resolvers.AbsoluteFileHandleResolver;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Cursor;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Pixmap;
import com.badlogic.gdx.graphics.Pixmap.Format;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.Texture.TextureFilter;
import com.badlogic.gdx.graphics.Texture.TextureWrap;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g2d.TextureAtlas;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.scenes.scene2d.ui.Skin;
import com.badlogic.gdx.utils.Disposable;
import dressing.config.WorkspaceConfiguration;
import dressing.io.IOUtilities;
import dressing.io.ImageLabelLoader;
import dressing.model.ModelProvider;
import dressing.model.ProjectManager;
import dressing.ui.engine3d.SceneTexture;
import dressing.ui.engine3d.SceneTexture.CATEGORY;
import gdxapp.shaders.PbrMaterial;
import param.MaterialType;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;

public class AssetsTextures implements Disposable, AssetErrorListener {

	private static final String TAG = AssetsTextures.class.getName();
	private AssetManager assetManager;
	private static AssetsTextures instance = new AssetsTextures();
	private Map<String, PbrMaterial> materials = new HashMap<String, PbrMaterial>();
	private final HashMap<String, Image> labelsImages = new HashMap<String, Image>(); // small size images
	private final HashMap<String, Image> previewImages = new HashMap<String, Image>(); // medium size images
	private final HashMap<String, Texture> icons = new HashMap<String, Texture>();
	private HashMap<String, Cursor> cursors;
	private BitmapFont uiFonts;
	private TextureAtlas uiAtlas;
	private Skin skin;
	private boolean loaded = false;
	private Texture defaultTexture;
	private TextureRegion defaultTextureRegion;

	private SceneTexture defaultSceneTexture;
	private ArrayList<PbrMaterial> pbrMaterials;
	private HashMap<String, String> wallStyles;

	private Image superCadLogo;

	public static Texture smallTexture;

	private AssetsTextures() {
	}

	public void init() {
		this.assetManager = new AssetManager(new AbsoluteFileHandleResolver());
		assetManager.setErrorListener(this);
		if (!loaded) {
			loadMaterialsfromEMF();
			loadMaterialsFromDisk();
			File pictures2D = new File(WorkspaceConfiguration.getDefaultConfDir() + "/pictures2d");
			File[] files2D = pictures2D.listFiles();
			for (File file : files2D) {
				SceneTexture sceneTexture = new SceneTexture(file.getAbsolutePath());
				sceneTexture.setCategory(CATEGORY.VIEW);
				ModelProvider.getTextures().add(sceneTexture);
			}
			File wall2D = new File(WorkspaceConfiguration.MATERIALSWall);
			File[] mur2D = wall2D.listFiles();
			for (File file : mur2D) {
				SceneTexture sceneTexture = new SceneTexture(file.getAbsolutePath());
				sceneTexture.setCategory(CATEGORY.WALL);
				ModelProvider.getTextures().add(sceneTexture);
			}
			loadDefaultTexture();
			if (!Boolean.parseBoolean(ProjectManager.getProperties().getProperty("performance.mode"))) {
				loadImages();
			}
			loadIcons();
			loaded = true;
		}
	}
	private void loadIcons() {
		File file = new File(WorkspaceConfiguration.ICONS);
		if (file.exists()) {
			for (File fileX : file.listFiles()) {
				Texture texture = new Texture(fileX.getAbsolutePath());
				icons.put(fileX.getName(), texture);
			}
		}
	}

	private void loadMaterialsFromDisk() {
		File jsonMtlFile = new File(WorkspaceConfiguration.MATERIALS_FILE);
		FileInputStream fis;
		try {
			fis = new FileInputStream(jsonMtlFile);
			long size = jsonMtlFile.length();
			byte[] buffer = new byte[(int) size];
			fis.read(buffer);
			String string = new String(buffer);
			JSONParser jsonParser = new JSONParser();
			JSONArray jsonArray = (JSONArray) jsonParser.parse(string);
			for (int i = 0; i < jsonArray.size(); i++) {
				JSONObject jsonObject = (JSONObject) jsonArray.get(i);
				String name = (String) jsonObject.get("name");
				CATEGORY category = CATEGORY.valueOf((String) jsonObject.get("category"));
				String albedoPath = (String) jsonObject.get("base_color_map");
				String normalPath = (String) jsonObject.get("normal_map");
				String roughnessPath = (String) jsonObject.get("roughness_map");
				String metalnessPath = (String) jsonObject.get("metalness_map");
				String aoPath = (String) jsonObject.get("ao");
				PbrMaterial mtl = new PbrMaterial(name);
				mtl.set(ColorAttribute.createDiffuse(Color.BLACK));
				if (albedoPath != null) {
					mtl.setAlbedoMapPath(albedoPath);
					SceneTexture sceneTexture = new SceneTexture(albedoPath);
					sceneTexture.setCategory(category);
					ModelProvider.getTextures().add(sceneTexture);
				}
				if (normalPath != null)
					mtl.setNormalMapPath(normalPath);
				if (roughnessPath != null)
					mtl.setRoughnessMapPath(roughnessPath);
				if (metalnessPath != null)
					mtl.setMetalnessMapPath(metalnessPath);
				if (aoPath != null)
					mtl.setAoMapPath(aoPath);
				ModelProvider.addPbrMaterial(mtl, category);
			}

		} catch (IOException | ParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	public void loadImages() {
		File pictures = new File(WorkspaceConfiguration.getDefaultConfDir() + "/pictures");
		File[] files = pictures.listFiles();
		for (File file : files) {
			SceneTexture sceneTexture = new SceneTexture(file.getAbsolutePath());
			sceneTexture.setCategory(CATEGORY.ELEMENT);
			ModelProvider.getTextures().add(sceneTexture);
		}

		Thread thread = new Thread(new ImageLabelLoader());
		thread.start();
	}

 

	public void loadDefaultTexture() {
		Pixmap pixmap = new Pixmap(1024, 1024, Format.RGB888);
		pixmap.setColor(0.9f, 0.9f, 0.9f, 1.0f); // light gray
		pixmap.fill();
		defaultTexture = new Texture(pixmap);
		defaultTexture.setWrap(TextureWrap.Repeat, TextureWrap.Repeat);
		defaultTexture.setFilter(TextureFilter.MipMapLinearNearest, TextureFilter.Linear);
		defaultSceneTexture = new SceneTexture("");
		defaultSceneTexture.setTexture(defaultTexture);
		defaultTextureRegion = new TextureRegion(defaultTexture);
		pixmap.dispose();
	}

	public SceneTexture getDefaultSceneTexture() {
		return defaultSceneTexture;
	}

	public void loadMaterialsfromEMF() {
		this.materials.clear();
		EList<MaterialType> materials = ModelProvider.getModelroot().getMaterialType().getMaterialTypes();
		if (materials != null) {
			for (MaterialType materialType : materials) {
				PbrMaterial pbrMaterial = new PbrMaterial(materialType);
				pbrMaterial.load();
				this.materials.put(materialType.getName(), pbrMaterial);
			}
		}
	}

	public static AssetsTextures getInstance() {
		return instance;
	}

	public TextureAtlas getUiAtlas() {
		return uiAtlas;
	}

	public void setUiAtlas(TextureAtlas uiAtlas) {
		this.uiAtlas = uiAtlas;
	}

	public Skin getSkin() {
		if (skin == null)
			loadSkin();
		return skin;
	}

	@Override
	public void error(AssetDescriptor asset, Throwable throwable) {

	}

	public TextureRegion getTextureRegion(String path) {
		TextureRegion region = null;
		try {
			SceneTexture sceneTexture = ModelProvider.getTexture(path);
			if (sceneTexture == null) {
				sceneTexture = new SceneTexture(path);
				ModelProvider.getTextures().add(sceneTexture);
			}
			region = sceneTexture.getTextureRegion();
			region.getTexture(); // just to invoke a null pointer exception when it s there
		} catch (Exception e) {
			region = AssetsTextures.getInstance().getDefaultTextureRegion();
		}
		return region;
	}

	public Texture getTexture(String path) {
		SceneTexture sceneTexture = ModelProvider.getTexture(path);
		if (sceneTexture == null) {
			File file = new File(path);
			if (!file.exists()) {
				String alternativePath = WorkspaceConfiguration.TEXTURES_FOLDER + path;
				File fileX = new File(alternativePath);
				if (!fileX.exists()) {
					return defaultTexture;
				} else {
					path = alternativePath;
				}
			}
			sceneTexture = new SceneTexture(path);
			ModelProvider.getTextures().add(sceneTexture);
		}
		return sceneTexture.getTexture();
	}

	public Texture loadTextureFromFile(String path, boolean mipmap) throws FileNotFoundException {
		File file = new File(path);
		if (!file.exists() || file.isDirectory()) {
			System.err.println("can not load " + path);
			return defaultTexture;
		}
		this.assetManager.load(path, Texture.class);
		this.assetManager.finishLoading();
		Texture texture = this.assetManager.get(path.replace("\\", "/"));
		texture.setWrap(TextureWrap.MirroredRepeat, TextureWrap.MirroredRepeat);
		texture.setFilter(TextureFilter.Linear, TextureFilter.Linear);
		if (mipmap) {
			texture.bind();
			Gdx.gl20.glGenerateMipmap(GL20.GL_TEXTURE_2D);
			texture.setFilter(TextureFilter.MipMapLinearNearest, TextureFilter.MipMapLinearNearest);
		}
		return texture;

	}

	public HashMap<String, Image> getLabelsImages() {
		return labelsImages;
	}

	public HashMap<String, Cursor> getCursors() {
		return cursors;
	}

	public PbrMaterial getMaterial(String key) {
		return this.materials.get(key).cpy();
	}

	public Texture getDefaultTexture() {
		return defaultTexture;
	}

	public ArrayList<PbrMaterial> getPbrMaterials() {
		return pbrMaterials;
	}

	public void setPbrMaterials(ArrayList<PbrMaterial> pbrMaterials) {
		this.pbrMaterials = pbrMaterials;
	}

	@Override
	public void dispose() {
		assetManager.dispose();
	}

	public void prepareMaterials() {
		for (PbrMaterial material : materials.values()) {
			if (!material.isReady()) {
				material.prepare();
			}
		}

	}

	public HashMap<String, Image> getPreviewImages() {
		return previewImages;
	}

	public void loadSkin() {
		skin = new Skin(Gdx.files.absolute(WorkspaceConfiguration.UI_SKIN));
	}

	public HashMap<String, String> getWallStyles() {
		if (wallStyles == null)
			loadWallStyles();
		return wallStyles;
	}

	private void loadWallStyles() {
		if (wallStyles == null) {
			wallStyles = new HashMap<String, String>();
		} else {
			wallStyles.clear();
		}
		File file = new File(WorkspaceConfiguration.WALL_STYLES);
		if (file.exists()) {
			try {
				FileInputStream fis = new FileInputStream(file);
				byte[] data = new byte[(int) file.length()];
				fis.read(data);
				String jsonString = new String(data);
				JSONArray array = (JSONArray) IOUtilities.getJsonParser().parse(jsonString);
				for (int i = 0; i < array.size(); i++) {
					JSONObject jsonObject = (JSONObject) array.get(i);
					String styleName = (String) jsonObject.get("name");
					String styleNormalPath = (String) jsonObject.get("normalMap");
					wallStyles.put(styleName, styleNormalPath);
				}
			} catch (Exception e) {
				System.err.println("could not load wall styles");
			}
		}
	}

	public AssetManager getAssetManager() {
		if (assetManager == null) {
			this.assetManager = new AssetManager(new AbsoluteFileHandleResolver());
			assetManager.setErrorListener(this);
		}
		return assetManager;
	}

	public TextureRegion getDefaultTextureRegion() {
		// TODO Auto-generated method stub
		return defaultTextureRegion;
	}

	public Texture getIcon(String key) {
		return icons.get(key);
	}

	public Image getSuperCadLogo() {
		if (superCadLogo == null) {
			superCadLogo = new Image(Display.getCurrent().getDefault(),
					WorkspaceConfiguration.ICONS + File.separator + "icon-supercad64.jpg");
		}
		return superCadLogo;
	}

}
