package api.mep;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eclipse.swt.events.MouseEvent;
import org.joml.Matrix4f;
import org.joml.Vector2f;
import org.joml.Vector3f;
import org.joml.Vector4f;

import com.badlogic.gdx.Application;
import com.sandmonkey.ttf_renderer.api.Text;

import api.backend.ApplicationContext;
import api.event.EventBus;
import api.event.EventHandler;
import api.event.EventType;
import api.event.MepEvent;
import api.graphics.Camera;
import api.graphics.IDrawable;
import api.graphics.ModelInstance;
import api.graphics.Shader;
import api.graphics.geometry.Line;
import api.graphics.geometry.ShapeDrawer;
import api.input.InputMultiplexer;
import api.input.MouseEventHandler;
import api.provider.FontProvider;

public class SurfaceController implements MouseEventHandler, EventHandler, IDrawable{

	private Matrix4f transform;
	private Matrix4f inverseTransform;

	private Vector3f upperBound;
	private Vector3f lowerBound;
	private List<ModelInstance> objects;
	private ModelInstance selection;
	private Camera camera;
	private List<Measure> measures;
	
	private ShapeDrawer shapeDrawer;
	private Text text;
	
	//input event attributes
	private Vector2f mouseOffset = new Vector2f();
	private boolean mouseDown = false;
	
	private Vector3f[] anchorPoints = {new Vector3f(), new Vector3f(), new Vector3f()};			//points used to reference the location of selected objected
	private ModelInstance target;
	
	
	private static SurfaceController controller;
	
	
	private SurfaceController() {
		super();
		shapeDrawer = new ShapeDrawer();
		shapeDrawer.prepare();
		text = new Text();
		subscribe(EventType.SURFACE_MODE);
		subscribe(EventType.OBJECT_ADDED);
	}
	
	

	@Override
	public boolean mouseDoubleClick(MouseEvent e) {
		remove();
		return false;
	}

	@Override
	public boolean mouseDown(MouseEvent e) {
		mouseOffset.set(e.x, e.y);
		mouseDown = true;
		return false;
	}

	@Override
	public boolean mouseUp(MouseEvent e) {
		mouseDown = false;
		if(target != null)
			displayLabelsForObject(target);

		return false;
	}

	@Override
	public boolean mouseMove(MouseEvent e) {
		if(mouseDown)
			return mouseDragged(e);
		return false;
	}

	@Override
	public boolean mouseScrolled(MouseEvent e) {
		return false;
	}

	@Override
	public boolean mouseDragged(MouseEvent e) {
		
		var selection = ObjectSelector.getSelector().getSelection();
		if(selection != null && selection instanceof ModelInstance) {
			ModelInstance target = (ModelInstance) selection;
			if(target.isMoveable()) {
				Vector2f currentMouse = new Vector2f(e.x, e.y);
				var offset = camera.unproject(new Vector3f(mouseOffset,0), ApplicationContext.getViewport());
				var current = camera.unproject(new Vector3f(currentMouse,0), ApplicationContext.getViewport());
				Vector3f translation = current.sub(offset).mul(1,-1,1);
				Vector4f localTranslation =  new Vector4f(translation,0).mul (transform).mul(1,1,0,1).mul(inverseTransform); 
				translation.set(localTranslation.x, localTranslation.y, localTranslation.z);
				target.translate(translation);
				generateQuotations();
				calculateAnchorPoints(target);
			}
		}
		mouseOffset.set(e.x, e.y);
		return false;
	}

	@Override
	public void handle(MepEvent event) {
		if(event.getType() == EventType.SURFACE_MODE) {
			transform= (Matrix4f) event.getDetail("transform");
			inverseTransform = new Matrix4f();
			transform.invert(inverseTransform);
			Vector4f lbound = new Vector4f((Vector3f) event.getDetail("lower bound"), 1).mul(transform);
			Vector4f hbound = new Vector4f((Vector3f) event.getDetail("upper bound"), 1).mul(transform);
			lowerBound = new Vector3f(lbound.x, lbound.y, lbound.z);
			upperBound = new Vector3f(hbound.x, hbound.y, hbound.z);
			objects = (List<ModelInstance>) event.getDetail("objects");
			ArrayList<ModelInstance> toRemove = new ArrayList<ModelInstance>();
			for(var object: objects) {
				Vector4f center4 = new Vector4f(object.getCenter(),1).mul(transform); 
				if(Math.abs(center4.x) > Math.abs(lowerBound.x) || Math.abs(center4.y) > Math.abs(lowerBound.y)) {
					toRemove.add(object);
					continue;
				}
				Vector4f size = new Vector4f(object.getSize(), 0).mul(transform);
				if(center4.sub(size.mul(0.5f)).z < -0.1f )
					toRemove.add(object);
			}
			HashMap<String, Object> details = new HashMap<String, Object>();
			details.put("elements", toRemove);
			EventBus.getInstance().notify(new MepEvent(EventType.HIDE_ELEMENT, details));
			generateQuotations();
			camera = (Camera) event.getDetail("camera");
			InputMultiplexer.getInstance().insertEventHandler(this, 0);
		}else if(event.getType() == EventType.OBJECT_ADDED) {
			generateQuotations();
		}
		
	}
	
	
	
	public Matrix4f getTransform() {
		return transform;
	}

	public void setTransform(Matrix4f transform) {
		this.transform = transform;
	}

	public Vector3f getUpperBound() {
		return upperBound;
	}

	public void setUpperBound(Vector3f upperBound) {
		this.upperBound = upperBound;
	}

	public Vector3f getLowerBound() {
		return lowerBound;
	}

	public void setLowerBound(Vector3f lowerBound) {
		this.lowerBound = lowerBound;
	}

	public List<ModelInstance> getObjects() {
		return objects;
	}

	public void setObjects(List<ModelInstance> objects) {
		this.objects = objects;
	}

	public ModelInstance getSelection() {
		return selection;
	}

	public void setSelection(ModelInstance selection) {
		this.selection = selection;
	}
	
	public static SurfaceController getController() {
		if(controller == null)
			controller = new SurfaceController();
		return controller;
	}

	public void remove() {
		objects.clear();
		measures.clear();
		HashMap<String, Object> details = new HashMap<String, Object>();
		details.put("drawable", this);
		EventBus.getInstance().notify(new MepEvent(EventType.REMOVE_DRAWABLE, details));
		InputMultiplexer.getInstance().removeEventHandler(this);
	}



	@Override
	public void draw(Shader shader) {
		var selection = ObjectSelector.getSelector().getSelection();
		if(selection != null && selection instanceof ModelInstance) {
			if(target == null || target != selection) {
				target = (ModelInstance) selection;
				calculateAnchorPoints(target);
			}
			drawDistanceLines(target)	;			
		}else {
			target = null;
		}
		drawQuotations();
		drawQuotableMeasures();
	}

	private void drawQuotableMeasures() {
		for(ModelInstance instance: objects) {
			if(!instance.isQuotable() || ObjectSelector.getSelector().getSelection() == instance)
				continue;
			Vector3f center = getLocalCenter(instance.getCenter());
			Vector3f hPoint= new Vector3f();
			if(center.x > 0) {
				hPoint = new Vector3f(upperBound);
			}else {
				hPoint = new Vector3f(lowerBound);
			}
			hPoint.y = center.y;
			Vector3f vPoint= new Vector3f();
			if(center.y > 0) {
				vPoint = new Vector3f(upperBound);
			}else {
				vPoint = new Vector3f(lowerBound);
			}
			vPoint.x = center.x;
			Vector3f opposingCorner = new Vector3f(hPoint.x, vPoint.y, center.z);
			Vector3f[] lines = new Vector3f[] {center, hPoint, center, vPoint, opposingCorner, hPoint, opposingCorner, vPoint};
			shapeDrawer.begin(camera);
			shapeDrawer.drawLines(lines, 1.2f, new Vector3f(), inverseTransform );
			shapeDrawer.end();
			
			
			Vector4f c0 = new Vector4f(new Vector3f(hPoint), 1).mul(inverseTransform);
			Vector4f c1 = new Vector4f(new Vector3f(vPoint),1).mul(inverseTransform);

			//display labels
			Vector3f hOnScreen0 = camera.project(new Vector3f(c0.x, c0.y, c0.z), ApplicationContext.getViewport());
			//centerOnScreen0.y = ApplicationContext.getHeight() - centerOnScreen0.y;
			Vector3f vOnScreen1 = camera.project(new Vector3f(c1.x, c1.y, c1.z), ApplicationContext.getViewport());
			//centerOnScreen1.y = ApplicationContext.getHeight() - centerOnScreen1.y;

			float hor = new Vector4f(new Vector3f(center).sub(hPoint), 0).mul(inverseTransform).mul(1,1,1,0).length();
			float vert = new Vector4f(new Vector3f(center).sub(vPoint), 0).mul(inverseTransform).mul(1,1,1,0).length();			
			
			text.render(FontProvider.getProvider().getMediumFont(), String.format("%.3f", vert),
					hOnScreen0.x, hOnScreen0.y, 1, 0, new Vector3f(0,0,1), camera);
			
			text.render(FontProvider.getProvider().getMediumFont(), String.format("%.3f", hor),
					vOnScreen1.x, vOnScreen1.y, 1, 0, new Vector3f(0,0,1), camera);

		}
		
	}



	private void drawQuotations() {
		for(Measure measure: measures) {
			Vector3f[] line = {measure.getV0(), measure.getV1()};
			shapeDrawer.begin(camera);
			shapeDrawer.drawLines(line, 1.2f, new Vector3f(), inverseTransform);
			shapeDrawer.end();
			Vector4f dir = new Vector4f(new Vector3f(measure.getV1()).sub(measure.getV0()),0).mul(inverseTransform);
			Vector3f screenDir =  camera.project(new Vector3f(dir.x, dir.y, dir.z), ApplicationContext.getViewport()).
					sub(camera.project(new Vector3f(), ApplicationContext.getViewport()));
   		 	float rotation = (float) Math.toDegrees(Math.atan2(screenDir.y, screenDir.x));
			Vector4f center = new Vector4f(measure.getCenter(),1).mul(inverseTransform);
			var screenLocation = camera.project(new Vector3f(center.x,  center.y, center.z), ApplicationContext.getViewport());
			
			text.render(FontProvider.getProvider().getMediumFont(), String.format("%.3f", measure.getValue()),
					screenLocation.x, screenLocation.y, 1, rotation, new Vector3f(),camera);
		}
	}



	public Vector3f getLocalCenter(Vector3f wolrdPoint) {
		Vector4f point4 = new Vector4f(new Vector3f(wolrdPoint),1).mul(transform);
		return new Vector3f(point4.x, point4.y, point4.z);
	}


	private void drawDistanceLines(ModelInstance target) {
		Vector3f[] lines = new Vector3f[] {anchorPoints[0], anchorPoints[1], anchorPoints[0], anchorPoints[2] };
		shapeDrawer.begin(camera);
		shapeDrawer.drawLines(lines, 1.2f, new Vector3f(), inverseTransform );
		shapeDrawer.end();
	}
	
	public void calculateAnchorPoints(ModelInstance anchored) {
		Vector3f center = getLocalCenter(anchored.getCenter());
		Vector3f hPoint= new Vector3f();
		if(center.x > 0) {
			hPoint = new Vector3f(upperBound);
		}else {
			hPoint = new Vector3f(lowerBound);
		}
		hPoint.y = center.y;
		Vector3f vPoint= new Vector3f();
		if(center.y > 0) {
			vPoint = new Vector3f(upperBound);
		}else {
			vPoint = new Vector3f(lowerBound);
		}
		vPoint.x = center.x;
		anchorPoints[0].set(center);
		anchorPoints[1].set(hPoint);
		anchorPoints[2].set(vPoint);
	}
	
	public void displayLabelsForObject(ModelInstance target) {
		Vector3f center = anchorPoints[0];
		Vector3f hPoint = anchorPoints[1];
		Vector3f vPoint = anchorPoints[2];
		Vector4f c0 = new Vector4f(new Vector3f(center).add(hPoint).mul(0.5f), 1).mul(inverseTransform);
		Vector4f c1 = new Vector4f(new Vector3f(center).add(vPoint).mul(0.5f), 1).mul(inverseTransform);

		//display labels
		Vector3f[] locationsOnScreen = new Vector3f[2];
		Vector3f centerOnScreen0 = camera.project(new Vector3f(c0.x, c0.y, c0.z), ApplicationContext.getViewport());
		centerOnScreen0.y = ApplicationContext.getHeight() - centerOnScreen0.y;
		Vector3f centerOnScreen1 = camera.project(new Vector3f(c1.x, c1.y, c1.z), ApplicationContext.getViewport());
		centerOnScreen1.y = ApplicationContext.getHeight() - centerOnScreen1.y;
		locationsOnScreen[0] = new Vector3f(centerOnScreen0.x, centerOnScreen0.y, centerOnScreen0.z);
		locationsOnScreen[1] = new Vector3f(centerOnScreen1.x, centerOnScreen1.y, centerOnScreen1.z);
		float hor = new Vector4f(new Vector3f(center).sub(hPoint), 0).mul(inverseTransform).mul(1,1,1,0).length();
		float vert = new Vector4f(new Vector3f(center).sub(vPoint), 0).mul(inverseTransform).mul(1,1,1,0).length();
		String[] labels= {String.format("%.0f", hor * 1000) ,String.format("%.0f", vert * 1000)};
		HashMap<String, Object> details = new HashMap<String, Object>();
		details.put("locations", locationsOnScreen);
		details.put("labels", labels );
		details.put("target", target);
		details.put("anchors", anchorPoints);
		EventBus.getInstance().notify(new MepEvent(EventType.DISPLAY_DISTANCE_LABELS, details));
	}

	
	private void generateQuotations() {
		if(objects == null)
			return;
		if(measures == null) {
			measures = new ArrayList<Measure>();
		}else {
			measures.clear();
		}
		
		ArrayList<Measure> leftMeasures = new ArrayList<Measure>();
		ArrayList<Measure> rightMeasures = new ArrayList<Measure>();
		ArrayList<Measure> topMeasures = new ArrayList<Measure>();
		ArrayList<Measure> bottomMeasures = new ArrayList<Measure>();

		
		for(ModelInstance instance: objects) {
			Vector3f center = getLocalCenter(instance.getCenter());
			Vector4f halfSize = new Vector4f(instance.getSize(), 0).mul(transform).mul(0.5f);			
			Measure xMeasure = new Measure(new Vector3f(center).sub(halfSize.x, 0, 0), new  Vector3f(center).add(halfSize.x, 0, 0));
			Measure yMeasure = new Measure(new Vector3f(center).sub(0,halfSize.y, 0), new  Vector3f(center).add(0,halfSize.y, 0)); 
			measures.add(xMeasure);
			measures.add(yMeasure);
			if(center.x > 0) {
				rightMeasures.add(yMeasure);
			}else {
				leftMeasures.add(yMeasure);
			}
			if(center.y > 0) {
				topMeasures.add(xMeasure);
			}else {
				bottomMeasures.add(xMeasure);
			}
		}
		//filter quotation{
		filter(bottomMeasures, new Vector3f(0, lowerBound.y, 0));
		filter(topMeasures, new Vector3f(0,upperBound.y,0));
		filter(leftMeasures, new Vector3f(lowerBound.x, 0, 0 ));
		filter(rightMeasures, new Vector3f(upperBound.x, 0,0));
		

	}



	private void filter(ArrayList<Measure> measures, Vector3f baseLevel) {
		measures.sort((m1, m2) -> {
			return (int) Math.signum(m1.getValue() - m2.getValue());
		});
		
		Vector3f levelStepper = new Vector3f(baseLevel).normalize().mul(0.1f);
		baseLevel.add(levelStepper);
		for(Measure measure: measures ) {
			Vector3f center = new Vector3f(measure.getV0()).add(measure.getV1()).mul(0.5f);
			Vector3f translation = new Vector3f(baseLevel).mul (new Vector3f(baseLevel).sub(center).dot(baseLevel) / baseLevel.lengthSquared());              
			System.out.println(new Vector3f(center).add(translation));
			measure.getV0().add(translation);
			measure.getV1().add(translation);
			baseLevel.add(levelStepper);
		}
		
	}





}
