package api.mep;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eclipse.swt.events.MouseEvent;
import org.joml.Matrix4f;
import org.joml.Vector2f;
import org.joml.Vector2i;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.lwjgl.opengl.GL45;

import com.sandmonkey.ttf_renderer.api.TextRenderer;

import api.backend.ApplicationContext;
import api.events.CustomEvent;
import api.events.EventBus;
import api.events.EventHandler;
import api.events.EventType;
import api.graphics.Camera;
import api.graphics.IDrawable;
import api.graphics.ModelInstance;
import api.graphics.RenderingMode;
import api.graphics.Shader;
import api.graphics.geometry.ShapeDrawer;
import api.input.InputMultiplexer;
import api.input.MouseEventHandler;
import api.provider.FontProvider;
import api.utils.MathUtilities;

public class SurfaceController implements MouseEventHandler, EventHandler, IDrawable{

	private Matrix4f transform; //the local to world transform
	private Matrix4f inverseTransform; //the world to local trnasform

	private Vector3f upperBound;
	private Vector3f lowerBound;
	private List<ModelInstance> objects = new ArrayList<ModelInstance>();
	private ModelInstance selection;
	private Camera camera;
	private List<Measure> measures;
	
	private ShapeDrawer shapeDrawer;
	private TextRenderer text;
	
	//input event attributes
	private Vector2f mouseOffset = new Vector2f();
	private boolean mouseDown = false;
	
	private Vector3f[] anchorPoints = {new Vector3f(), new Vector3f(), new Vector3f(), new Vector3f(), new Vector3f()};			//points used to reference the location of selected objected
	private ModelInstance target;
	
	private static SurfaceController controller;
	
	private SurfaceController() {
		super();
		shapeDrawer = new ShapeDrawer();
		shapeDrawer.prepare();
		text = new TextRenderer();
		subscribe(EventType.SURFACE_MODE);
		subscribe(EventType.OBJECT_ADDED);
	}
	
	

	@Override
	public boolean mouseDoubleClick(MouseEvent e) {
		remove();
		EventBus.getInstance().notify(new CustomEvent(EventType.END_ISOLATE, null));
		return false;
	}

	@Override
	public boolean mouseDown(MouseEvent e) {
		mouseOffset.set(e.x, e.y);
		mouseDown = true;
		ObjectSelector.getSelector().mouseDown(e);
		
		var selection =  ObjectSelector.getSelector().getSelection();
		if(selection instanceof ModelInstance) {
			ModelInstance instance = (ModelInstance) selection;
			if(instance.isMoveable()) {
				target = instance;
			}else {
				target = null;
			}
		}
		return false;
	}

	@Override
	public boolean mouseUp(MouseEvent e) {
		mouseDown = false;
		if(target != null) {
			drawAnchorsEditor(false);
		}else {
			EventBus.getInstance().notify(new CustomEvent(EventType.DISPLAY_ANCHORS, null));
		}
		return false;
	}

	@Override
	public boolean mouseMove(MouseEvent e) {
		if(mouseDown)
			return mouseDragged(e);
		return false;
	}

	@Override
	public boolean mouseScrolled(MouseEvent e) {
		return false;
	}

	@Override
	public boolean mouseDragged(MouseEvent e) {
		var selection = ObjectSelector.getSelector().getSelection();
		if(selection != null && selection instanceof ModelInstance) {
			ModelInstance target = (ModelInstance) selection;
			if(target.isMoveable()) {
				Vector2f currentMouse = new Vector2f(e.x, e.y);
				var offset = camera.unproject(new Vector3f(mouseOffset,0), ApplicationContext.getViewport());
				var current = camera.unproject(new Vector3f(currentMouse,0), ApplicationContext.getViewport());
				Vector3f translation = current.sub(offset).mul(1,-1,1);
				Vector4f localTranslation =  new Vector4f(translation,0).mul (inverseTransform).mul(1,1,0,1).mul(transform); 
				translation.set(localTranslation.x, localTranslation.y, localTranslation.z);
				target.translate(translation);
				calculateAnchorPoints(target);
				generateQuotations();
			}
		}
		mouseOffset.set(e.x, e.y);
		return false;
	}

	@Override
	public void handle(CustomEvent event) {
		if(event.getType() == EventType.SURFACE_MODE) {
			transform= (Matrix4f) event.getDetail("transform");
			inverseTransform = new Matrix4f();
			transform.invert(inverseTransform);
			Vector4f lbound = new Vector4f((Vector3f) event.getDetail("lower bound"), 1).mul(inverseTransform);
			Vector4f hbound = new Vector4f((Vector3f) event.getDetail("upper bound"), 1).mul(inverseTransform);
			lowerBound = new Vector3f(lbound.x, lbound.y, lbound.z);
			upperBound = new Vector3f(hbound.x, hbound.y, hbound.z);
			var objectsRecieved = (List<ModelInstance>) event.getDetail("objects");
			var toRemove = new ArrayList<ModelInstance>();
			var instance = (ModelInstance)event.getDetail("instance");
			for(var object: objectsRecieved) {
				if(object == instance) {
					objects.add(object);
					continue;
				}
				Vector4f center4 = new Vector4f(object.getCenter(),1).mul(inverseTransform); 
				if(Math.abs(center4.x) > Math.abs(lowerBound.x) || Math.abs(center4.y) > Math.abs(lowerBound.y)) {
					toRemove.add(object);
					continue;
				}
				Vector4f size = new Vector4f(object.getSize(), 0).mul(inverseTransform).mul(0.5f).absolute();
				if(Math.abs(center4.z - size.z) > 0.02f  ) {
					toRemove.add(object);
					continue;
				}
				object.setRenderingMode(RenderingMode.EDGES);
				objects.add(object);
			}
			HashMap<String, Object> details = new HashMap<String, Object>();
			details.put("elements", toRemove);
			EventBus.getInstance().notify(new CustomEvent(EventType.HIDE_ELEMENT, details));
			generateQuotations();
			camera = (Camera) event.getDetail("camera");
			InputMultiplexer.getInstance().insertEventHandler(this, 0);
		}else if(event.getType() == EventType.OBJECT_ADDED) {
			if(transform != null) {
				ModelInstance instance =  (ModelInstance) event.getDetail("object");
				this.objects.add(instance);
				generateQuotations();
			}
		}
	}
	
	public Matrix4f getTransform() {
		return transform;
	}

	public void setTransform(Matrix4f transform) {
		this.transform = transform;
	}

	public Vector3f getUpperBound() {
		return upperBound;
	}

	public void setUpperBound(Vector3f upperBound) {
		this.upperBound = upperBound;
	}

	public Vector3f getLowerBound() {
		return lowerBound;
	}

	public void setLowerBound(Vector3f lowerBound) {
		this.lowerBound = lowerBound;
	}

	public List<ModelInstance> getObjects() {
		return objects;
	}

	public void setObjects(List<ModelInstance> objects) {
		this.objects = objects;
	}

	public ModelInstance getSelection() {
		return selection;
	}

	public void setSelection(ModelInstance selection) {
		this.selection = selection;
	}
	
	public static SurfaceController getController() {
		if(controller == null)
			controller = new SurfaceController();
		return controller;
	}

	public void remove() {
		for(var obj: objects) {
			if(obj.isMoveable()) {
				obj.setRenderingMode(RenderingMode.FACETS);
			}else {
				obj.setRenderingMode(RenderingMode.WIRED_FACETS);
			}
		}
		objects.clear();
		measures.clear();
		HashMap<String, Object> details = new HashMap<String, Object>();
		details.put("drawable", this);
		EventBus.getInstance().notify(new CustomEvent(EventType.REMOVE_DRAWABLE, details));
		InputMultiplexer.getInstance().removeEventHandler(this);
	}

	@Override
	public void draw(Shader shader, Camera camera) {
		for(var instance: objects) {
			if(instance.isMoveable()) {
				calculateAnchorPoints(instance);
				boolean drawText =(instance == target)?false:true;
				drawDistanceLines(instance, false, drawText);
			}
		}
		drawQuotations();
		drawQuotableMeasures();
	}

	private void drawAnchorsEditor(boolean centered) {
		calculateAnchorPoints(target);
		HashMap<String, Object> details = new HashMap<String, Object>();
		Vector3f[] lines;
		if(centered) {
			lines = new Vector3f[] {anchorPoints[0], anchorPoints[3], anchorPoints[0], anchorPoints[4] };			
		}else {
			lines = new Vector3f[] {anchorPoints[1], anchorPoints[3], anchorPoints[2], anchorPoints[4] };			
		}	
		Vector3f centerH = new Vector3f(lines[0]).add(lines[1]).mul(0.5f);
		Vector3f centerV = new Vector3f(lines[2]).add(lines[3]).mul(0.5f);
		var worldH4 = new Vector4f(centerH,1.0f).mul(transform);
		var worldV4 = new Vector4f(centerV,1.0f).mul(transform);
		Vector3f worldH = new Vector3f(worldH4.x, worldH4.y, worldH4.z);
		Vector3f worldV = new Vector3f(worldV4.x, worldV4.y, worldV4.z);
		Vector3f screenH = camera.project(worldH, ApplicationContext.getViewport());
		Vector3f screenV = camera.project(worldV, ApplicationContext.getViewport());
		
		Vector3f vSeg = new Vector3f(lines[3]).sub(lines[2]).mul(1,1,0);
		var hSeg = new Vector3f(lines[1]).sub(lines[0]).mul(1,1,0);
		float dstH = new Vector4f(hSeg, 0).mul(transform).length() * 1000;
		float dstV = new Vector4f(vSeg, 0).mul(transform).length() * 1000;
		
		details.put("href", new Vector3f(lines[0]));
		details.put("hanchor", new Vector3f(lines[1]));
		details.put("vref", new Vector3f(lines[2]));
		details.put("vanchor", new Vector3f(lines[3]));
		details.put("target", target);
		details.put("hpos", new Vector2i((int)screenH.x, (int) screenH.y));
		details.put("vpos", new Vector2i((int)screenV.x, (int) screenV.y));
		details.put("hValue", String.format("%.0f", dstH));
		details.put("vValue", String.format("%.0f", dstV));
		
		
		EventBus.getInstance().notify(new CustomEvent(EventType.DISPLAY_ANCHORS, details));
	}



	private void drawQuotableMeasures() {
		for(ModelInstance instance: objects) {
			if(!instance.isQuotable() || ObjectSelector.getSelector().getSelection() == instance)
				continue;
			Vector3f center = getLocalCenter(instance.getCenter());
			Vector3f hPoint= new Vector3f();
			float rotationH = 0;
			float rotationV = 0;
			if(center.x > 0) {
				hPoint = new Vector3f(upperBound);
				 rotationH = -90;

			}else {
				hPoint = new Vector3f(lowerBound);
				rotationH = 90;
			}
			hPoint.y = center.y;
			hPoint.add(new Vector3f(hPoint ).sub(center).normalize().mul(0.1f) );
			Vector3f vPoint= new Vector3f();
			if(center.y > 0) {
				vPoint = new Vector3f(upperBound);
			}else {
				vPoint = new Vector3f(lowerBound);
			}
			vPoint.x = center.x;
			vPoint.add( new Vector3f(vPoint).sub(center).normalize().mul(0.1f)  );
			Vector3f opposingCorner = new Vector3f(hPoint.x, vPoint.y, center.z);
			Vector3f[] lines = new Vector3f[] {center, hPoint, center, vPoint, opposingCorner, hPoint, opposingCorner, vPoint};
			shapeDrawer.begin(camera);
			shapeDrawer.drawLines(lines, 1.2f, new Vector3f(), inverseTransform );
			shapeDrawer.end();
			
			Vector4f c0 = new Vector4f(new Vector3f(hPoint), 1).mul(inverseTransform);
			Vector4f c1 = new Vector4f(new Vector3f(vPoint),1).mul(inverseTransform);

			//display labels
			Vector3f hOnScreen0 = camera.project(new Vector3f(c0.x, c0.y, c0.z), ApplicationContext.getViewport());
			//centerOnScreen0.y = ApplicationContext.getHeight() - centerOnScreen0.y;
			Vector3f vOnScreen1 = camera.project(new Vector3f(c1.x, c1.y, c1.z), ApplicationContext.getViewport());
			//centerOnScreen1.y = ApplicationContext.getHeight() - centerOnScreen1.y;

			float hor = new Vector4f(new Vector3f(center).sub(hPoint), 0).mul(inverseTransform).mul(1,1,1,0).length();
			float vert = new Vector4f(new Vector3f(center).sub(vPoint), 0).mul(inverseTransform).mul(1,1,1,0).length();
			
			text.render(FontProvider.getProvider().getMediumFont(), String.format("%.3f", vert),
					hOnScreen0.x, hOnScreen0.y, 1, rotationH, new Vector3f(0,0,1));
			text.render(FontProvider.getProvider().getMediumFont(), String.format("%.3f", hor),
					vOnScreen1.x, vOnScreen1.y, 1, rotationV, new Vector3f(0,0,1));
		}	
	}
	
	

	private void drawQuotations() {
		for(Measure measure: measures) {
			var director = new Vector3f(measure.getV1()).sub(measure.getV0()).normalize();
			var normal = new Vector3f(); 
			director.rotateZ((float) Math.toRadians(90), normal);
			var diag = new Vector3f(normal).add(director).mul(0.02f);
			
			Vector3f[] line = {measure.getV0(), measure.getV1(),
//					measure.getO0(), measure.getV0(),
//					measure.getO1(), measure.getV1(),
					new Vector3f(measure.getV0()).add(diag), new Vector3f(measure.getV0()).sub(diag),
					new Vector3f(measure.getV1()).add(diag), new Vector3f(measure.getV1()).sub(diag)
			};
			shapeDrawer.begin(camera);
			shapeDrawer.drawLines(line, 1.2f, new Vector3f(0.3f,0.3f,0.6f), transform);
			shapeDrawer.end();
 			Vector4f dir = new Vector4f(new Vector3f(measure.getV1()).sub(measure.getV0()),0).mul(transform);
			Vector3f screenDir =  camera.project(new Vector3f(dir.x, dir.y, dir.z), ApplicationContext.getViewport()).
					sub(camera.project(new Vector3f(), ApplicationContext.getViewport()));
   		 	float rotation = (float) Math.toDegrees(Math.atan2(screenDir.y, screenDir.x));
			Vector4f center = new Vector4f(measure.getCenter(),1).mul(transform);
			Vector3f screenNor = new Vector3f();
			screenDir.rotateZ((float) Math.toRadians(90),screenNor);
			screenNor.normalize().mul(0.05f);
			var screenLocation = camera.project(new Vector3f(center.x + screenNor.x,  center.y + screenNor.y, center.z), ApplicationContext.getViewport());
			text.render(FontProvider.getProvider().getMediumFont(), String.format("%.0f", measure.getValue() * 1000),
					screenLocation.x, screenLocation.y, 1, rotation, new Vector3f());
		}
	}

	public Vector3f getLocalCenter(Vector3f wolrdPoint) {
		Vector4f point4 = new Vector4f(new Vector3f(wolrdPoint),1).mul(inverseTransform);
		return new Vector3f(point4.x, point4.y, point4.z);
	}


	private void drawDistanceLines(ModelInstance target,boolean centered, boolean drawText) {
		Vector3f[] lines;
		if(centered) {
			lines = new Vector3f[] {anchorPoints[0], anchorPoints[3], anchorPoints[0], anchorPoints[4] };			
		}else {
			lines = new Vector3f[] {anchorPoints[1], anchorPoints[3], anchorPoints[2], anchorPoints[4] };			
		}
		GL45.glDepthFunc(GL45.GL_ALWAYS);
		shapeDrawer.begin(camera);
		shapeDrawer.drawLines(lines, 1.2f, new Vector3f(), transform );
		shapeDrawer.end();		
		if(drawText) {
			Vector3f centerH = new Vector3f(lines[0]).add(lines[1]).mul(0.5f);
			Vector3f centerV = new Vector3f(lines[2]).add(lines[3]).mul(0.5f);
			var worldH4 = new Vector4f(centerH,1.0f).mul(transform);
			var worldV4 = new Vector4f(centerV,1.0f).mul(transform);
			Vector3f worldH = new Vector3f(worldH4.x, worldH4.y, worldH4.z);
			Vector3f worldV = new Vector3f(worldV4.x, worldV4.y, worldV4.z);
			Vector3f screenH = camera.project(worldH, ApplicationContext.getViewport());
			Vector3f screenV = camera.project(worldV, ApplicationContext.getViewport());
			Vector3f vSeg = new Vector3f(lines[3]).sub(lines[2]).mul(1,1,0);
			var hSeg = new Vector3f(lines[1]).sub(lines[0]).mul(1,1,0);
			float dstH = new Vector4f(hSeg, 0).mul(transform).length() * 1000;
			float dstV = new Vector4f(vSeg, 0).mul(transform).length() * 1000;
			text.render(FontProvider.getProvider().getBoldFont(), String.format("%.0f", dstH),
					screenH.x, screenH.y + 5, 1.0f, 0, new Vector3f(1.0f,0,0));
			text.render(FontProvider.getProvider().getBoldFont(), String.format("%.0f", dstV),
					screenV.x, screenV.y, 1, 0, new Vector3f(1.0f,0,0));
		}
		GL45.glDepthFunc(GL45.GL_LEQUAL);
	}
	
	public void calculateAnchorPoints(ModelInstance anchored) {
		Vector3f center = getLocalCenter(anchored.getCenter()).mul(1,1,0);
		Vector3f hPoint= new Vector3f();
		if(center.x > 0) {
			hPoint = new Vector3f(upperBound).mul(1,1,0);
		}else {
			hPoint = new Vector3f(lowerBound).mul(1,1,0);
		}
		hPoint.y = center.y;
		Vector3f vPoint= new Vector3f();
		if(center.y > 0) {
			vPoint = new Vector3f(upperBound).mul(1,1,0);
		}else {
			vPoint = new Vector3f(lowerBound).mul(1,1,0);
		}
		vPoint.x = center.x;
		Vector3f halfSize = anchored.getSize().mul(0.5f);
		Vector3f hEdge = new Vector3f(center).add(new Vector3f(hPoint).sub(center).mul(1,1,0).normalize(halfSize.x));
		Vector3f vEdge = new Vector3f(center).add(new Vector3f(vPoint).sub(center).mul(1,1,0).normalize(halfSize.y));
		anchorPoints[0].set(center);
		anchorPoints[1].set(hEdge);
		anchorPoints[2].set(vEdge);
		anchorPoints[3].set(hPoint);
		anchorPoints[4].set(vPoint);
	}
	
	private void generateQuotations() {
		if(objects == null)
			return;
		if(measures == null) {
			measures = new ArrayList<Measure>();
		}else {
			measures.clear();
		}
		ArrayList<Measure> leftMeasures = new ArrayList<Measure>();
		ArrayList<Measure> rightMeasures = new ArrayList<Measure>();
		ArrayList<Measure> topMeasures = new ArrayList<Measure>();
		ArrayList<Measure> bottomMeasures = new ArrayList<Measure>();
		for(ModelInstance instance: objects) {
			Vector3f center = getLocalCenter(instance.getCenter());
			Vector4f halfSize = new Vector4f(instance.getSize(), 0).mul(inverseTransform).mul(0.5f);			
			Measure xMeasure = new Measure(new Vector3f(center).sub(halfSize.x, 0, 0), new  Vector3f(center).add(halfSize.x, 0, 0));
			Measure yMeasure = new Measure(new Vector3f(center).sub(0,halfSize.y, 0), new  Vector3f(center).add(0,halfSize.y, 0)); 
			measures.add(xMeasure);
			measures.add(yMeasure);
			if(center.x > 0) {
				rightMeasures.add(yMeasure);
			}else {
				leftMeasures.add(yMeasure);
			}
			if(center.y > 0) {
				topMeasures.add(xMeasure);
			}else {
				bottomMeasures.add(xMeasure);
			}
		}
		this.measures.clear();
		//filter quotation{
		filter(bottomMeasures, new Vector3f(0, lowerBound.y, 0));
		filter(topMeasures, new Vector3f(0,upperBound.y,0));
		filter(leftMeasures, new Vector3f(lowerBound.x, 0, 0 ));
		filter(rightMeasures, new Vector3f(upperBound.x, 0,0));
	}

	private void filter(ArrayList<Measure> measures, Vector3f baseLevel) {
		if(measures.isEmpty())
			return;
		measures.sort((m1, m2) -> {
			return (int) Math.signum(m1.getValue() - m2.getValue());
		});
		Vector3f levelStepper = new Vector3f(baseLevel).normalize().mul(0.1f);
		baseLevel.add(levelStepper);
		//project on the base line
		for(Measure measure: measures ) {
			Vector3f center = new Vector3f(measure.getV0()).add(measure.getV1()).mul(0.5f);
			Vector3f delta = new Vector3f(baseLevel).sub(center);
			Vector3f translation = new Vector3f(baseLevel).mul(delta.dot(baseLevel)/baseLevel.lengthSquared());
			measure.translateBy(translation);
			measure.scaleBy(new Vector3f(1,1,0));
		}
		//remove redundunt
		removeRedundunt(measures);
		//solve intersection
		ArrayList<ArrayList<Measure>> groups = new ArrayList<ArrayList<Measure>>();
		groups.add(new ArrayList<Measure>(measures));
		boolean stop;
		do {
			stop = true;
			for(int g = 0; g <groups.size(); g++) {
				ArrayList<Measure> grp = groups.get(g);
				HashMap<Measure, Integer> intersection = new HashMap<Measure, Integer>();
				Vector3f[] tmpVerticesList = new Vector3f[4];
				for(int i =0; i < grp.size(); i++) {
					int score = 0;
					for(int j = 0; j < grp.size(); j++) {
						if(i == j)
							continue;
						tmpVerticesList[0] = grp.get(i).getV0();
						tmpVerticesList[1] = grp.get(i).getV1();
						tmpVerticesList[2] = grp.get(j).getV0();
						tmpVerticesList[3] = grp.get(j).getV1();
						var bounds = MathUtilities.getBoundaries(tmpVerticesList);
						float boundsLength = bounds[1].distance(bounds[0]);
						float sumOfLength = grp.get(i).len() + grp.get(j).len();
						if(boundsLength + 0.01f < sumOfLength)
							score++;
					}
					intersection.put( grp.get(i),score);
				}
				Measure biggestTroubleMaker = grp.get(0);
				int topScore = intersection.get(biggestTroubleMaker);
				for(int i = 1; i < grp.size(); i++) {
					int currentScore = intersection.get(grp.get(i));
					if( currentScore > topScore) {
						biggestTroubleMaker = grp.get(i);
						topScore = currentScore;
					}else if(currentScore == topScore) {
						if(grp.get(i).getValue() > biggestTroubleMaker.getValue()) {
							biggestTroubleMaker = grp.get(i);
						}
					}
				}
				if(topScore > 0) {
					grp.remove(biggestTroubleMaker);
					ArrayList<Measure> destinationGrp = null;
					if(g == groups.size() - 1) {
						destinationGrp = new ArrayList<Measure>();
						groups.add(destinationGrp);
					}else {
						destinationGrp = groups.get(g + 1);
					}
					destinationGrp.add(biggestTroubleMaker);
					stop = false;
					break;
				}
			}
		}while(!stop);
		Vector3f step = new Vector3f(baseLevel).normalize().mul(0.2f);
		Vector3f translation = new Vector3f();
		for(var grp: groups) {
			translation.add(step);
			for(var measure: grp) {
				measure.translateBy(translation);
				this.measures.add(measure);
			}
		}
		
		System.out.println(groups);
	}
	



	private void removeRedundunt(ArrayList<Measure> measures) {
		boolean stop;
		do {
			stop = true;
			for(int i = 0; i < measures.size() -1; i++) {
				var measure = measures.get(i);
				boolean redundunt = false;
				for(int j = i+1; j < measures.size(); j++) {
					if(measures.get(j).equals(measure)){
						redundunt = true;
						break;
					}
				}
				if(redundunt) {
					measures.remove(i);
					stop = false;
					break;
				}
			}
			
		}while(!stop);		
	}



	@Override
	public void drawVertices(Camera camera) {
		
	}



	@Override
	public void drawSilhouette(Camera camera) {
		
	}
}