package gdxapp.object3d;

import java.util.ArrayList;

import org.frs.rule_engin.FactsRealm;
import org.frs.rule_engin.NaiveRuleExecutor;
import org.frs.rule_engin.Rule;

import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;

import dressing.model.DesignObject3D;
import dressing.model.Piece2D;
import param.MechanicPublicParam;

public class ProfilePositioner {
	
	public enum SIDE{
		TOP, BOTTOM, LEFT, RIGHT
	}
	
	public enum ELEMENT_TYPE {
		HAUT, BAS, COLONNE, BLOC_TIROIR
	};
	private FactsRealm facts;
	private NaiveRuleExecutor ruleExecutor;
	private Rule existenceRule;
	private Rule basRule;
	private Rule hautRule;
	private Rule btRule;
	private Rule columnRule;
	
	public void init() {
		facts = new FactsRealm();
		createExistenceRule();
		createBasRule();
		createHautRule();
		createBtRule();
		createColumnRule();
		ruleExecutor = new NaiveRuleExecutor();
		ruleExecutor.addRules(existenceRule, basRule, hautRule,btRule, columnRule);
	}
	
	
	public Matrix4 calculateProfileTransform(Piece2D facade, KitchenElement element) {
		bind(facade, element);
		ruleExecutor.execute(facts);
		Vector3 size = facts.getFact("size", Vector3.class);
		Quaternion quaternion = new Quaternion();
		Vector3 translation = new Vector3();
		var profileDims =  element.getProfile().getDimension();
		Vector3 scale = new Vector3(size.x/profileDims.x, 1, size.z/profileDims.z);
		SIDE side = facts.getFact("side", SIDE.class);
		Vector3 facadeCenter = facade.getCenter().scl(0.001f);
		Vector3 facadeSize = facade.getSize().scl(0.001f);
		var profileHalfSize = element.getProfile().getDimension().cpy().scl(0.25f);
		
		switch (side) {
		case TOP:
			translation.set(facadeCenter.cpy().add(0, facadeSize.y/2.0f + profileHalfSize.y,0));
			quaternion.set(Vector3.Y, 180);
			break;
		case BOTTOM:
			translation.set(facadeCenter.cpy().sub(0, facadeSize.y/2.0f + profileHalfSize.y,0));
			quaternion.set(Vector3.Z, 180).mulLeft(new Quaternion(Vector3.Y, 180));
			break;
		default:
			throw new IllegalArgumentException("Unexpected value: " + side);
		}
		if(facade.getSize().x < facade.getSize().z) {
			quaternion.mulLeft(new Quaternion(Vector3.Y, -90));
		}
		var local = new Matrix4(translation, quaternion, scale);
		
		return local;
	}
		
	
	public void bind(Piece2D facade, KitchenElement parent) {
		facts.clear();
		facts.addFact("facade", facade);
		facts.addFact("element", parent);
		//check the orientation of the piece
		var size = facade.getSize().scl(0.001f);
		var center = facade.getCenter().scl(0.001f);
		Matrix4 pieceTransform = new Matrix4().setToTranslation(center).inv();
		if(size.x < size.z) {
			pieceTransform.set(new Vector3(0,0,-1), Vector3.Y, new Vector3(1,0,0), center.cpy()).inv();
			size.mul(pieceTransform).sub(new Vector3().mul(pieceTransform));
			size.x = Math.abs(size.x);
			size.y = Math.abs(size.y);
			size.z = Math.abs(size.z);
		}
		facts.addFact("transform", pieceTransform);
		facts.addFact("size", size);
		DesignObject3D design = parent.getDesignObject();
		var caissonType = design.getDesignCaissonType();
		ELEMENT_TYPE type = ELEMENT_TYPE.BLOC_TIROIR;
		if(caissonType.toUpperCase().contains("HAUT")) {
			type = ELEMENT_TYPE.HAUT;
		}else if(caissonType.toUpperCase().contains("BAS")) {
			type = ELEMENT_TYPE.BAS;
		}else if(caissonType.toUpperCase().contains("COL")) {
			type = ELEMENT_TYPE.COLONNE;
		}
		facts.addFact("element_type", type);
		MechanicPublicParam paramPorteDirection = null;
		try {
			paramPorteDirection = design.getElementPorteDirection();
			var value = paramPorteDirection.getTypedefelement();
			facts.addFact("porte_dir", value.getKey());
		}catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	


	public void createExistenceRule() {
		existenceRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				
			}
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				return true;
			}
		};
	}

	public void createBasRule() {
		 basRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				 facts.addFact("side", SIDE.TOP);
			}
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				var type = realm.getFact("element_type",ELEMENT_TYPE.class);
				return type!= null && type.equals(ELEMENT_TYPE.BAS);
			}
		};
	}
	
	public void createHautRule() {
		hautRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				 facts.addFact("side", SIDE.BOTTOM);
			}
			
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				var type = realm.getFact("element_type",ELEMENT_TYPE.class);
				return type!= null && type.equals(ELEMENT_TYPE.HAUT);
			}
		};
	}

	public void createBtRule() {
		 btRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				var facade = facts.getFact("facade", Piece2D.class);
				var element = facts.getFact("element", KitchenElement.class);
				var facades =  new ArrayList<Piece2D>( element.getDesignObject().getListPieces().stream().filter( piece -> 
					piece.getPiecetype().isFacade()).toList());
				facades.sort( (p1, p2) -> {
					return (int) Math.round(p1.getYPosABS() - p2.getYPosABS());
				});
				int index = facades.indexOf(facade);
				SIDE side = SIDE.BOTTOM;
				if(index == 0  || index == facades.size() - 1)
					 side = SIDE.TOP;
				 facts.addFact("side", side);
			}
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				var type = realm.getFact("element_type",ELEMENT_TYPE.class);
				return type!= null && type.equals(ELEMENT_TYPE.BLOC_TIROIR);
			}
		};
	}

	public void createColumnRule() {
		columnRule = new Rule() {
			
			@Override
			public void execute(FactsRealm realm) {
				// TODO Auto-generated method stub
				
			}
			
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				var type = realm.getFact("element_type",ELEMENT_TYPE.class);
				return type!= null && type.equals(ELEMENT_TYPE.COLONNE);
			}
		};
	}
	
	

}
