package gdxapp.object3d;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.frs.supercad.parametric_model.api.core.DeepPolyMesh;
import com.frs.supercad.parametric_model.api.core.IndexedMesh;
import com.frs.supercad.parametric_model.api.core.Mesh.MESH_TYPE;
import com.frs.supercad.parametric_model.api.core.ParametricModel;
import com.frs.supercad.parametric_model.api.core.ParametricModelCompiler;

import gdxapp.shaders.PbrMaterial;

public class ParamertricModelFactory {

	private ParametricModelCompiler compiler;

	private DeepPolyBuilder deepPolyBuilder = new DeepPolyBuilder();
	private IndexedMeshBuilder indexedMeshBuilder = new IndexedMeshBuilder();

	public ParamertricModelFactory(ParametricModelCompiler compiler) {
		super();
		this.compiler = compiler;
	}

	public void buildDeepPoly(ParametricModel model, Map<String, Object> args, ModelBuilder modelBuilder,
			Matrix4 parentTransform) {
		compiler.compileModel(model, args);

		Vector3 translation = new Vector3(model.getTranslation());
		Vector3 scale = new Vector3(model.getScale());
		float[] rotation = model.getRotation();
		Vector3 rotationAxis = new Vector3(rotation);
		Matrix4 transform = new Matrix4().setToTranslation(translation)
				.mul(new Matrix4().setToRotation(rotationAxis, rotation[3])).mul(new Matrix4().setToScaling(scale));
		transform.mulLeft(parentTransform);

		Float l = Float.parseFloat((String) args.get("l"));
		Float h = Float.parseFloat((String) args.get("h"));
		Float depth = Float.parseFloat((String) args.get("d"));
		Vector3 size = new Vector3(l, h, depth); // used for uv coords generation
		for (var part : model.getParts()) {
			PbrMaterial material = (PbrMaterial) args.get(part.getMaterialRef());
			for (var mesh : part.getMeshes()) {
				if (mesh.getType() == MESH_TYPE.POLY) {
					Vector3 meshSize = new Vector3(size);
					meshSize.z = ((DeepPolyMesh) mesh).getDepth();
					ArrayList<Vector2> vertices = assembleVertices(mesh.getVertexData());
					deepPolyBuilder.buildPoly(part.getName(), vertices, meshSize, transform, material, modelBuilder);
				}else if (mesh.getType() == MESH_TYPE.ANNULAR_POLY) {
					Vector3 meshSize = new Vector3(size);
					meshSize.z = ((DeepPolyMesh) mesh).getDepth();
					ArrayList<Vector2> vertices = assembleVertices(mesh.getVertexData());
					ArrayList<Vector2> holeVertice = assembleVertices(mesh.getHoleVertexData());
					deepPolyBuilder.buildAnnularPoly(part.getName(), vertices, holeVertice, meshSize, transform, material, modelBuilder);
				}
			}
		}
	}
	
	public void buildIndexedMesh(ParametricModel model, HashMap<String, Object> args, ModelBuilder modelBuilder,
			Matrix4 parentTransform) {
		compiler.compileModel(model, args);

		Vector3 translation = new Vector3(model.getTranslation());
		Vector3 scale = new Vector3(model.getScale());
		float[] rotation = model.getRotation();
		Vector3 rotationAxis = new Vector3(rotation);
		Matrix4 transform = new Matrix4().mul(new Matrix4().setToRotation(rotationAxis, rotation[3]))
				.mul(new Matrix4().setToScaling(scale)).setToTranslation(translation);
		transform.mulLeft(parentTransform);

		Float l = Float.parseFloat((String) args.get("l"));
		Float h = Float.parseFloat((String) args.get("h"));
		Float depth = Float.parseFloat((String) args.get("d"));
		Vector3 size = new Vector3(l, h, depth); // used for uv coords generation
		for (var part : model.getParts()) {
			PbrMaterial material = (PbrMaterial) args.get(part.getMaterialRef());
			for (var meshX : part.getMeshes()) {
				if (meshX.getType() == MESH_TYPE.QUADS) {
					IndexedMesh mesh = (IndexedMesh) meshX;
					ArrayList<Vector3> vertices = assembleVertices3D(mesh.getVertexData());
					Vector3[] vertexData = vertices.toArray(new Vector3[0]);
					indexedMeshBuilder.buildQuads(part.getName(), vertexData, mesh.getIndices(), size, transform, material, modelBuilder);
				}
			}
		}		
	}
	public ArrayList<Vector2> assembleVertices(float[] vertexData){
		ArrayList<Vector2> vertices = new ArrayList<Vector2>();
		Vector2 tmp = new Vector2();
		for (int i = 0; i < vertexData.length; i++) {
			int index = (i) % 2;
			if (index == 0) {
				tmp.x = vertexData[i];
			} else {
				tmp.y = vertexData[i];
				vertices.add(tmp.cpy());
			}
		}
		return vertices;
	}
	
	public ArrayList<Vector3> assembleVertices3D(float[] vertexData){
		ArrayList<Vector3> vertices = new ArrayList<Vector3>();
		Vector3 tmp = new Vector3();
		for (int i = 0; i < vertexData.length; i++) {
			int index = (i) % 3;
			if (index == 0) {
				tmp.x = vertexData[i];
			} else if(index == 1){
				tmp.y = vertexData[i];
			}else {
				tmp.z = vertexData[i];
				vertices.add(tmp.cpy());
			}
		}
		return vertices;
	}


}
