package gdxapp.object3d;

import java.util.ArrayList;
import java.util.List;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelBatch;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.Vector2;
import dressing.mathutils.MathUtilities;
import gdxapp.assets.AssetFont;
import gdxapp.quotation.Object3DQuotation;

import org.frs.svg.LineSegment;

public class Object3D extends ModelInstance {

	private WorldObject worldObject;
	private final Vector3 error = new Vector3(0, 0, 0);
	private final RotatedBoundingBox rotatedBoundingBox = new RotatedBoundingBox();
	private Vector3 dimensions = new Vector3();
	private Model sketchModel;
	private Object3D sketchInstance;
	private float optimalVolumeAngle;
	private static ShapeRenderer shapeRenderer;
	private List<Object3D> addons = new ArrayList<Object3D>();
	private Vector3 pivot = new Vector3();

	private Vector3 worldPosition = new Vector3();

	public Object3D(Model model) {
		super(model);
		transform = new Matrix4().idt();
		calculateProperties();
	}

	public Object3D(WorldObject worldObject) {
		super(worldObject.getModel().getModel());
		this.worldObject = worldObject;
		calculateProperties();
		Vector3 boxSize = new Vector3();
		rotatedBoundingBox.getDimensions(boxSize);
		Vector3 scale = new Vector3(dimensions.x / boxSize.x, dimensions.y / boxSize.y, dimensions.z / boxSize.z);
		Vector3 center = new Vector3();
		rotatedBoundingBox.getCenter(center);
		Vector3 translationToOrigin = center.cpy().scl(-1);
		Matrix4 scaling = new Matrix4().setToScaling(scale);
		Matrix4 rotation = new Matrix4().setToRotation(Vector3.Y, worldObject.getRotation());
		transform = new Matrix4().setToTranslation(worldPosition)
				.mul(new Matrix4().setToTranslation(this.pivot.cpy().scl(-1))).mul(rotation)
				.mul(new Matrix4().setToTranslation(this.pivot)).mul(scaling)
				.mul(new Matrix4().setToTranslation(translationToOrigin));
		if (worldObject instanceof KitchenElement)
			createSketch();
	}

	private void createSketch() {
		this.sketchModel = DesignModelFactory.getFactory().createSketchFromVertices((KitchenElement) worldObject);
		this.sketchInstance = new Object3D(sketchModel);
		this.sketchInstance.transform.set(transform);
	}

	public void updatePosition() {
		calculateProperties();
		Vector3 boxSize = new Vector3();
		rotatedBoundingBox.getDimensions(boxSize);
		Vector3 scale = new Vector3(dimensions.x / boxSize.x, dimensions.y / boxSize.y, dimensions.z / boxSize.z);
		Vector3 center = new Vector3();
		rotatedBoundingBox.getCenter(center);
		Matrix4 scaling = new Matrix4().setToScaling(scale);
		Matrix4 rotation = new Matrix4().setToRotation(Vector3.Y, worldObject.getRotation());
		if (worldObject instanceof GeometryObject) {
			transform = new Matrix4().idt();
		} else {
			transform = new Matrix4().setToTranslation(worldPosition).mul(rotation).mul(scaling)
					.mul(new Matrix4().setToTranslation(pivot.cpy().scl(-1)));
		}
		rotatedBoundingBox.setTransform(transform.cpy().mul(rotatedBoundingBox.getTransform()));
		if (sketchInstance != null) {
			sketchInstance.transform.set(transform);
		}
	}

	public WorldObject getWorldObject() {
		return worldObject;
	}

	public void setWorldObject(WorldObject worldObject) {
		this.worldObject = worldObject;
	}

	public Vector3 getExteriorDimension(WorldObject worldObject) {
		Vector3 dims = new Vector3(worldObject.getRealWorldDimension().x, worldObject.getRealWorldDimension().y,
				worldObject.getRealWorldDimension().z);
		if (worldObject instanceof KitchenElement) {
			KitchenElement kitchenElement = (KitchenElement) worldObject;
			if (!kitchenElement.getDesignObject().getDesignCaissonType().contains("COINS")) {
				float facadeThickness = kitchenElement.getDesignObject().getFacadeThickness() * 0.001f;
				dims.z += facadeThickness;
			}
		}
		return dims;
	}

	public void calculateProperties() {
		
		Vector3 boxSize = new Vector3();
		Vector3 center = new Vector3();
		rotatedBoundingBox.clr();
		calculateBoundingBox(rotatedBoundingBox);
		rotatedBoundingBox.getDimensions(boxSize);
		rotatedBoundingBox.getCenter(center);
		if(worldObject != null) {
			this.worldPosition = worldObject.getRealWorldPosition().cpy();
			this.dimensions = worldObject.getRealWorldDimension().xyz();
			if (worldObject instanceof KitchenElement) {
				this.pivot.set(dimensions.cpy().scl(0.5f));
				this.dimensions.set(boxSize);
			} else {
				this.pivot.set(center);
			}
		}else {
			this.dimensions = boxSize;
		}
		
	}

	public boolean contains(Vector3 point) {
		return rotatedBoundingBox.contains(point);
	}

	public boolean intersects(Object3D other) {
		return getRotatedBoundingBox().intersects(other.getRotatedBoundingBox());
	}

	public List<Object3DQuotation> getQuotations() {
		ArrayList<Object3DQuotation> quots = new ArrayList<Object3DQuotation>();
		Vector3 v000 = new Vector3();
		getRotatedBoundingBox().getCorner000(v000);
		// v000.mul(getRotatedBoundingBox().getTransform());
		Vector3 v001 = new Vector3();
		getRotatedBoundingBox().getCorner001(v001);
		// v001.mul(getRotatedBoundingBox().getTransform());
		Vector3 v101 = new Vector3();
		getRotatedBoundingBox().getCorner101(v101);
		// v101.mul(getRotatedBoundingBox().getTransform());
		Vector3 v011 = new Vector3();
		getRotatedBoundingBox().getCorner011(v011);
		// v011.mul(getRotatedBoundingBox().getTransform());
		boolean flippedXZ = Math.abs(v001.dst(v101) - worldObject.getRealWorldDimension().x) > Math
				.abs(v001.dst(v101) - worldObject.getRealWorldDimension().z);
		float xValue = flippedXZ ? worldObject.getRealWorldDimension().z : worldObject.getRealWorldDimension().x;
		float zValue = flippedXZ ? worldObject.getRealWorldDimension().x : worldObject.getRealWorldDimension().z;

		quots.add(new Object3DQuotation(v001.cpy(), v101.cpy(), this, xValue));
		quots.add(new Object3DQuotation(v001.cpy(), v011.cpy(), this, worldObject.getRealWorldDimension().y));
		quots.add(new Object3DQuotation(v000.cpy(), v001.cpy(), this, zValue));
		return quots;
	}

	public void translate(Vector3 translation) {
		transform.translate(translation);
		calculateProperties();
	}

	public void rotate(Vector3 axis, float degree) {
		transform.translate(error.cpy()).rotate(axis, degree).translate(error.cpy().scl(-1));
	}

	public Vector3[] getCorners() {
		Vector3 corner0 = new Vector3();
		Vector3 corner1 = new Vector3();
		Vector3 corner2 = new Vector3();
		Vector3 corner3 = new Vector3();
		rotatedBoundingBox.getCorner000(corner0);
		rotatedBoundingBox.getCorner100(corner1);
		rotatedBoundingBox.getCorner010(corner2);
		rotatedBoundingBox.getCorner001(corner3);
		return new Vector3[] { corner0, corner1, corner2, corner3 };
	}

	public Vector3[] getBounds() {
		ArrayList<Vector3> corners = new ArrayList<Vector3>();

		corners.add(rotatedBoundingBox.getCorner000(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner001(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner010(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner011(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner100(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner101(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner110(new Vector3()));
		corners.add(rotatedBoundingBox.getCorner111(new Vector3()));

		return MathUtilities.getBoundaries(corners);
	}

	public float getRadius() {
		return rotatedBoundingBox.getRadius();
	}

	public Vector3 getPosition() {
		return worldPosition;
	}

	public Vector3 getDimension() {
		return dimensions.cpy();
	}

	boolean visible(Camera camera) {
		return false;
	}

	public void drawBoundaries(Matrix4 vp, Vector3 color, float lineWidth) {
		rotatedBoundingBox.draw(vp, color, lineWidth);
	}

	public Object3D getSketch() {
		return sketchInstance;
	}

	public void render(ModelBatch batch) {
		batch.render(this);
		if (sketchInstance != null)
			batch.render(sketchInstance);
		for (var addon : addons) {
			Matrix4 relativeTransform = addon.transform.cpy();
			addon.transform.mulLeft(this.transform);
			addon.render(batch);
			addon.transform.set(relativeTransform);
		}
	}

	public RotatedBoundingBox calculateBoundingBox(RotatedBoundingBox out) {
		calculateTransforms();
		super.calculateBoundingBox(out);
		out.normalize();
		return out;
	}

	public float calculateEffectiveBoundingBox() {
		Matrix4 original = this.transform.cpy();
		Vector3 tmpMin = new Vector3(), tmpMax = new Vector3();
		rotatedBoundingBox.getMin(tmpMin);
		rotatedBoundingBox.getMax(tmpMax);
		Vector3 tmpVec = tmpMax.cpy().sub(tmpMin);
		float minVol = Math.abs(tmpVec.x * tmpVec.y * tmpVec.z);
		optimalVolumeAngle = 0;
		rotatedBoundingBox.set(rotatedBoundingBox);
		rotatedBoundingBox.setTransform(new Matrix4().setToTranslation(getPosition())
				.rotate(Vector3.Y, -optimalVolumeAngle).translate(getPosition().cpy().scl(-1)));
		for (int i = 1; i < 180; i++) {
			rotate(Vector3.Y, 0.5f);
			calculateBoundingBox(rotatedBoundingBox);
			rotatedBoundingBox.getMin(tmpMin);
			rotatedBoundingBox.getMax(tmpMax);
			tmpVec = tmpMax.cpy().sub(tmpMin);
			float newVolu = Math.abs(tmpVec.x * tmpVec.y * tmpVec.z);
			if (newVolu < minVol) {
				optimalVolumeAngle = i * 0.5f;
				minVol = newVolu;
				rotatedBoundingBox.set(rotatedBoundingBox);
				rotatedBoundingBox.setTransform(new Matrix4().setToTranslation(getPosition())
						.rotate(Vector3.Y, -optimalVolumeAngle).translate(getPosition().cpy().scl(-1)));
			}
		}
		this.transform.set(original);
		return optimalVolumeAngle;

	}

	public float getOptimalVolumeAngle() {
		return optimalVolumeAngle;
	}

	public void setOptimalVolumeAngle(float optimalVolumeAngle) {
		this.optimalVolumeAngle = optimalVolumeAngle;
	}

	public RotatedBoundingBox getRotatedBoundingBox() {
		return rotatedBoundingBox;
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();
	}

	public void cleanUp() {
		if (sketchModel != null)
			sketchModel.dispose();
		if (sketchInstance != null)
			sketchInstance.cleanUp();
		worldObject = null;
		worldPosition = null;
		dimensions = null;
		sketchModel = null;
		sketchInstance = null;

	}

	@Override
	public String toString() {
		return "Object3D [worldObject=" + worldObject.getName() + ", radius=" + getRadius() + ", position="
				+ worldPosition + ", dimensions=" + dimensions + "]";
	}

	public void drawQuotations(Batch batch, Camera camera) {
		List<Vector3> corners = getRotatedBoundingBox().getCorners();
		Vector3 furthest = null;
		for (Vector3 corner : corners) {
			if (furthest == null || furthest.dst2(camera.position) < corner.dst2(camera.position)) {
				furthest = corner;
			}
		}

		List<LineSegment> edges = getRotatedBoundingBox().getEdges();
		// remove edges connected to the hidden corner
		boolean stop;
		do {
			stop = true;
			for (LineSegment lineSegment : edges) {
				if (lineSegment.startOrEndWith(furthest, 0.001f)) {
					edges.remove(lineSegment);
					stop = false;
					break;
				}
			}
		} while (!stop);
		// remove quotations from hidden edges
		if (shapeRenderer == null)
			shapeRenderer = new ShapeRenderer();
		Vector3 furthestFromCamera = null;
		for (LineSegment segment : edges) {
			shapeRenderer.begin(ShapeType.Filled);
			shapeRenderer.setColor(Color.BLUE);
			Vector3 v0 = camera.project(segment.getV0().cpy());
			Vector3 v1 = camera.project(segment.getV1().cpy());
			Vector3 f = segment.getV0().dst2(camera.position) > segment.getV1().dst2(camera.position)
					? segment.getV0().cpy()
					: segment.getV1().cpy();
			if (furthestFromCamera == null || furthestFromCamera.dst2(camera.position) < f.dst2(camera.position))
				furthestFromCamera = f;
			shapeRenderer.circle(v0.x, v0.y, 4);
			shapeRenderer.circle(v1.x, v1.y, 4);
			shapeRenderer.line(v0, v1);
			shapeRenderer.end();
			String length = String.format("%.3f", v0.dst(v1));
			Vector3 v = v1.cpy().add(v0).scl(0.5f);
			float angle = new Vector2(v1.x - v0.x, v1.y - v0.y).angle() % 180;
			Matrix4 transform = new Matrix4();
			// transform.setToRotation(Vector3.Z, 45);
			transform.set(v.cpy().scl(1, 1, 0), new Quaternion(Vector3.Z, angle));
			String measure = String.format("%.0f", segment.len() * 1000);
			batch.setTransformMatrix(transform);
			batch.begin();
			// AssetFont.getInstance().getSmallFont().draw(batch, measure, 0, 0);
			AssetFont.getInstance().getFont().drawUnscaled(batch, measure, angle, angle, angle, angle, angle, null);
			batch.end();

		}
		camera.project(furthestFromCamera);
//		shapeRenderer.setColor(Color.RED);
//		shapeRenderer.circle(furthestFromCamera.x, furthestFromCamera.y,10);
//		shapeRenderer.end();
	}

	public void addOns(Object3D... addons) {
		for (var addon : addons) {
			this.addons.add(addon);
		}
	}

	public List<Object3D> getAddons() {
		return addons;
	}
	
	
}
