package gdxapp.object3d;

import java.util.ArrayList;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.model.Node;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder.VertexInfo;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import dressing.mathutils.EarClipper;
import dressing.mathutils.MathUtilities;
import gdxapp.shaders.PbrMaterial;
import geometry.Triangle3D;

public class DeepPolyBuilder {

	
	private MeshPartBuilder partBuilder;
	
	//build a 3d polygon part wrapped in its own node
	public void buildPoly(String partName, ArrayList<Vector2> planeVertices, Vector3 size, Matrix4 transform, PbrMaterial mtl, ModelBuilder modelBuilder) {
		MathUtilities.setWinding(planeVertices, 1);
		Matrix3 rotationTransform = MathUtilities.extractRotation(transform);
		Node node = modelBuilder.node();
		node.id = partName;
		int attrs = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal | VertexAttributes.Usage.TextureCoordinates;
		partBuilder = modelBuilder.part(partName, GL20.GL_TRIANGLES, attrs , mtl);
		buildFrontAndBackFaces(planeVertices, size, mtl, transform, rotationTransform);
		buildSideFaces(planeVertices, transform, rotationTransform, size);
	}
	
	public void buildAnnularPoly(String partName, ArrayList<Vector2> outerVertices, ArrayList<Vector2> innerVertices, Vector3 size, Matrix4 transform, PbrMaterial mtl, ModelBuilder modelBuilder) {
		ArrayList<Vector2> readyVertices = sortVertices(outerVertices, innerVertices);
		buildPoly(partName, readyVertices, size, transform, mtl, modelBuilder);
	}

	public ArrayList<Vector2> sortVertices(ArrayList<Vector2> polygonVertices, ArrayList<Vector2> holeVertices) {
		MathUtilities.setWinding(polygonVertices, 1);
		ArrayList<Vector2> vertices = new ArrayList<>();
		if(!(holeVertices == null ||holeVertices.isEmpty())) {
			MathUtilities.setWinding(holeVertices, -1);
			Vector2[] cut = MathUtilities.findMutuallyVisibleVertices(polygonVertices, holeVertices);
			for (int i = 0; i <= polygonVertices.indexOf(cut[1]); i++) {
				vertices.add(polygonVertices.get(i));
			}
			int index = holeVertices.indexOf(cut[0]);
			for (int i = 0; i < holeVertices.size(); i++) {
				vertices.add(holeVertices.get(index));
				index++;
				index = index % holeVertices.size();
			}
			vertices.add(cut[0]);
			for (int i = polygonVertices.indexOf(cut[1]); i < polygonVertices.size(); i++) {
				vertices.add(polygonVertices.get(i));
			}
		}
		else {
			vertices.addAll(polygonVertices);
		}
		return vertices;
	}
	
	public void buildFrontAndBackFaces(ArrayList<Vector2> vertices, Vector3 modelSize, PbrMaterial material, Matrix4 transform, Matrix3 rotationTransform) {
		var triangles = EarClipper.triangulate(vertices);
		ArrayList<Triangle3D> faceTriangles = new ArrayList<Triangle3D>();
		ArrayList<Triangle3D> backTriangles = new ArrayList<Triangle3D>();
		for(var tr: triangles) {
			faceTriangles.add(new Triangle3D(new Vector3(tr.getV0(), modelSize.z), new Vector3(tr.getV1(), modelSize.z), new Vector3(tr.getV2(),modelSize.z)));
			backTriangles.add(new Triangle3D(new Vector3(tr.getV0(), -modelSize.z), new Vector3(tr.getV1(), -modelSize.z), new Vector3(tr.getV2(),-modelSize.z)));
		}
		
		if(!material.isReady())
			material.prepare();
		triangles.toString();
		//face triangles
		for(int i = 0 ; i < triangles.size(); i++) {
			var faceVertices = faceTriangles.get(i).getVertices();
			Vector2[] uvs = new Vector2[3];
			int c = 0;
			float scale = Math.round(Math.max(modelSize.x, modelSize.y)/0.5f) + 1;
			for(var vertex: faceVertices) {
				float u = scale * (vertex.x + modelSize.x); 
				float v = scale * (vertex.y + modelSize.y);
				uvs[c++] = new Vector2(u, v);
			}
			Vector3 normal = Vector3.Z.cpy();
			Vector3 N = normal.cpy().mul(rotationTransform);
			var backVertices = backTriangles.get(i).getVertices();
			
			int[] t1Indices = {0,1,2};
			//swap if order is wrong
			Vector3 computedNormal = faceVertices[1].cpy().sub(faceVertices[0]).crs(faceVertices[2].cpy().sub(faceVertices[0])).nor();
			if(computedNormal.dot(normal) < 0)
				t1Indices = new int[]{0,2,1};
			VertexInfo[] infos = {
				new VertexInfo().setPos(faceVertices[0].mul(transform)).setNor(N).setUV(uvs[0]),
				new VertexInfo().setPos(faceVertices[1].mul(transform)).setNor(N).setUV(uvs[1]),
				new VertexInfo().setPos(faceVertices[2].mul(transform)).setNor(N).setUV(uvs[2])
			};

			VertexInfo[] infosB = {
					new VertexInfo().setPos(backVertices[0].mul(transform)).setNor(N.cpy().scl(-1)).setUV(uvs[0]),
					new VertexInfo().setPos(backVertices[1].mul(transform)).setNor(N.cpy().scl(-1)).setUV(uvs[1]),
					new VertexInfo().setPos(backVertices[2].mul(transform)).setNor(N.cpy().scl(-1)).setUV(uvs[2])
			};
			
			partBuilder.triangle(infos[t1Indices[0]], infos[t1Indices[1]], infos[t1Indices[2]]);
			partBuilder.triangle(infosB[t1Indices[0]], infosB[t1Indices[2]], infosB[t1Indices[1]]);
		}
	}
	
	public void buildSideFaces(ArrayList<Vector2> vertices, Matrix4 transform, Matrix3 rotationTransform, Vector3 modelSize) {
		int size = vertices.size();
		for(int i = 0; i < vertices.size(); i++) {
			Vector2 current = vertices.get(i);
			Vector2 next = vertices.get((i+1)%size);
			Vector3 normal = new Vector3(next.cpy().sub(current),0).rotate(Vector3.Z.cpy(), -90).nor();
			Vector3 N = normal.cpy().mul(rotationTransform);
			
			Vector3[] positions = { new Vector3(current, modelSize.z),
									new Vector3(next, modelSize.z),
									new Vector3(current, -modelSize.z),
									new Vector3(next, -modelSize.z)};
				
			Vector2[] uvs = new Vector2[4];
			float scale = Math.round(Math.max(modelSize.z, modelSize.y)/0.5f) + 1;
			for(int j = 0; j < positions.length; j++) {
				uvs[j] = new Vector2().set(positions[j].z + modelSize.z, positions[j].y).scl(scale);
			}
			
			int[] t1Indices = {0,1,2};
			int[] t2Indices = {3,2,1};
			//swap if order is wrong
			if(positions[1].cpy().sub(positions[0]).crs(positions[2].cpy().sub(positions[0])).dot(normal) < 0) {
				t1Indices = new int[]{0,2,1};
			}
			
			if(positions[2].cpy().sub(positions[3]).crs(positions[1].cpy().sub(positions[3])).dot(normal) < 0) {
				t2Indices = new int[]{3,1,2};
			}
					
			VertexInfo[] infos = {
					new VertexInfo().setPos(positions[0].mul(transform)).setNor(N).setUV(uvs[0]),
					new VertexInfo().setPos(positions[1].mul(transform)).setNor(N).setUV(uvs[1]),
					new VertexInfo().setPos(positions[2].mul(transform)).setNor(N).setUV(uvs[2]),
					new VertexInfo().setPos(positions[3].mul(transform)).setNor(N).setUV(uvs[3])
			};

			partBuilder.triangle(infos[t1Indices[0]], infos[t1Indices[1]], infos[t1Indices[2]]);
			partBuilder.triangle(infos[t2Indices[0]], infos[t2Indices[1]], infos[t2Indices[2]]);
		}
	}
	
	
}
