package gdx.app.camera;

import java.io.FileNotFoundException;
import java.util.Stack;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.ModelBatch;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.g3d.Renderable;
import com.badlogic.gdx.graphics.g3d.Shader;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder.VertexInfo;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.g3d.utils.RenderContext;
import com.badlogic.gdx.graphics.g3d.utils.ShaderProvider;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.ui.ImageButton;
import com.badlogic.gdx.scenes.scene2d.ui.Table;
import com.badlogic.gdx.scenes.scene2d.utils.ChangeListener;
import com.badlogic.gdx.utils.GdxRuntimeException;
import gdxapp.assets.AssetsTextures;
import gdxapp.screens.room3d.MouseOnlyCameraController;
import gdxapp.ui.InMemorySkinService;

public class CameraViewSelector {

	private final String vertexSrc = """
			#version 330 core
			in vec4 a_position;
			in vec2 a_texCoord0;
			uniform mat4 projection;
			uniform mat4 model;
			out vec2 v_texcoord;
			void main(){
				v_texcoord = a_texCoord0;
			    gl_Position = projection * model * a_position;
			}
						""";
	private final String pixelSrc = """
			#version 330 core
			in vec2 v_texcoord;
			uniform sampler2D u_texture;
			out vec4 color;
			void main(){
			    color = vec4(texture(u_texture, v_texcoord).rgb, 0.75f);
			}
						""";
	
	private final String[] sideTextures = { 
			"C:\\ProgramData\\supercad\\face\\top.png",
			"C:\\ProgramData\\supercad\\face\\bottom.png",
			"C:\\ProgramData\\supercad\\face\\front.png",
			"C:\\ProgramData\\supercad\\face\\back.png",
			"C:\\ProgramData\\supercad\\face\\left.png",
			"C:\\ProgramData\\supercad\\face\\right.png"
	};
	
	private ShaderProgram program;
	private ModelInstance renderable;
	private ModelBatch batch;
	private Shader shader;
	
	private Table uiComponent;
	private float animationDuration = 1000;  // animmation should take 1 second
	private float animationTime;             // how much we went through the animation
	private Vector3 animationAxis = new Vector3();
	private float animationAngle;
	
	private Stack<Runnable> tasks = new Stack<Runnable>();
	private Runnable currentTask;
	private MouseOnlyCameraController cameraController;
	
	public CameraViewSelector() {
		init();
	}

	private void init() {
		createRenderable();
		program = new ShaderProgram(vertexSrc, pixelSrc);
		if(!program.isCompiled())
            throw new GdxRuntimeException(program.getLog());
		createShader();
		batch = new ModelBatch(new ShaderProvider() {
			@Override
			public Shader getShader(Renderable renderable) {
				return shader;
			}
			@Override
			public void dispose() {
				shader.dispose();
			}
		});
		
		createUiContribution();
	}
	
	

	public void draw(Camera camera, float delta) {

		Gdx.gl.glEnable(GL20.GL_BLEND);

		if(currentTask == null && !tasks.isEmpty()) {
			currentTask = tasks.pop();
			currentTask.run();
		}
		animate(delta);
		batch.begin(camera);
		Gdx.gl.glDepthFunc(GL20.GL_ALWAYS);
		Gdx.gl.glEnable(GL20.GL_BLEND);

		batch.render(renderable);
		Gdx.gl.glDepthFunc(GL20.GL_LEQUAL);
		batch.end();
		
		//for  debug
//		Quaternion q = new Quaternion();
//		renderable.transform.getRotation(q, true);
//		Vector3 axis = new Vector3();
//		float angle = q.getAxisAngle(axis);
//		Vector3 v = new Vector3(0,0,1);
//		q.transform(v);
//		System.out.println(v + ":" + angle);
	}
	
	private void createUiContribution() {
		var skin = InMemorySkinService.getInstance().getSkin();
		ImageButton upBtn = new ImageButton(skin, "up-chevron");
		upBtn.addListener(new ChangeListener() {
			@Override
			public void changed(ChangeEvent event, Actor actor) {
				tasks.add(() -> rotate(Vector3.X.cpy(), 90));
				Quaternion qRot = new Quaternion();
				renderable.transform.getRotation(qRot);
				qRot.mul(new Quaternion(Vector3.X.cpy(), 90));

				
			}
		});
		ImageButton downBtn = new ImageButton(skin, "down-chevron");
		downBtn.addListener(new ChangeListener() {
			@Override
			public void changed(ChangeEvent event, Actor actor) {
				tasks.add(() -> rotate(Vector3.X.cpy(), -90));
				Quaternion qRot = new Quaternion();
				renderable.transform.getRotation(qRot);
				qRot.mulLeft(new Quaternion(Vector3.X.cpy(), -90));
			}

			
		});
		ImageButton leftBtn = new ImageButton(skin, "left-chevron");
		leftBtn.addListener(new ChangeListener() {
			@Override
			public void changed(ChangeEvent event, Actor actor) {
				tasks.add(() -> rotate(Vector3.Y.cpy(), 90));
				Quaternion qRot = new Quaternion();
				renderable.transform.getRotation(qRot);
				new Quaternion(Vector3.Y.cpy(), 90).mulLeft(qRot);
			}
		});
		ImageButton rightBtn = new ImageButton(skin, "right-chevron");
		rightBtn.addListener(new ChangeListener() {
			@Override
			public void changed(ChangeEvent event, Actor actor) {
				tasks.add(() -> rotate(Vector3.Y.cpy(), -90));
				Quaternion qRot = new Quaternion();
				renderable.transform.getRotation(qRot);
				qRot.mulLeft(new Quaternion(Vector3.Y.cpy(), -90));
			}
		});
		uiComponent = new Table();
		uiComponent.defaults().size(48).pad(5).center();
		
		uiComponent.add();
		uiComponent.add(downBtn).center();
		uiComponent.add();
		uiComponent.row();
		uiComponent.add(rightBtn);
		uiComponent.add();
		uiComponent.add(leftBtn);
		uiComponent.row();
		uiComponent.add();
		uiComponent.add(upBtn);
		uiComponent.add();
	}
	
	
	
	private void rotate(Vector3 axis, float degrees) {
		this.animationAxis = axis;
		this.animationAngle = degrees;
		//reset the animation time
		this.animationTime = animationDuration;
	}
	
	private void setToOrtho(Vector3 side) {
		if(side.epsilonEquals(new Vector3(0, 0, -1), 0.001f))	{
			cameraController.keyDown(Input.Keys.NUMPAD_8);
		}else if(side.epsilonEquals(new Vector3(0, -1, 0), 0.001f))	{
			cameraController.keyDown(Input.Keys.NUMPAD_5);
		}else if(side.epsilonEquals(new Vector3(1, 0, 0), 0.001f))	{
			cameraController.keyDown(Input.Keys.NUMPAD_4);
		}else if(side.epsilonEquals(new Vector3(-1, 0, 0), 0.001f))	{
			cameraController.keyDown(Input.Keys.NUMPAD_6);
		}
	}
	
	
	//fix to do: make sure the final orientation of the cube at the end of the animation match
	//the desired orientation
	private void animate(float delta) {
		if(!animationAxis.epsilonEquals(Vector3.Zero) && animationTime > 0) {
			if(animationTime - delta < 0)
				delta = animationTime;
			float angle = (delta/animationDuration) * animationAngle;
			Quaternion q = new Quaternion(animationAxis, angle);
			Matrix4 rotation = new Matrix4(q);
			renderable.transform.mul(rotation);
			animationTime -= delta;
		}else {
			if(currentTask != null) {
				Quaternion q = new Quaternion();
				renderable.transform.getRotation(q, true);
				Vector3 v = new Vector3(0,0,1);
				q.transform(v);
				setToOrtho(v);
			}
			animationAxis.set(0,0,0);
			currentTask = null;
			
		}
	}
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	// initialization functions
	
	private void createShader() {
		shader = new Shader() {
			@Override
			public void dispose() {
				program.dispose();
			}
			@Override
			public void render(Renderable renderable) {
				TextureAttribute textureDiff = (TextureAttribute) renderable.material.get(TextureAttribute.Diffuse);
				textureDiff.textureDescription.texture.bind(0);
                program.setUniformi("u_texture", 0);
                program.setUniformMatrix("model", renderable.worldTransform);
                renderable.meshPart.render(program);
			}
			
			@Override
			public void init() {
				ShaderProgram.pedantic = false;
			}
			
			@Override
			public void end() {
		        program.end();
			}
			
			@Override
			public int compareTo(Shader other) {
				// TODO Auto-generated method stub
				return 0;
			}
			
			@Override
			public boolean canRender(Renderable instance) {
				return true;
			}
			
			@Override
			public void begin(Camera camera, RenderContext context) {
				program.begin();
				program.setUniformMatrix("projection", camera.projection);
			}
		};		
	}

	private void createRenderable() {
		Vector3[] position = {
				new Vector3(-0.5f, -0.5f, -0.5f),
				new Vector3(-0.5f, -0.5f,  0.5f),
				new Vector3(-0.5f,  0.5f, -0.5f),
				new Vector3(-0.5f,  0.5f,  0.5f),

				new Vector3( 0.5f, -0.5f, -0.5f),
				new Vector3( 0.5f, -0.5f,  0.5f),
				new Vector3( 0.5f,  0.5f, -0.5f),
				new Vector3( 0.5f,  0.5f,  0.5f),
		};
		Vector2[] uvs = {
				new Vector2(0, 0),
				new Vector2(0, 1),
				new Vector2(1, 0),
				new Vector2(1, 1),
		};
		
		ModelBuilder modelBuilder = new ModelBuilder();
		Material[] materials = new Material[6];

		try {
			for(int i = 0; i < 6; i++) {
				var texture = AssetsTextures.getInstance().loadTextureFromFile(sideTextures[i], false);
				materials[i] = new Material();
				materials[i].set(TextureAttribute.createDiffuse(texture));
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		modelBuilder.begin();
		var partBuilder =  modelBuilder.part("top", GL20.GL_TRIANGLES, VertexAttributes.Usage.Position | VertexAttributes.Usage.TextureCoordinates, materials[0]);
		VertexInfo v0 = new VertexInfo().setPos(position[2]).setUV(uvs[0]);
		VertexInfo v1 = new VertexInfo().setPos(position[3]).setUV(uvs[1]);
		VertexInfo v2 = new VertexInfo().setPos(position[7]).setUV(uvs[3]);
		VertexInfo v3 = new VertexInfo().setPos(position[6]).setUV(uvs[2]);
		partBuilder.triangle(v0, v1, v2);		
		partBuilder.triangle(v2, v3, v0);
		partBuilder =  modelBuilder.part("bottom", GL20.GL_TRIANGLES, VertexAttributes.Usage.Position | VertexAttributes.Usage.TextureCoordinates, materials[1]);
		v0 = new VertexInfo().setPos(position[0]).setUV(uvs[0]);
		v1 = new VertexInfo().setPos(position[4]).setUV(uvs[1]);
		v2 = new VertexInfo().setPos(position[5]).setUV(uvs[3]);
		v3 = new VertexInfo().setPos(position[1]).setUV(uvs[2]);
		partBuilder.triangle(v0, v1, v2);		
		partBuilder.triangle(v2, v3, v0);
		partBuilder =  modelBuilder.part("front", GL20.GL_TRIANGLES, VertexAttributes.Usage.Position | VertexAttributes.Usage.TextureCoordinates, materials[2]);
		v0 = new VertexInfo().setPos(position[3]).setUV(uvs[0]);
		v1 = new VertexInfo().setPos(position[1]).setUV(uvs[1]);
		v2 = new VertexInfo().setPos(position[5]).setUV(uvs[3]);
		v3 = new VertexInfo().setPos(position[7]).setUV(uvs[2]);
		partBuilder.triangle(v0, v1, v2);		
		partBuilder.triangle(v2, v3, v0);	
		partBuilder =  modelBuilder.part("back", GL20.GL_TRIANGLES, VertexAttributes.Usage.Position | VertexAttributes.Usage.TextureCoordinates, materials[3]);
		v0 = new VertexInfo().setPos(position[6]).setUV(uvs[0]);
		v1 = new VertexInfo().setPos(position[4]).setUV(uvs[1]);
		v2 = new VertexInfo().setPos(position[0]).setUV(uvs[3]);
		v3 = new VertexInfo().setPos(position[2]).setUV(uvs[2]);
		partBuilder.triangle(v0, v1, v2);		
		partBuilder.triangle(v2, v3, v0);
		partBuilder =  modelBuilder.part("left", GL20.GL_TRIANGLES, VertexAttributes.Usage.Position | VertexAttributes.Usage.TextureCoordinates, materials[4]);
		v0 = new VertexInfo().setPos(position[2]).setUV(uvs[0]);
		v1 = new VertexInfo().setPos(position[0]).setUV(uvs[1]);
		v2 = new VertexInfo().setPos(position[1]).setUV(uvs[3]);
		v3 = new VertexInfo().setPos(position[3]).setUV(uvs[2]);
		partBuilder.triangle(v0, v1, v2);		
		partBuilder.triangle(v2, v3, v0);
		partBuilder =  modelBuilder.part("right", GL20.GL_TRIANGLES, VertexAttributes.Usage.Position | VertexAttributes.Usage.TextureCoordinates, materials[5]);
		v0 = new VertexInfo().setPos(position[7]).setUV(uvs[0]);
		v1 = new VertexInfo().setPos(position[5]).setUV(uvs[1]);
		v2 = new VertexInfo().setPos(position[4]).setUV(uvs[3]);
		v3 = new VertexInfo().setPos(position[6]).setUV(uvs[2]);
		partBuilder.triangle(v0, v1, v2);		
		partBuilder.triangle(v2, v3, v0);
		var model =  modelBuilder.end();
		model.calculateTransforms();
		renderable = new ModelInstance(model);
		renderable.transform.idt().translate(-Gdx.graphics.getWidth()/2.0f + 100, -Gdx.graphics.getHeight()/2.0f + 100 , 0).rotate(Vector3.Y, 180).scale(50, 50, 50);
	}


	
	// getters and setters
	public Table getUiComponent() {
		return uiComponent;
	}

	public MouseOnlyCameraController getCameraController() {
		return cameraController;
	}

	public void setCameraController(MouseOnlyCameraController cameraController) {
		this.cameraController = cameraController;
	}
	
	

}
