package geometry;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.InputListener;

import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.ui.project.CreateFormWizard;
import dressing.ui.util.EdgeConfigWindow;
import dressing.ui.util.OpenWizardDialog;
import gdxapp.assets.AssetFont;
import gdxapp.assets.DrawingHelper;
import gdxapp.object3d.FullScreenActor;
import gdxapp.object3d.GeometryObject;
import gdxapp.object3d.Wall;
import gdxapp.screens.room.RoomController;
import gdxapp.ui.CursorProvider;

public class PolygonDrawer extends FullScreenActor implements IDrawer<Polygon> {

	private Polygon out; // the object being drawn
	private GeometryObject editable;
	private final ArrayList<Vector2> vertices = new ArrayList<>();
	private final ArrayList<Edge> edges = new ArrayList<Edge>();
	private Vector2 tempVertex;
	private List<Vector2> referencePoints;
	private Vector2 hRefPnt;
	private Vector2 vRefPnt;
	private Vector2 mouseOver;
	private Edge hoveredEdge;
	private final Vector2 mouseLocation = new Vector2();
	private ShapeRenderer shapeRenderer;

	private Vector3[] angles;
	private Vector2 selectedVertex;
	private boolean closed;
	private Vector2[] bounds;
	private float stageScale = 10000;
	private PolygonDrawerListenner listener;
	private Matrix4 transform; // the transform where the vertices have been picked
	private boolean createMode;

	private static PolygonDrawer instance;

	private PolygonDrawer() {
		super();
		listener = new PolygonDrawerListenner();
		addListener(listener);
	}

	@Override
	public void begin(boolean closed) {
		if (editable != null) {
			end();
			RoomController.getInstance().updateObject(editable);
		}
		clear();
		this.closed = closed;
		createMode = true;
		out = new Polygon();
		out.setClosed(closed);
		onCameraChanged();
	}

	public void edit(GeometryObject geometryObject) {
		this.editable = geometryObject;
		Polygon poly = (Polygon) geometryObject.getGeometry().getBorder().getNodes().get(0);
		edit(poly);
	}

	@Override
	public void edit(Polygon shape) {
		out = shape;
		createMode = false;
		this.vertices.clear();
		for (Vector2 vertex : shape.getVertices()) {
			this.vertices.add(vertex);
		}
		updateEdgesList();
		calculateAngles();
	}

	@Override
	public Polygon end() {
		if (this.vertices != null && this.vertices.size() > 1) {
			out.setVertices(vertices);
			return out;
		}
		return null;
	}

	public void clear() {
		vertices.clear();
		bounds = null;
		angles = null;
		edges.clear();
		tempVertex = null;
		selectedVertex = null;
		editable = null;
	}



	private boolean selectEdge() {
		float pixel = getPixelWidth();
		hoveredEdge = null;
		if (!this.edges.isEmpty()) {
			Edge nearest = edges.get(0);
			for (int i = 1; i < edges.size(); i++) {
				if (edges.get(i).distanceTo(mouseLocation) < nearest.distanceTo(mouseLocation))
					nearest = edges.get(i);
			}
			if (nearest.distanceTo(mouseLocation) < 10 * pixel) {
				hoveredEdge = nearest;
				tempVertex = null;
				return true;
			}
		}
		return false;
	}

	public void updateEdgesList() {
		edges.clear();
		int count = vertices.size();
		if (count > 1) {
			for (int i = 0; i < count - 1; i++)
				edges.add(new Edge(vertices.get(i), vertices.get(i + 1)));
			if (closed)
				edges.add(new Edge(vertices.get(count - 1), vertices.get(0)));
		}
	}

	private void addVertex(Vector2 point) {
		MathUtilities.round(point, 0);
		if (vertices.isEmpty()) {
			vertices.add(point);
		} else {
			if (vertices.size() > 2) { // check for intersection
				Edge edge = new Edge(vertices.get(vertices.size() - 1), point);
				for (Edge edgeX : edges) {
					Vector2 intersection = MathUtilities.getIntersectionPoint(edgeX, edge);
					if (!(edgeX.hasVertex(intersection) || edge.hasVertex(intersection))
							&& (edgeX.contains(intersection) && edge.contains(intersection))) {
						System.out.println("plygon is self intersecting!");
						float a = edgeX.getV0().dst(intersection);
						float b = edgeX.getV1().dst(intersection);
						float c = edgeX.getV0().dst(edgeX.getV1());
						System.out.println(a + " " + b + " " + c);
						return;
					}
				}
			}
			rectifyPosition(point);
			vertices.add(point);
			updateBoundaries();
			updateEdgesList();
		}

	}

	private void updateBoundaries() {
		Vector2 lastPosition = vertices.get(vertices.size() - 1);
		if (this.bounds == null) {
			this.bounds = new Vector2[4];
			this.bounds[0] = lastPosition.cpy();
			this.bounds[1] = lastPosition.cpy();
			this.bounds[2] = lastPosition.cpy();
			this.bounds[3] = lastPosition.cpy();

		} else {
			this.bounds[0].x = Math.min(lastPosition.x, bounds[0].x);
			this.bounds[0].y = Math.min(lastPosition.y, bounds[0].y);
			this.bounds[1].x = Math.max(lastPosition.x, bounds[1].x);
			this.bounds[1].y = Math.max(lastPosition.y, bounds[1].y);
			this.bounds[2].set(this.bounds[0].x, this.bounds[1].y);
			this.bounds[3].set(this.bounds[1].x, this.bounds[0].y);
		}
	}

	public boolean link(Vector2 point) {
		Vector2 nearestPoint = null;
		for (Vector2 attachmentPoint : RoomController.getInstance().getCriticalPoints()) {
			if (nearestPoint == null) {
				nearestPoint = attachmentPoint;
				continue;
			}
			if (attachmentPoint.dst(point) < nearestPoint.dst(point)) {
				nearestPoint = attachmentPoint;
			}
		}

		if (nearestPoint != null && nearestPoint.dst(point) < 20) {
			point.set(nearestPoint);
			return true;
		}
		return false;
	}

	private void rectifyPosition(Vector2 point) {

		int index = vertices.indexOf(point);
		
		if(hRefPnt != null || vRefPnt != null) {
			point.set(alignPoint(point));

		}else {
			if (vertices.size() < 3)
				return;
			Vector2 original = point.cpy();
			Vector2 next = index > vertices.size() - 2 ? null : vertices.get(index + 1);
			Vector2 previous = index > 0 ? vertices.get(index - 1) : null;
			try {
				if (previous == null) {
					Vector2 v0 = vertices.get(index + 2).cpy();
					Vector2 v1 = vertices.get(index + 1).cpy();
					Vector2 v01 = new Vector2(v0).sub(v1);
					Vector2 v21 = new Vector2(point).sub(v1);
					float angle = MathUtilities.signedAngle(v01, v21);
					float rads5 = (float) Math.toRadians(15);
					float rectifiedAngle = Math.round(angle / rads5) * rads5;
					Vector2 rectifiedDirection = v01.nor().rotateRad(rectifiedAngle).scl(v21.len());
					point.set(v1).add(rectifiedDirection);
				} else {
					Vector2 v0 = vertices.get(index - 2).cpy();
					Vector2 v1 = vertices.get(index - 1).cpy();
					Vector2 v01 = new Vector2(v0).sub(v1);
					Vector2 v21 = new Vector2(point).sub(v1);
					float angle = MathUtilities.signedAngle(v01, v21);
					float rads5 = (float) Math.toRadians(15);
					float rectifiedAngle = Math.round(angle / rads5) * rads5;
					Vector2 rectifiedDirection = v01.nor().rotateRad(rectifiedAngle).scl(v21.len());
					point.set(v1).add(rectifiedDirection);
				}
			} catch (IndexOutOfBoundsException e) {
				point.set(original);
			}
		}
	}



	void calculateAngles() {
		if (this.vertices != null && this.vertices.size() > 2) {
			this.angles = new Vector3[this.vertices.size()];
			int previousIndex, nextIndex;
			Vector2 v01, v21;
			for (int i = 0; i < vertices.size(); i++) {
				previousIndex = (i - 1) % vertices.size();
				if (previousIndex < 0)
					previousIndex += vertices.size();
				nextIndex = (i + 1) % vertices.size();
				v01 = new Vector2(vertices.get(previousIndex)).sub(vertices.get(i));
				v21 = new Vector2(vertices.get(nextIndex)).sub(vertices.get(i));
				float angle = MathUtilities.signedAngle(v01, v21);
				angles[i] = new Vector3(vertices.get(i), angle);
			}
		}
	}

	@Override
	public void draw(Batch batch, float parentAlpha) {
		getStage().setKeyboardFocus(this);
		draw(batch.getProjectionMatrix());
		drawQuotations(batch);
	}

	public void draw(Matrix4 projection) {
		if (shapeRenderer == null)
			shapeRenderer = new ShapeRenderer();
		float pixel = getPixelWidth();
		shapeRenderer.setProjectionMatrix(camera.combined);
		shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);

		// draw reference points
		if (referencePoints != null) {
			shapeRenderer.setColor(Color.YELLOW);
			Vector2 tmp = new Vector2(pixel, pixel).scl(5);
			for (Vector2 point : referencePoints) {
				shapeRenderer.rectLine(point.cpy().add(tmp.cpy().scl(1, 1)), point.cpy().add(tmp.cpy().scl(-1, -1)),
						2 * pixel);
				shapeRenderer.rectLine(point.cpy().add(tmp.cpy().scl(-1, 1)), point.cpy().add(tmp.cpy().scl(1, -1)),
						2 * pixel);
			}
		}

		int verticesCount = closed ? vertices.size() + 1 : vertices.size();
		int nextIndex;
		for (int i = 0; i < vertices.size(); i++) {
			shapeRenderer.setColor(Color.GREEN);
			if (vertices.get(i) == selectedVertex)
				shapeRenderer.setColor(Color.RED);
			shapeRenderer.circle(vertices.get(i).x, vertices.get(i).y, 4 * pixel);
		}

		for (int i = 0; i < verticesCount - 1; i++) {
			nextIndex = (i + 1) % (vertices.size());
			shapeRenderer.setColor(Color.GREEN);
			shapeRenderer.rectLine(vertices.get(i), vertices.get(nextIndex), 3 * pixel);
		}

		if (hRefPnt != null) {
			Vector2 tmpPoint = alignPoint(mouseLocation);
			DrawingHelper.drawCutLine(tmpPoint, hRefPnt, getStage().getBatch(), Color.BLUE, pixel, 12 * pixel,
					4 * pixel);

		}
		if (vRefPnt != null) {
			Vector2 tmpPoint = alignPoint(mouseLocation);
			DrawingHelper.drawCutLine(tmpPoint, vRefPnt, getStage().getBatch(), Color.BLUE, pixel, 12 * pixel,
					4 * pixel);
		}

		if (hoveredEdge != null) {
			shapeRenderer.setColor(1, 0, 0, 1);
			shapeRenderer.rectLine(hoveredEdge.getV0(), hoveredEdge.getV1(), pixel * 2);
		}

		shapeRenderer.end();
	}

	private void drawQuotations(Batch batch) {
		float pixel = getPixelWidth();
		if (edges != null) {
			for (Edge edge : edges) {
				float length = edge.getLength() / RoomController.getInstance().getScaleX();
				length = Math.round(1000 * length);
				Vector2 position = edge.getMiddle().add(edge.getNormal().scl(15 * pixel));
				float rotation = edge.getV1().cpy().sub(edge.getV0()).angle();

				AssetFont.getInstance().getSmallFont().drawTransformed(batch, String.format("%.0f", length), position.x,
						position.y, 24 * pixel, rotation, 0, 0, Color.BLUE);
			}
		}
		if (vertices.size() > 2) {
			for (int i = 0; i < vertices.size(); i++) {
				if (!closed && (i == 0 || i == vertices.size() - 1))
					continue;
				int nextVertex = (i + 1) % vertices.size();
				int previousVertex = (i - 1) % vertices.size();
				if (closed && i == 0) {
					previousVertex = vertices.size() - 1;
				}
				if (closed && i == vertices.size() - 1) {
					nextVertex = 0;
				}
				Vector2 vp = vertices.get(previousVertex).cpy().sub(vertices.get(i));
				Vector2 vn = vertices.get(nextVertex).cpy().sub(vertices.get(i));
				float value = (float) Math.toDegrees(MathUtilities.signedAngle(vp, vn));
				Vector2 translation = vp.nor().add(vn.nor()).nor().rotate(180).scl(pixel * 20);
				float rotation = translation.angle() % 180 - 90;
				translation.add(vertices.get(i));
				AssetFont.getInstance().getSmallFont().drawTransformed(batch, String.format("%.0f", value),
						translation.x, translation.y, 24 * pixel, rotation, 0, 0, Color.BLUE);
			}
		}
	}

	private Vector2 alignPoint(Vector2 point) {
		if (vRefPnt == null && hRefPnt == null)
			return point.cpy();
		Vector2 aligned = new Vector2(point);

		if (hRefPnt != null) {
			aligned.x = hRefPnt.x;
		}
		if (vRefPnt != null) {
			aligned.y = vRefPnt.y;
		}

		return aligned;
	}

	private void removeVertex(Vector2 vertex) {
		hoveredEdge = null;
		selectedVertex = null;
		vertices.remove(vertex);
	}

	public void setTempVertex(Vector2 tempVertex) {
		this.tempVertex = tempVertex;
	}

	@Override
	public ArrayList<Vector2> getVertices() {
		return vertices;
	}

	public boolean isClosed() {
		return closed;
	}

	public void setClosed(boolean closed) {
		this.closed = closed;
	}

	public Matrix4 getTransform() {
		return transform;
	}

	public void setTransform(Matrix4 transform) {
		this.transform = transform;
	}

	public static PolygonDrawer getInstance() {
		if (instance == null)
			instance = new PolygonDrawer();
		return instance;
	}

	public static void setInstance(PolygonDrawer instance) {
		PolygonDrawer.instance = instance;
	}

	public List<Vector2> getReferencePoints() {
		return referencePoints;
	}

	public void setReferencePoints(List<Vector2> referencePoints) {
		this.referencePoints = referencePoints;
	}

	
	public void quit() {
		if (!(createMode || editable == null))
			RoomController.getInstance().updateObject(editable);
		clear();
		remove();
		CursorProvider.getInstance().setDefault();
	}
	
	/////////////////////
	/////////////////////
	////////////////////

	class PolygonDrawerListenner extends InputListener {

		private float xoffset, yoffset;

		@Override
		public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
			xoffset = x;
			yoffset = y;
			setCurrentMouseLocation(x, y);
			if (mouseOver == null) {
				if (hoveredEdge != null) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							Wall wall = null;
							if (editable instanceof Wall) {
								wall = (Wall) editable;
							}
							EdgeConfigWindow window = new EdgeConfigWindow(new Shell(), hoveredEdge, wall);
							window.setEdges(edges);
							window.open();
							if (!createMode) {
								InputEvent fakeEvent = new InputEvent();
								fakeEvent.setKeyCode(Input.Keys.ENTER);
								keyDown(fakeEvent, Input.Keys.ENTER);
							}
						}
					});
				} 
			}else {
				if (vertices.contains(mouseOver)) {
					selectedVertex = mouseOver;
				}
			}
			return true;
		}

		@Override
		public void touchUp(InputEvent event, float x, float y, int pointer, int button) {
			if (button == Input.Buttons.LEFT) {
				if (selectedVertex == null && hoveredEdge == null) {
					if (mouseOver != null) {
						vertices.add(mouseOver.cpy());
					} else {
						addVertex(mouseLocation.cpy());
					}
				} else {
					if (selectedVertex != null ) {
						if(! (Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT) || Gdx.input.isKeyPressed(Input.Keys.SHIFT_RIGHT)))
							rectifyPosition(selectedVertex);
					}
				}
			}
			if (button == Input.Buttons.RIGHT) {
				if (mouseOver != null && vertices.contains(mouseOver))
					removeVertex(mouseOver);
			}
			updateEdgesList();
			calculateAngles();
			setTempVertex(null);
			selectedVertex = null;
		}

		@Override
		public void touchDragged(InputEvent event, float x, float y, int pointer) {
			setCurrentMouseLocation(x, y);
			findReferencePoint();
			float xmov = x - xoffset;
			float ymov = y - yoffset;
			if (vertices.size() > 0 && Gdx.input.isButtonPressed(Input.Buttons.LEFT) && selectedVertex == null
					&& hoveredEdge == null) {
				setTempVertex(mouseLocation);
			}
			if (selectedVertex != null) {
				selectedVertex.add(xmov, ymov);
			} 
			xoffset = x;
			yoffset = y;
		}

		private void setCurrentMouseLocation(float x, float y) {
			mouseLocation.set(localToStageCoordinates(new Vector2(x, y)));
		}

		@Override
		public boolean mouseMoved(InputEvent event, float x, float y) {
			setCurrentMouseLocation(x, y);
			findReferencePoint();
			if (mouseOver != null) {
				CursorProvider.getInstance().useHand();
			} else {
				CursorProvider.getInstance().usePencil();
			}

			return false;
		}

		@Override
		public boolean keyDown(InputEvent event, int keycode) {
			if (keycode == Input.Keys.F) {
				closed = !closed;
			}
			if (keycode == Input.Keys.ENTER) {
				end();
				if (createMode) {
					CompoundShape border = new CompoundShape(out);
					CompoundObject geometry = new CompoundObject(border, null);
					geometry.setWorldTransform(transform);
					Display.getDefault().asyncExec(new Runnable() {
						@Override
						public void run() {
							CreateFormWizard formWizard = new CreateFormWizard(geometry);
							WizardDialog formWizarddialogue = new OpenWizardDialog(new Shell(Display.getCurrent()),
									formWizard) {

								@Override
								public void showPage(IWizardPage page) {
									super.showPage(page);
									getShell().pack();
								}

							};
							formWizard.setContainer(formWizarddialogue);
							formWizarddialogue.setModal(true);
							formWizarddialogue.setShellStyle(SWT.APPLICATION_MODAL);
							formWizarddialogue.open();
						}
					});
				} else {
					RoomController.getInstance().updateObject(editable);
				}
				clear();
				remove();
				CursorProvider.getInstance().setDefault();
			} else if (keycode == Input.Keys.ESCAPE) {
				quit();
			}
			return false;
		}

		
		
		public Vector2 getCurrentMouseLocation() {
			return mouseLocation;
		}

		// this function is adapative to the camera zoom
		private void findReferencePoint() {
			hRefPnt = null;
			vRefPnt = null;
			mouseOver = null;
			float closestDstX = Float.MAX_VALUE;
			float closestDstY = Float.MAX_VALUE;
			Vector2 closestPointX = null;
			Vector2 closestPointY = null;
			ArrayList<Vector2> copy = new ArrayList<Vector2>(referencePoints);
			copy.addAll(vertices);
			for (Vector2 point : copy) {
				if(point == selectedVertex)
					continue;
				Vector2 translation = point.cpy().sub(mouseLocation);
				float dstX = Math.abs(translation.x);
				float dstY = Math.abs(translation.y);
				if (Math.abs(dstX - closestDstX) < 1e-6f) { // almost equals
					if (point.dst(mouseLocation) < closestPointX.dst(mouseLocation)) {
						closestDstX = dstX;
						closestPointX = point;
					}
				} else if (dstX < closestDstX) {
					closestDstX = dstX;
					closestPointX = point;
				}

				if (Math.abs(dstY - closestDstY) < 1e-6f) { // almost equals
					if (point.dst(mouseLocation) < closestPointY.dst(mouseLocation)) {
						closestDstY = dstY;
						closestPointY = point;
					}
				} else if (dstY < closestDstY) {
					closestDstY = dstY;
					closestPointY = point;
				}
			}

			if (closestDstX < 20 * getPixelWidth()) {
				hRefPnt = closestPointX;
				if (hRefPnt.dst(mouseLocation) <10 * getPixelWidth())
					mouseOver = hRefPnt;
			}
			if (closestDstY < 20 * getPixelWidth()) {
				vRefPnt = closestPointY;
				if (vRefPnt.dst(mouseLocation) < 10 * getPixelWidth()) {
					if (mouseOver == null) {
						mouseOver = vRefPnt;
					} else {
						mouseOver = mouseOver.dst(mouseLocation) < vRefPnt.dst(mouseLocation) ? mouseOver : vRefPnt;
					}
				}
			}
			if (mouseOver == null)
				selectEdge();

		}

	}

}
