package dressing.ui;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.ConnectException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.frs.supercad.config.ConfigurationManager;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;

import dressing.config.WorkspaceConfiguration;
import dressing.config.persistence.ResourceManagers;
import dressing.util.UtilDTO;
import dressing.util.dto.Accessory;
import dressing.util.dto.AccessoryComboView;
import dressing.util.dto.AccessoryComboViewResponse;
import dressing.util.dto.AccessoryResponse;
import dressing.util.dto.Media;
import param.Accessoire;
import param.ArticleFamily;
import param.impl.AccessoireImpl;
import param.impl.ArticleFamilyImpl;
import param.impl.ImageImpl;
import param.impl.URLImpl;

public class ImportAccessoiresWindow {
	private static String BASE_URL = "https://api.catalogue.supercad.frsdev.ovh/api/v1";

	protected Shell shell;

	/**
	 * Launch the application.
	 * 
	 * @param args
	 */

	public static void main(String[] args) {
		try {
			ImportAccessoiresWindow window = new ImportAccessoiresWindow();
			window.open();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Open the window.
	 */
	public void open() {
		Display display = Display.getDefault();
		AccessoryComboViewResponse accessoryComboViewResponse = new AccessoryComboViewResponse();
		
		BundleContext ctx = FrameworkUtil.getBundle(getClass()).getBundleContext();
		ServiceReference<ConfigurationManager> ref = ctx.getServiceReference(ConfigurationManager.class);
		ConfigurationManager configManager = ctx.getService(ref);
		BASE_URL = configManager.getString("URL_CAD_STORE",BASE_URL);

		try {
			accessoryComboViewResponse = UtilDTO.parseUrlResposne(new URL(BASE_URL + "/accessory/summary"),
					AccessoryComboViewResponse.class);

		} catch (ConnectException e) {
			e.printStackTrace();

			MessageDialog.openError(new Shell(display), "Connexion échouée timeout",
					"Impossible de se connecter au supercad store.Veuillez vérifier l’URL de configuration ou votre connexion Internet.");
			return;
		} catch (IOException e) {
			e.printStackTrace();

			MessageDialog.openError(new Shell(display), "Connexion échouée",
					"Impossible de se connecter au supercad store.Veuillez vérifier l’URL de configuration ou votre connexion Internet.");
			return;
		}

		createContents(accessoryComboViewResponse);
		shell.open();
		shell.layout();
		while (!shell.isDisposed()) {
			if (!display.readAndDispatch()) {
				display.sleep();
			}
		}
	}

	/**
	 * Create contents of the window.
	 */
	protected void createContents(AccessoryComboViewResponse accessoryComboViewResponse) {
		shell = new Shell();
		shell.setSize(800, 600);
		shell.setText("Import Material");
		shell.setLayout(new FillLayout());

		ScrolledComposite scrolledComposite = new ScrolledComposite(shell, SWT.V_SCROLL | SWT.H_SCROLL);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);

		Composite container = new Composite(scrolledComposite, SWT.NONE);
		container.setLayout(new GridLayout(1, false));

		for (AccessoryComboView accessoryView : accessoryComboViewResponse) {
			Composite itemComposite = new Composite(container, SWT.BORDER);
			itemComposite.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
			itemComposite.setLayout(new GridLayout(2, false));
			itemComposite.setBackgroundMode(SWT.INHERIT_FORCE);

			Composite contentGrid = new Composite(itemComposite, SWT.NONE);
			contentGrid.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
			contentGrid.setLayout(new GridLayout(3, false));
			createInfoLabel(contentGrid, "Source: " + accessoryView.getSource(),true);
			createInfoLabel(contentGrid, "Category: " + accessoryView.getCategory(),true);
			createInfoLabel(contentGrid, "Count: " + accessoryView.getCount(),false);

			Button downloadBtn = new Button(itemComposite, SWT.PUSH);
			downloadBtn.setText("Importer");
			downloadBtn.setLayoutData(new GridData(SWT.RIGHT, SWT.TOP, false, false));
			downloadBtn.setData(accessoryView);

			downloadBtn.addListener(SWT.Selection, e -> {
				AccessoryComboView selected = (AccessoryComboView) downloadBtn.getData();
				runDownloadTask(selected);
			});
		}

		container.pack();
		scrolledComposite.setContent(container);
		scrolledComposite.setMinSize(container.computeSize(SWT.DEFAULT, SWT.DEFAULT));

	}

	private void createInfoLabel(Composite parent, String text, boolean expandHorizontally) {
		Label label = new Label(parent, SWT.NONE);
		label.setText(text);
		label.setLayoutData(
				new GridData(expandHorizontally ? SWT.FILL : SWT.LEFT, SWT.CENTER, expandHorizontally, false));
	}
	
	private void runDownloadTask(AccessoryComboView accessoryComboView) {
		Shell shell = Display.getDefault().getActiveShell();
		ProgressMonitorDialog dialog = new ProgressMonitorDialog(shell);

		try {
			dialog.run(true, true, monitor -> {
				monitor.beginTask("Téléchargement des accessories...", accessoryComboView.getCount().intValue());
				try {
					performDownload(accessoryComboView, monitor);
				} catch (IOException | InterruptedException e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			});
		} catch (InvocationTargetException | InterruptedException e) {
			Throwable cause = e.getCause();
			if (cause instanceof InterruptedException) {
				MessageDialog.openWarning(shell, "Annulé", "L'action a été annulée. Elle sera reprise la prochaine fois, si la reprise est prise en charge.");
			} else {
				MessageDialog.openError(shell, "Erreur", "Échec de l'importation :\n" + cause.getMessage());
				cause.printStackTrace();
			}
		}
	}


	private void performDownload(AccessoryComboView accessoryComboView, IProgressMonitor monitor) throws IOException, InterruptedException {

		String requestUrl = buildTextureCombinationRequestUrl(BASE_URL, accessoryComboView);
		int skipped = 0;
		int downloaded = 0;

			AccessoryResponse accessoryResponse = UtilDTO.parseUrlResposne(new URL(requestUrl),AccessoryResponse.class);

			for (int i = 0; i < accessoryResponse.size(); i++) {
				if (monitor.isCanceled())
					throw new InterruptedException("Téléchargement annulé par l'utilisateur");

				Accessory accessory = accessoryResponse.get(i);
				monitor.subTask("Téléchargement " + (i + 1) + "/" + accessoryResponse.size() + " : " + accessory.getName());
				
				if (materialExists(accessory)) {
					skipped++;
					continue;
				}
				Accessoire newMaterial = createAccessory(accessory);
				addMaterialToSystem(newMaterial);
				downloaded++;
				monitor.worked(1);
			}

			ResourceManagers.getIntance().saveLibraryResource();
			showDownloadSummary(downloaded, skipped);

		}

	private boolean materialExists(Accessory accessory) {
		return ResourceManagers.getIntance().getAccessoireGroup().getAccessoires().stream().anyMatch(acc -> {
			boolean matched = acc.getImportID() == accessory.getId();
			return matched;
		});
	}

	private void addMaterialToSystem(Accessoire accessoire) {
		ResourceManagers.getIntance().getAccessoireGroup().getAccessoires().add(accessoire);
	}

	private Accessoire createAccessory(Accessory accessory) {

		Accessoire acceesoryWeb = new AccessoireImpl();

		if (!accessory.getImages().isEmpty()) {
			
			Media coverImage = accessory.getCoverImage() == null ? accessory.getImages().get(0) : accessory.getCoverImage();
			
			Path imagePath = downloadImage(coverImage.rowUrl);
			ImageImpl image = new ImageImpl();
			image.setName(imagePath.getFileName().toString());
			image.setPath(imagePath.toString());
			image.setOriginalPath(imagePath.toString());
			acceesoryWeb.setImage(image);
		}
		acceesoryWeb.setImportID(accessory.getId());
		acceesoryWeb.setCode(accessory.getCode());
		acceesoryWeb.setName(accessory.getName());
		acceesoryWeb.setPrix(0);
		acceesoryWeb.getUrls().clear();
		acceesoryWeb.getUrls().add(getOrCreateUrl(accessory));
		acceesoryWeb.setParentFamily(getOrCreateArticleFamily(accessory));
		return acceesoryWeb;

	}

	private URLImpl getOrCreateUrl(Accessory accessory) {
		URLImpl urlImpl = new URLImpl();
		urlImpl.setName(accessory.getSource());
		urlImpl.setPath(accessory.getSourceUrl());
		return urlImpl;
	}

	private ArticleFamily getOrCreateArticleFamily(Accessory accessory) {
		String[] categoryPath = accessory.getCategoryTree().split("\\.");

		ArticleFamily current = ResourceManagers.getIntance().getApplication().getArticleFamilies().getFamilles()
				.stream().filter(f -> f.getName().equalsIgnoreCase(accessory.getSource())).findFirst().orElseGet(() -> {
					ArticleFamilyImpl newFamily = new ArticleFamilyImpl();
					newFamily.setName(accessory.getSource());
					ResourceManagers.getIntance().getApplication().getArticleFamilies().getFamilles().add(newFamily);
					ResourceManagers.getIntance().saveLibraryResource();
					return newFamily;
				});

		for (String category : categoryPath) {
			ArticleFamily parent = current;
			current = current.getFamilies().stream().filter(f -> f.getName().equalsIgnoreCase(category)).findFirst()
					.orElseGet(() -> {
						ArticleFamilyImpl newFamily = new ArticleFamilyImpl();
						newFamily.setName(category);
						newFamily.setParentFamily(parent);
						ResourceManagers.getIntance().saveLibraryResource();
						return newFamily;
					});
		}

		return current;

	}


	private void showDownloadSummary(int downloaded, int skipped) {
		String message = String.format("Download complete!\nDownloaded: %d\nSkipped: %d", downloaded, skipped);
		Display.getDefault().asyncExec(() -> {
			MessageDialog.openInformation(Display.getDefault().getActiveShell(), "Download Summary", message);
		});
	}

	private Path downloadImage(String imageUrl) {
		try {
			String fileName = imageUrl.substring(imageUrl.lastIndexOf("/") + 1);
			URL url = new URL(imageUrl);
			InputStream in = url.openStream();
			Path destinationPath = Paths.get(WorkspaceConfiguration.ACCESSOIRES_IMAGES, fileName);
			Files.createDirectories(destinationPath.getParent());
			Files.copy(in, destinationPath, StandardCopyOption.REPLACE_EXISTING);
			in.close();
			return destinationPath;
		} catch (IOException e) {
			e.printStackTrace();
			System.out.println("Failed to download image: " + e.getMessage());
			return null;
		}

	}

	public String buildTextureCombinationRequestUrl(String baseUrl, AccessoryComboView accessoryComboView) {

		StringBuilder requestUrlBuilder = new StringBuilder();

		try {
			String encodedSource = URLEncoder.encode(accessoryComboView.getSource(), "UTF-8");
			String encodedCategory = URLEncoder.encode(accessoryComboView.getCategory(), "UTF-8");
			requestUrlBuilder.append(baseUrl).append("/accessory/filter?").append("source=").append(encodedSource)
					.append("&category=").append(encodedCategory);
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("UTF-8 encoding not supported", e);
		}

		return requestUrlBuilder.toString();
	}

}
