package dressing.mathutils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g2d.PolygonBatch;
import com.badlogic.gdx.graphics.g2d.PolygonSpriteBatch;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.graphics.g3d.utils.DefaultShaderProvider;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;

import dressing.model.ProjectManager;
import gdxapp.screens.room.RoomController;
import geometry.Arc;
import geometry.Ray;

public class Triangle extends Actor{
	
	private  Vector2 v0;
    private  Vector2 v1;
    private  Vector2 v2;

    private Matrix4 worldTransform;


    private final float[] angles = new float[3];
    private final Ray[] bisectors = new Ray[3];
    private Edge[] edges = new Edge[3];

    private Arc circumscribedCircle;
    private Arc incircle;
    private Vector3[] worldCoords = new Vector3[3];


    public Triangle(Vector2 v0, Vector2 v1, Vector2 v2) {
        this.v0 = v0;
        this.v1 = v1;
        this.v2 = v2;
        calculateEdges();
    }
    
    
    
    public Triangle(Vector3 v0, Vector3 v1, Vector3 v2) {
        worldCoords[0] = v0;
        worldCoords[1] = v1;
        worldCoords[2] = v2;
    }
    
    public static Edge complement(Edge edge, Edge otherEdge) {
        Vector2[] vertices = new Vector2[3];
        vertices[0] = edge.getV0();
        vertices[1] = edge.getV1();
        Vector2 communVertex = null;
        if(!(vertices[0].epsilonEquals(otherEdge.getV0()) || vertices[1].epsilonEquals(otherEdge.getV0()))){
            vertices[2] = otherEdge.getV0();
            communVertex = otherEdge.getV1();
        }else{
            vertices[2] = otherEdge.getV1();
            communVertex = otherEdge.getV0();
        }
        Vector2[] edgeV = new Vector2[2];

        int counter = 0;
        for(int i = 0; i < 3; i++){
            if(vertices[i].epsilonEquals(communVertex))
                continue;
            edgeV[counter] = vertices[i];
            counter++;
        }
        return new Edge(edgeV[0],edgeV[1]);
    }

    public void wind(boolean ccw){
        Vector2 v10 = v1.cpy().sub(v0);
        Vector2 v20 = v2.cpy().sub(v0);
        float angle = MathUtilities.signedAngle(v10,v20);
        if((ccw && angle < 0) || !ccw && angle > 0){
            Vector2 bucket = new Vector2(v1.x,v1.y);
            v1.set(v2.x,v2.y);
            v2.set(bucket.x,bucket.y);
        }
    }

    public void calculateAngles(){
        Vector2 u, v;
        u = v1.cpy().sub(v0);
        v = v2.cpy().sub(v0);
        angles[0] = MathUtilities.calculateUnsignedAngle(u,v);
        u = v2.cpy().sub(v1);
        v = v0.cpy().sub(v1);
        angles[1] = MathUtilities.calculateUnsignedAngle(u,v);
        u = v0.cpy().sub(v2);
        v = v1.cpy().sub(v2);
        angles[2] = MathUtilities.calculateUnsignedAngle(u,v);
    }

    public void calculateBissectors(){
        calculateAngles();
        Vector2 bisectorDir = MathUtilities.rotate(new Vector2(v1).sub(v0).nor(), angles[0]/2);
        bisectors[0] = new Ray(v0,bisectorDir.cpy());
        bisectorDir = MathUtilities.rotate(new Vector2(v2).sub(v1).nor(), angles[1]/2);
        bisectors[1] = new Ray(v1,bisectorDir.cpy());
        bisectorDir = MathUtilities.rotate(new Vector2(v0).sub(v2).nor(), angles[2]/2);
        bisectors[2] = new Ray(v2,bisectorDir.cpy());
    }
    public void calculateIncircle(){
        calculateBissectors();
        Vector2 incircleCenter = MathUtilities.getIntersectionPoint(bisectors[0].getUnitaryEdge(),bisectors[1].getUnitaryEdge());
        Vector2 projected = MathUtilities.projectPoint(incircleCenter,v0,v1);
        this.incircle = new Arc(incircleCenter, projected.dst(incircleCenter));
    }

    public void calculateCircumscribedCircle(){
        Edge edge_0 = new Edge(v1,v2);
        Edge edge_1 = new Edge(v0,v2);
        Edge m1 = edge_0.getMedian();
        Edge m2 = edge_1.getMedian();
        Vector2 circumCenter = MathUtilities.getIntersectionPoint(m1,m2);
        this.circumscribedCircle =  new Arc(circumCenter, v1.cpy().sub(circumCenter).len());
    }


    public boolean contains(Vector2 point, float precision){
        Vector2 p0 = this.v0.cpy().sub(point);
        Vector2 p1 = this.v1.cpy().sub(point);
        Vector2 p2 = this.v2.cpy().sub(point);
        float a2 = MathUtilities.calculateUnsignedAngle(p0,p1);
        float a1 = MathUtilities.calculateUnsignedAngle(p0,p2);
        float a0 = MathUtilities.calculateUnsignedAngle(p2,p1);
        double difference = Math.abs(a0 +a1 +a2 - Math.toRadians(360));
        if(difference > precision)
        	return false;
        return true;
    }

    public boolean contains(Vector2 point){
    	return contains(point, 0.001f);
    }
    
    public boolean contains(Edge edge) {
    	for(var edgeX: getEdges()) {
    		if(edgeX.equals(edge, 0.0001f, false))
    			return true;
    	}
    	return false;
    }


    public void draw(ShapeRenderer shapeRenderer) {
        if(circumscribedCircle == null)
            calculateCircumscribedCircle();
        if(incircle == null)
            calculateIncircle();
        shapeRenderer.setColor(Color.BLUE);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.triangle(v0.x, v0.y, v1.x, v1.y,v2.x, v2.y);
        /*shapeRenderer.setColor(Color.RED);
        shapeRenderer.circle(v0.x,v0.y,4);
        shapeRenderer.setColor(Color.GREEN);
        shapeRenderer.circle(v1.x,v1.y,4);
        shapeRenderer.setColor(Color.BLUE);
        shapeRenderer.circle(v2.x,v2.y,4);*/
        shapeRenderer.end();
        if(Gdx.input.isKeyPressed(Input.Keys.C))
            circumscribedCircle.drawCircle(shapeRenderer);
        /*incircle.drawCircle(shapeRenderer);
        for(int i = 0; i < 3; i ++){
            bisectors[i].draw(shapeRenderer);
        }*/
    }
    public void drawFilled(ShapeRenderer shapeRenderer) {
        Vector2 v10 = this.v1.cpy().sub(v0);
        Vector2 v20 = this.v2.cpy().sub(v0);
        float angle = MathUtilities.signedAngle(v10,v20);
        Color winding = (angle > 0.0f)?Color.GREEN:Color.RED;
        shapeRenderer.setColor(winding);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
        shapeRenderer.triangle(v0.x, v0.y, v1.x, v1.y,v2.x, v2.y);
        shapeRenderer.end();
    }




    public Arc getCircumscribedCircle() {
        if(circumscribedCircle == null)
            calculateCircumscribedCircle();
        return circumscribedCircle;
    }

    public void calculateEdges() {
        this.edges[0] = new Edge(v1,v2);
        this.edges[1] = new Edge(v0,v2);
        this.edges[2] = new Edge(v0,v1);
    }

    public Vector3[] getSpaceCoords(Matrix4 view, float depth){
        Vector3[] vertices = new Vector3[3];
        vertices[0] = new Vector3(this.v0,depth).mul(view);
        vertices[1] = new Vector3(this.v1,depth).mul(view);
        vertices[2] = new Vector3(this.v2,depth).mul(view);
        return vertices;
    }

    public void calculatePlaneCoords(Matrix4 transform){
        Vector3 v = worldCoords[0].cpy().mul(transform);
        if(v0 == null)
            v0 = new Vector2();
        if(v1 == null)
            v1 = new Vector2();
        if(v2 == null)
            v2 = new Vector2();

        this.v0.set(v.x,v.y);
        v = worldCoords[1].cpy().mul(transform);
        this.v1.set(v.x,v.y);
        v = worldCoords[2].cpy().mul(transform);
        this.v2.set(v.x,v.y);
    }


    public Vector2 getV0() {
        return v0;
    }

    public Vector2 getV1() {
        return v1;
    }

    public Vector2 getV2() {
        return v2;
    }

    public Vector3[] getWorldCoords() {
        return worldCoords;
    }

    public Edge[] getEdges() {
        if(edges == null)
            calculateEdges();
        return edges;
    }
    @Override
    public String toString() {
        return "triangle: " +  v0.toString() + "|" + v1.toString() + "|" + v2.toString() ;
    }

    public Set<Triangle> images(){
        Set<Triangle> images = new HashSet<>();
        images.add(this);
        Triangle traingle021 = new Triangle(this.v0, this.v2, this.v1);
        Triangle traingle102 = new Triangle(this.v1, this.v0, this.v2);
        Triangle traingle120= new Triangle(this.v1, this.v2, this.v0);
        Triangle traingle201 = new Triangle(this.v2, this.v0, this.v1);
        Triangle traingle210 = new Triangle(this.v2, this.v1, this.v0);
        images.add(traingle021);
        images.add(traingle102);
        images.add(traingle120);
        images.add(traingle201);
        images.add(traingle210);
        return images;
    }
    


    public boolean equivalentTo(Triangle other){
        return  images().contains(other);
    }

    @Override
    public int hashCode() {
        return Objects.hash(v0, v1, v2);
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Triangle triangle = (Triangle) o;
        return v0.epsilonEquals(triangle.v0) && v1.epsilonEquals(triangle.v1) && v2.epsilonEquals(triangle.v2);
    }

    public static void main(String[] args){
        Vector2 a = new Vector2(2,0);
        Vector2 b = new Vector2(5,3);
        Vector2 c = new Vector2(7,2);
        Vector2 cprime = new Vector2(7/2.0f,2/2.0f).scl(2);
        Triangle abc = new Triangle(a,b,c);
        Triangle bac = new Triangle(b,a,c);
        Triangle bacp = new Triangle(b,a,cprime);

        System.out.println("abc equals bca: " + abc.equals(bac) );
        System.out.println("abc equals bca: " + abc.equivalentTo(bac) );
        System.out.println("bca equals bcap: " + bac.equals(bacp) );
        System.out.println("c equals cprime: " + c.epsilonEquals(c) );
        Edge ab = new Edge(a,b);
        Edge ac = new Edge(a,c);
        Edge edge = complement(ab,ac);
        System.out.println(edge.toString());
        System.out.println(8%7 + "");

    }
}
