package gdxapp.shaders;

import java.util.ArrayList;

import org.eclipse.osgi.service.debug.DebugTrace;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Cubemap;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.g3d.Renderable;
import com.badlogic.gdx.graphics.g3d.Shader;
import com.badlogic.gdx.graphics.g3d.attributes.BlendingAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.graphics.g3d.utils.RenderContext;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.utils.GdxRuntimeException;

import dressing.config.WorkspaceConfiguration;

public class SceneShader implements Shader {

	private ShaderProgram program;
    private Camera camera;
    private RenderContext context;
    private ArrayList<Light> lights;
    private boolean mapEnvironment;
	private boolean debug = false;
	private boolean debugMet = false;
	private boolean debugRough = false;
	private boolean debugAlbedo = false;
	private long deltaTime;
	private int directionalLightCounter = 0;
    private int spotLightCounter = 0;
	private int poinctualLightCounter;
	private Cubemap environmentMap;
	private boolean quotsMode;
	
    @Override
    public void init() {
        String vertx = Gdx.files.absolute(WorkspaceConfiguration.SHADERS_PATH + "/scene/vertx.glsl").readString();
        String frag = Gdx.files.absolute(WorkspaceConfiguration.SHADERS_PATH + "/scene/frag.glsl").readString();
        program = new ShaderProgram(vertx,frag);
        if(!program.isCompiled())
            throw new GdxRuntimeException(program.getLog());
        program.pedantic = false;
        
    }

    @Override
    public int compareTo(Shader other) {
        return 0;
    }

    @Override
    public boolean canRender(Renderable instance) {
        return true;
    }

    @Override
    public void begin(Camera camera, RenderContext context) {
        this.camera = camera;
        this.context = context;
        context.setCullFace(GL20.GL_BACK);
        context.setDepthTest(GL20.GL_LESS);
        program.begin();
        program.setUniformMatrix("vp", camera.combined);
        program.setUniformf("camera_position",camera.position);
        //setUpLightning();
        program.setUniformf("gamma", 2.23f);
        program.setUniformf("exposure", 3);
    }

    @Override
    public void render(Renderable renderable) {
    	if(renderable.material instanceof PbrMaterial) {
    		PbrMaterial mat = (PbrMaterial) renderable.material;
    		if(!mat.isReady())
    			mat.prepare();
    	}
        program.setUniformMatrix("model", renderable.worldTransform);
        try {
        	 Matrix4 normal = new Matrix4().set(renderable.worldTransform).cpy().inv().tra();
             program.setUniformMatrix("normal",new Matrix3().set(normal));
        }catch (Exception e) {
			e.printStackTrace();
		}
       
        if(renderable.material.has(TextureAttribute.Diffuse) ){
            TextureAttribute textureAttribute = (TextureAttribute) renderable.material.get(TextureAttribute.Diffuse);
            if(textureAttribute.textureDescription.texture.getTextureData() != null){
                program.setUniformi("albedoMap", context.textureBinder.bind(textureAttribute.textureDescription));
                program.setUniformf("hasAlbedoMap",1.0f);
            }
        }else if(renderable.material.has(ColorAttribute.Diffuse)){
            program.setUniformf("hasAlbedoMap",0.0f);
            Color diffuse = ((ColorAttribute)renderable.material.get(ColorAttribute.Diffuse)).color;
            Vector3 color = new Vector3(diffuse.r,diffuse.g,diffuse.b);
            program.setUniformf("material.albedo",color);
        }
        

               
        program.setUniformf("material.opacity", 1.0f);
        if(renderable.material.has(BlendingAttribute.Type)){
        	BlendingAttribute attr = ((BlendingAttribute)renderable.material.get(BlendingAttribute.Type));
            Float opacity  = attr.opacity;
            context.setBlending(true, attr.sourceFunction, attr.destFunction);
            program.setUniformf("material.opacity",opacity);
        }
        renderable.meshPart.render(program);

    }

    @Override
    public void end() {
        program.end();
    }

    @Override
    public void dispose() {
        program.dispose();
    }

    public void addDirectionalLight(DirectionalLight light) {
        program.setUniformf("directional_lights[" + directionalLightCounter+ "].intensity", light.getIntensity());
        program.setUniformf("directional_lights[" + directionalLightCounter+ "].direction", light.getDirection());
    }
    

    public void addSpotLight(SpotLight light) {
        program.setUniformf("SLights[" + spotLightCounter+ "].intensity", light.getIntensity());
        program.setUniformf("SLights[" + spotLightCounter+ "].direction", light.getDirection());
        program.setUniformf("SLights[" + spotLightCounter+ "].exponent", light.getExponent());
        program.setUniformf("SLights[" + spotLightCounter+ "].cutoff", light.getCutoff());
        program.setUniformf("SLights[" + spotLightCounter+ "].position", light.getPosition());
    }

    public void setUpLightning() {
        //reset counters
        directionalLightCounter = 0;
        spotLightCounter = 0;
        poinctualLightCounter = 0;
        for(Light light: lights) {
        	if(light instanceof DirectionalLight) {
                addDirectionalLight((DirectionalLight) light);
                directionalLightCounter++;
        	}
        	if(light instanceof PonctualLight) {
        		addPoinctualLight((PonctualLight) light);
        		poinctualLightCounter++;
        	}
        }
    }
    

    
    

    public boolean isQuotsMode() {
		return quotsMode;
	}

	public void setQuotsMode(boolean quotsMode) {
		this.quotsMode = quotsMode;
	}

	private void addPoinctualLight(PonctualLight light) {
    	 program.setUniformf("point_lights[" + poinctualLightCounter+ "].intensity",light.getIntensity());
         program.setUniformf("point_lights[" + poinctualLightCounter+ "].position", light.getPosition());
	}

	public ArrayList<Light> getLights() {
        return lights;
    }

    public void setLights(ArrayList<Light> lights) {
        this.lights = lights;
    }
    
    
	public boolean isMapEnvironment() {
		return mapEnvironment;
	}

	public void setMapEnvironment(boolean mapEnvironment) {
		this.mapEnvironment = mapEnvironment;
	}

	public void toggleDebug(String mode) {
		if(mode.contentEquals("normal")) {
			debug = !debug ;
			debugMet = false;
			debugRough = false;
			debugAlbedo = false;
		}
		if(mode.contentEquals("metalness")) {
			debug = false;
			debugRough = false;
			debugAlbedo = false;
			debugMet = !debugMet;
		}
		if(mode.contentEquals("roughness")) {
			debug = false;
			debugMet = false;
			debugAlbedo = false;
			debugRough = !debugRough;
		}
		if(mode.contentEquals("albedo")) {
			debug = false;
			debugMet = false;
			debugRough = false;
			debugAlbedo = !debugAlbedo;
		}
	}

	public Cubemap getEnvironmentMap() {
		return environmentMap;
	}

	public void setEnvironmentMap(Cubemap environmentMap) {
		this.environmentMap = environmentMap;
	}

	public ShaderProgram getProgram() {
		return program;
	}

	public void setProgram(ShaderProgram program) {
		this.program = program;
	}
	
	
  
}
