package gdx.app.camera;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.PerspectiveCamera;
import com.badlogic.gdx.math.Vector3;
import dressing.events.Event;
import dressing.events.EventDriver;
import gdxapp.scenes.SceneEvent;

public class PerspectiveCameraController {
	
	protected Camera camera;
	
	protected Vector3 orbitCenter;
	protected Vector3[] sceneBounds;
	
	public PerspectiveCameraController() {
		orbitCenter = new Vector3();
	}
	
	public void reset() {
		camera.viewportWidth = 4;
		camera.viewportHeight = 4 * (float)Gdx.graphics.getHeight()/Gdx.graphics.getWidth();
		camera.position.set(orbitCenter.cpy().add(0, 0, 2));
		camera.up.set(0,1,0);
		camera.lookAt(orbitCenter.cpy().add(0,0,-1));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
	}
	
	public void moveForward(float amount) {
		amount *= orbitCenter.dst(camera.position)/ camera.far;
		amount = Math.signum(amount) * Math.max(0.1f, Math.abs(amount));
		camera.position.add(camera.direction.cpy().scl(1,0,1).nor().scl(amount));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
	}
	
	public void moveLaterally(float amount) {
		amount *= orbitCenter.dst(camera.position)/ camera.far;
		camera.position.add(camera.up.cpy().nor().crs(camera.direction.cpy().nor()).scl(amount));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
	}
	
	public void moveVertically(float amount) {
		amount *= orbitCenter.dst(camera.position)/ camera.far;
		camera.position.add(new Vector3(0,1,0).scl(amount));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void lookLaterally(float degree) {
//		camera.rotate(camera.up, degree );
//		camera.update();
//		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
	}
	
	public void tild(float degree) {
		float det = camera.direction.cpy().nor().scl(1, 0, 1).len();
		float pitch = (float) Math.toDegrees(Math.atan2(camera.direction.y, det));
		float sum = pitch + degree;
		float clampedValue = Math.min(60, Math.max(-60, sum));
		float rotation = clampedValue - pitch;
		if(rotation * rotation > 0.01f) {
			camera.rotate(camera.direction.cpy().crs(camera.up), rotation);
			Vector3 left = camera.up.cpy().crs(camera.direction).nor();
			camera.direction.set(left.cpy().crs(camera.up).nor());
			camera.up.set(camera.direction.cpy().crs(left).nor());
			camera.update();
			EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
		}
	}
	
	public void adjustFiledOfView(float amount) {
		camera.viewportWidth *= amount;
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
	}

	public Camera getCamera() {
		return camera;
	}

	public void setCamera(Camera camera) {
		this.camera = camera;
	}
	
	public void orbit(float amount) {
		Vector3 cc = camera.position.cpy().sub(orbitCenter);
		float angle = amount/cc.len();
		
		cc.rotate(Vector3.Y, angle);
		Vector3 position = orbitCenter.cpy().add(cc);	
		camera.position.set(position);
		camera.rotate(Vector3.Y, angle);
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
		System.out.println("camera y: " + camera.direction.y);

	}
	
	public void advance(float amount) {
		camera.position.add(camera.direction.cpy().scl(amount));
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void orbit(float y, float z) {
		Vector3 cc = camera.position.cpy().sub(orbitCenter);
		cc.rotate(Vector3.Y, y);
		cc.rotate(Vector3.Z, z);
		Vector3 position = orbitCenter.cpy().add(cc);
		camera.position.set(position);
		camera.up.set(0,1,0);
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}

	public Vector3 getOrbitCenter() {
		return orbitCenter;
	}

	public void setOrbitCenter(Vector3 orbitCenter) {
		this.orbitCenter = orbitCenter;
	}
	

	
	public void handleKeyDown() {}
	
	public void update() {
		handleKeyDown();
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}

	public Vector3[] getSceneBounds() {
		return sceneBounds;
	}

	public void setSceneBounds(Vector3[] sceneBounds) {
		this.sceneBounds = sceneBounds;
	}
	
	/**
	 * @return the horizental field of view in degrees.
	 */
	public float getHFoV() {
		PerspectiveCamera perspectiveCam = (PerspectiveCamera) camera;
		float ratio = perspectiveCam.viewportWidth/ perspectiveCam.viewportHeight;
		double vFoF = Math.toRadians(perspectiveCam.fieldOfView/2.0f);
		double halfAngle = Math.toDegrees(Math.atan(Math.tan(vFoF) * ratio));
		return (float) halfAngle * 2.0f;
		
	}
	
	
}
