package gdxapp.object3d;

import java.io.IOException;
import java.io.PrintStream;
import java.math.MathContext;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import org.frs.svg.LineSegment;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.GL30;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.VertexAttribute;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.collision.BoundingBox;

import dressing.config.WorkspaceConfiguration;
import gdxapp.fabs3d.Quad;

public class RotatedBoundingBox extends BoundingBox {
	
	private final Matrix4 transform = new Matrix4().idt();
	private final Matrix4 inverseTransform = new Matrix4().idt();

	private static ShaderProgram shader;
	private static Mesh mesh;
	
	public RotatedBoundingBox() {
		super(Vector3.Zero, Vector3.Zero);
	}
	
	public RotatedBoundingBox(BoundingBox boundingBox) {
		super(boundingBox);
	}
	
	
	public void normalize() {
		Vector3 min = new Vector3();
		Vector3 max = new Vector3();
		getMin(min);
		getMax(max);
		Vector3 center = min.cpy().add(max).scl(0.5f);
		Vector3 halfSize = max.cpy().sub(min).scl(0.5f);
		Matrix4 scale = new Matrix4().setToScaling(halfSize.cpy().scl(2));
		Matrix4 transform = new Matrix4().setToTranslation(center);
		setTransform(transform.mul(scale));
		set(new Vector3(-0.5f, -0.5f, -0.5f), new Vector3(0.5f, 0.5f, 0.5f));		
	}
	
	private static void loadShader() throws IOException {
		String vertexSrc = Files.readString(Path.of(WorkspaceConfiguration.SHAPE_SHADER_VERTEX));
		String fragSrc = Files.readString(Path.of(WorkspaceConfiguration.SHAPE_SHADER_FRAG));
		shader = new ShaderProgram(vertexSrc, fragSrc);
	}
	
	private static void createMesh() {
		float[] vertices = {
				-0.5f, -0.5f, -0.5f,
				-0.5f, -0.5f,  0.5f,
				-0.5f,  0.5f, -0.5f,
				-0.5f,  0.5f,  0.5f,

				 0.5f, -0.5f, -0.5f,
				 0.5f, -0.5f,  0.5f,
				 0.5f,  0.5f, -0.5f,
				 0.5f,  0.5f,  0.5f,
		};
		short[] indices = {
				0,1, 1,3, 3,2, 2,0,	//left
				5,4, 4,6, 6,7, 7,5, //right
				2,3, 3,7, 7,6, 6,2, //top
				0,4, 4,5, 5,1, 1,0, //bottom,
				1,5, 5,7, 7,3 ,3,1, //front,
				0,4 ,4,6, 6,2, 2,0, //back
		};
				
		mesh = new Mesh(true, 8, 48, VertexAttribute.Position());
		mesh.setVertices(vertices);
		mesh.setIndices(indices);
		
	}
	
	public void draw(Matrix4 viewProjection, Vector3 color, float lineWidth) {
		if(shader == null) {
			try {
				loadShader();
			} catch (IOException e) {
				e.printStackTrace();
				return;
			}
		}
		if(mesh == null)
			createMesh();
		shader.begin();
		Gdx.gl30.glDepthFunc(GL30.GL_ALWAYS);
		Gdx.gl30.glLineWidth(lineWidth);
		shader.setUniformMatrix("vp", viewProjection);
		shader.setUniformMatrix("model", transform);
		shader.setUniformf("color", color);
		shader.setUniformf("opacity", 1.0f);
		mesh.render(shader, GL30.GL_LINES);
		shader.end();
		Gdx.gl30.glLineWidth(1.0f);
		Gdx.gl30.glDepthFunc(GL30.GL_LEQUAL);
	}
	
	public float sdf(Vector3 worldPoint) {
		Vector3 boxPoint = worldPoint.cpy().mul(inverseTransform);   //first world point transformed to box space and then chosing the equivalent point in the positive quarter
		boxPoint.x = Math.abs(boxPoint.x);
		boxPoint.y = Math.abs(boxPoint.y);
		boxPoint.z = Math.abs(boxPoint.z);
		Vector3 halfSize = new Vector3(0.5f,0.5f,0.5f);
		Vector3 q = boxPoint.sub(halfSize);
		Vector3 out = new Vector3(); 
		out.x = Math.max(q.x, 0);
		out.y = Math.max(q.y, 0);
		out.z = Math.max(q.z, 0);
		float in = Math.min(Math.max(q.x,  Math.max(q.y, q.z)), 0.0f);
		return out.len() + in;
	}
	
	

	@Override
	public Vector3 getCorner000(Vector3 out) {
		return super.getCorner000(out).mul(transform);
	}

	@Override
	public Vector3 getCorner001(Vector3 out) {
		return super.getCorner001(out).mul(transform);
	}

	@Override
	public Vector3 getCorner010(Vector3 out) {
		return super.getCorner010(out).mul(transform);
	}

	@Override
	public Vector3 getCorner011(Vector3 out) {
		return super.getCorner011(out).mul(transform);
	}

	@Override
	public Vector3 getCorner100(Vector3 out) {
		return super.getCorner100(out).mul(transform);
	}

	@Override
	public Vector3 getCorner101(Vector3 out) {
		return super.getCorner101(out).mul(transform);
	}

	@Override
	public Vector3 getCorner110(Vector3 out) {
		return super.getCorner110(out).mul(transform);
	}

	@Override
	public Vector3 getCorner111(Vector3 out) {
		return super.getCorner111(out).mul(transform);
	}
	
	

	public Matrix4 getTransform() {
		return transform;
	}

	public void setTransform(Matrix4 transform) {
		this.transform.set(transform);
		this.inverseTransform.set(transform.cpy().inv());
	}

	public List<LineSegment> getEdges() {
		ArrayList<LineSegment> edges = new ArrayList<LineSegment>();
		//height cots
		edges.add(new LineSegment(getCorner000(new Vector3()), getCorner010(new Vector3())));
		edges.add(new LineSegment(getCorner100(new Vector3()), getCorner110(new Vector3())));
		edges.add(new LineSegment(getCorner001(new Vector3()), getCorner011(new Vector3())));
		edges.add(new LineSegment(getCorner101(new Vector3()), getCorner111(new Vector3())));
		//width edges
		edges.add(new LineSegment(getCorner000(new Vector3()), getCorner100(new Vector3())));
		edges.add(new LineSegment(getCorner010(new Vector3()), getCorner110(new Vector3())));
		edges.add(new LineSegment(getCorner001(new Vector3()), getCorner101(new Vector3())));
		edges.add(new LineSegment(getCorner011(new Vector3()), getCorner111(new Vector3())));
		////depth edges
		edges.add(new LineSegment(getCorner000(new Vector3()), getCorner001(new Vector3())));
		edges.add(new LineSegment(getCorner010(new Vector3()), getCorner011(new Vector3())));
		edges.add(new LineSegment(getCorner100(new Vector3()), getCorner101(new Vector3())));
		edges.add(new LineSegment(getCorner110(new Vector3()), getCorner111(new Vector3())));
		
		return edges;
	}

	
	public void debug(PrintStream out) {
		out.println("000:" + getCorner000(new Vector3()).toString() );
		out.println("001:" + getCorner001(new Vector3()).toString() );
		out.println("010:" + getCorner010(new Vector3()).toString() );
		out.println("011:" + getCorner011(new Vector3()).toString() );

		out.println("100:" + getCorner100(new Vector3()).toString() );
		out.println("101:" + getCorner101(new Vector3()).toString() );
		out.println("110:" + getCorner110(new Vector3()).toString() );
		out.println("111:" + getCorner111(new Vector3()).toString() );
	}
	
	public List<Vector3> getCorners() {
		ArrayList<Vector3> corners = new ArrayList<Vector3>();
		corners.add(getCorner000(new Vector3()));
		corners.add(getCorner001(new Vector3()));
		corners.add(getCorner010(new Vector3()));
		corners.add(getCorner011(new Vector3()));

		corners.add(getCorner100(new Vector3()));
		corners.add(getCorner101(new Vector3()));
		corners.add(getCorner110(new Vector3()));
		corners.add(getCorner111(new Vector3()));
		return corners;
	}

	public List<Quad> getSides() {
		ArrayList<Quad> sides = new ArrayList<Quad>();
		//left
		sides.add(new Quad("left", getCorner000(new Vector3()), getCorner001(new Vector3()),
				getCorner010(new Vector3()), getCorner011(new Vector3())));
		//right
		sides.add(new Quad("right", getCorner100(new Vector3()), getCorner101(new Vector3()),
				getCorner110(new Vector3()), getCorner111(new Vector3())));
		//top
		sides.add(new Quad("top", getCorner010(new Vector3()), getCorner011(new Vector3()),
				getCorner110(new Vector3()), getCorner111(new Vector3())));
		//bottom
		sides.add( new Quad("bottom",getCorner000(new Vector3()), getCorner001(new Vector3()),
				getCorner100(new Vector3()), getCorner101(new Vector3())));
		//front
		sides.add(new Quad("front", getCorner001(new Vector3()), getCorner011(new Vector3()),
				getCorner101(new Vector3()), getCorner111(new Vector3())));
		//back
		sides.add(new Quad("back",getCorner000(new Vector3()), getCorner010(new Vector3()),
				getCorner100(new Vector3()), getCorner110(new Vector3())));
		
//		for(Quad quad: sides)
//			System.err.println(quad);
		return sides;
	}
	
	public boolean intersects(RotatedBoundingBox other) {
//		debug(System.out);
		for(Quad quad: this.getSides()) {
			for(Quad quadX: other.getSides()) {
				if(quad.scaledCopy(1.7f).intersect(quadX.scaledCopy(1.2f)))
					return true;
			}
		}
		return false;
	}
	
	public static void main(String...strings) {
		float x = -0.01f;
		Quad q0 = new Quad(new Vector3(), new Vector3(1,0,0), new Vector3(1,0,1), new Vector3(0,0,1));
		Quad q1 = new Quad(new Vector3(0,0,x), new Vector3(1,0,x), new Vector3(1,1,x), new Vector3(0,1,x));
		System.out.println(q0.intersect(q1));
	}

	@Override
	public Vector3 getDimensions(Vector3 out) {
		
		
		Vector3 min = new Vector3(Float.MAX_VALUE, Float.MAX_VALUE, Float.MAX_VALUE);
		Vector3 max = min.cpy().scl(-1);
		
		for(Vector3 corner: getCorners()) {
			min.x = Math.min(min.x, corner.x);
			min.y = Math.min(min.y, corner.y);
			min.z = Math.min(min.z, corner.z);
			max.x = Math.max(max.x, corner.x);
			max.y = Math.max(max.y, corner.y);
			max.z = Math.max(max.z, corner.z);
		}
		out.set(max.sub(min));
		return out;
	}

	@Override
	public Vector3 getCenter(Vector3 out) {
		Vector3 tmpCenter = new Vector3();
		super.getCenter(tmpCenter);
		out.set(tmpCenter.mul(transform));
		return out;
	}

	public float getRadius() {
		return getDimensions(new Vector3()).scl(0.5f).len();
	}

	@Override
	public BoundingBox clr() {
		set(new Vector3(), new Vector3());
		transform.idt();
		return this;
		
	}

	@Override
	public boolean contains(Vector3 v) {
		return super.contains(v.cpy().mul(inverseTransform));
	}
	
	public BoundingBox getWorldBox() {
		return new BoundingBox(min.cpy().mul(transform), max.cpy().mul(transform)) ;
	}

	public boolean intersectsOther(RotatedBoundingBox other) {
		return getWorldBox().intersects(other.getWorldBox());
	}
	
	
	
	
}
