package gdxapp.assets;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.files.FileHandle;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g2d.GlyphLayout;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.utils.Align;

public class SDFFont extends BitmapFont {

    private final ShaderProgram program;
    private final Matrix4 transform = new Matrix4();

    private static final String VERT =
            """
            attribute vec4 a_position;
            attribute vec2 a_texCoord0;
            attribute vec4 a_color;

            uniform mat4 u_projTrans;
            uniform mat4 u_transform;

            varying vec4 v_color;
            varying vec2 v_texCoord;

            void main() {
                gl_Position = u_projTrans * u_transform * a_position;
                v_color = a_color;
                v_texCoord = a_texCoord0;
            }
            """;

    private static final String FRAG =
            """
            #ifdef GL_ES
            precision mediump float;
            #endif

            uniform sampler2D u_texture;
            uniform float u_smoothing;

            varying vec4 v_color;
            varying vec2 v_texCoord;

            void main() {
                float sd = texture2D(u_texture, v_texCoord).a;

    		// Adaptive smoothing based on true derivative
    		float w = fwidth(sd);
    		w = max(w, 0.002); // optional safety clamp

    		float alpha = smoothstep(0.5 - w, 0.5 + w, sd);

    		gl_FragColor = vec4(v_color.rgb, v_color.a * alpha);
            }
            """;

    public SDFFont(FileHandle fnt, TextureRegion region, boolean flip) {
        super(fnt, region, flip);

        program = new ShaderProgram(VERT, FRAG);

        if (!program.isCompiled()) {
            Gdx.app.error("SDFFont", "Shader compile error:\n" + program.getLog());
        }

        // Always draw glyphs at scale = 1. Transform happens in shader.
        this.getData().setScale(1f, 1f);
    }

    /**
     * Draw with a transform (position, scale, rotation).
     *
     * @param batch    SpriteBatch
     * @param text     Char sequence
     * @param x        world x
     * @param y        baseline y
     * @param scale    size multiplier (1 = original SDF size)
     * @param rotation degrees
     * @param color    draw color
     */
    
    
    public void drawTransformed(Batch batch, CharSequence text,
                                float x, float y,
                                float scale,
                                float rotation,
                                int hAlign,
                                int vAlign,
                                Color color)
    {
        ShaderProgram prev = batch.getShader();
        Color prevColor = batch.getColor();

        batch.setShader(program);
        setColor(color);
        scale /= getData().lineHeight;  
        //scale = 1;
        // Build transform matrix
        transform.idt();
        transform.translate(x, y, 0f);
        transform.rotate(0f, 0f, 1f, rotation);
        transform.scale(scale, scale, 1f);
        
        GlyphLayout layout = new GlyphLayout(this, text);
        Vector2 halfSize = new Vector2(layout.width, layout.height).scl(0.5f);
        Vector2 translation = new Vector2(halfSize);
        translation.x += hAlign == 0? 0 : halfSize.x *  Math.signum(hAlign); 
        translation.y += vAlign == 0?0: halfSize.y * Math.signum(vAlign); 
        
        program.setUniformMatrix("u_transform", transform);
        
        // Draw at (0,0) because transform already handles full positioning
        super.draw(batch, text, -translation.x,  translation.y);
        batch.setShader(prev);
        setColor(prevColor);
    }
    
    public void drawUnscaled(Batch batch, String value, float x, float y, float rotation, float hAlign, float vAlign, Color color) {
    	ShaderProgram prev = batch.getShader();
        Color prevColor = batch.getColor();

        batch.setShader(program);
        setColor(color);
        // Build transform matrix
        transform.idt();
        transform.translate(x, y, 0f);
        transform.rotate(0f, 0f, 1f, rotation);
        
        GlyphLayout layout = new GlyphLayout(this, value);
        Vector2 halfSize = new Vector2(layout.width, layout.height).scl(0.5f);
        Vector2 translation = new Vector2(halfSize);
        translation.x += hAlign == 0? 0 : halfSize.x *  Math.signum(hAlign); 
        translation.y += vAlign == 0?0: halfSize.y * Math.signum(vAlign); 
        program.setUniformMatrix("u_transform", transform);        
        
        super.draw(batch, value, -translation.x,  translation.y);
        batch.setShader(prev);
        setColor(prevColor);
    }
}
