package geometry;

import java.util.ArrayList;
import java.util.Arrays;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.scenes.scene2d.Actor;

import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import jakarta.xml.bind.annotation.XmlRootElement;


@XmlRootElement(name="polygon")
public class Polygon extends Actor implements Shape{
	
	private ArrayList<Vector2> vertices = new ArrayList<Vector2>();
	private transient ArrayList<Edge> edges;
	private transient ArrayList<Ray> bisectors;
	private boolean closed;
	
	public Polygon() {}
	
	public Polygon(Vector2...vertices) {
		this.vertices.addAll(Arrays.asList(vertices));
	}
	
	@Override
	public ArrayList<Vector2> getVertices() {
		return vertices;
	}
	
	@Override
	public Vector2 getlastVertex() {
		if(!vertices.isEmpty())
			return vertices.get(vertices.size()-1);
		return null;
	}
	
	public void setVertices(ArrayList<Vector2> vertexList) {
		vertices.clear();
		for(Vector2 vertex: vertexList) {
			vertices.add(vertex);
		}
	}
	
	public boolean isClosed() {
		return closed;
	}
	
	public void setClosed(boolean closed) {
		this.closed = closed;
	}
	
	public ArrayList<Edge> getEdges() {
		if(edges == null)
			calculateEdges();
		return edges;
	}
	public void calculateEdges() {
		if(edges == null) {
			edges = new ArrayList<Edge>();
		}else {
			edges.clear();
		}
		for(int i = 0; i < vertices.size()-1; i++) {
			edges.add(new Edge(vertices.get(i), vertices.get(i+1)));
		}
	}
	
	public void calculateBisectors() {
		if(bisectors == null) {
			bisectors = new ArrayList<Ray>();
		}else {
			bisectors.clear();
		}
		if(vertices.size() > 2) {
			for(int i = 1; i < vertices.size() -1; i++) {
				Vector2 a = vertices.get(i-1).cpy().sub(vertices.get(i)).nor();
				Vector2 b = vertices.get(i+1).cpy().sub(vertices.get(i)).nor();
				Vector2 director = a.add(b).nor();
				bisectors.add(new Ray(vertices.get(i).cpy(), director));
			}
		}
	}
	
	public Vector2 getCenter() {
		Vector2[] bounds = MathUtilities.getBoundariesPlane(this.vertices);
		return bounds[1].add(bounds[0]).scl(0.5f);
	}
	
	public int getWinding() {
		return MathUtilities.getWinding(vertices.toArray(new Vector2[0]));
	}
	

	
//	public void flipWinding() {
//		ArrayList<Vector2> flippedList = new ArrayList<Vector2>();
//		int size = this.vertices.size() - 1;
//		for(int i = size; i >= 0 ;i--) {
//			flippedList.add(this.vertices.get(i));
//		}
//		this.vertices.clear();
//		this.vertices.addAll(flippedList);
//		calculateEdges();
//		calculateBisectors();
//	}
	
	@Override
	public void draw(ShapeRenderer shapeRenderer, Color color) {
		int size = vertices.size() -1;
		shapeRenderer.setColor(color);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
        int nextIndex;
        for(int i = 0; i < size ; i++){
            nextIndex = (i+1)%(vertices.size());
            shapeRenderer.rectLine(vertices.get(i),vertices.get(nextIndex),5);
            shapeRenderer.circle(vertices.get(i).x, vertices.get(i).y,7.0f);
        }
        shapeRenderer.end();
        shapeRenderer.setColor(Color.BLACK);
	}



	@Override
	public void scale(Vector2 scale) {
		for(Vector2 vertex: this.vertices) {
			vertex.scl(scale);
		}
	}

	@Override
	public Shape cpy() {
		Polygon clone = new Polygon();
		clone.setVertices(this.getVertices());
		return clone;
	}

	@Override
	public float distanceTo(Vector2 position) {
		if(edges == null)
			calculateEdges();
		float dst = 1000000000;
		for(Edge edge: edges) {
			float distance = edge.distanceTo(position);
			if(distance < dst) {
				dst = distance;
			}
		}
		return dst;
	}

	@Override
	public Vector2 projectPoint(Vector2 point) {
		ArrayList<Vector2> projections = new ArrayList<Vector2>();
		for(Edge edge: getEdges()) {
			projections.add(edge.projectPoint(point));
		}
		Vector2 closestPoint = projections.get(0);
		for(int i = 1; i < projections.size(); i++) {
			if(point.dst(projections.get(i)) < point.dst(closestPoint))
				closestPoint = projections.get(i);
		}
		return closestPoint;
	}

	@Override
	public Object select(Vector2 position) {
		Edge nearestEdge = null;
		Vector2 nearestVertex = null;
		for(Vector2 vertex: vertices) {
			if(vertex.dst(position) < 10) {
				if(nearestVertex == null) {
					nearestVertex = vertex;
					continue;
				}else {
					if(nearestVertex.dst(position) > vertex.dst(position)) {
						nearestVertex = vertex;
					}
				}
			}
		}
		
		for(Edge edge: edges) {
			Object selection = edge.select(position);
			if(selection != null && selection instanceof Edge) {
				if(nearestEdge == null) {
					nearestEdge = (Edge) selection;
					continue;
				}
				if(nearestEdge.distanceTo(position) > edge.distanceTo(position)) {
					nearestEdge = edge;
				}
			}
		}
		Object selection  = nearestVertex;
		if(nearestEdge != null) {
			if(selection == null || nearestEdge.distanceTo(position) < nearestVertex.dst(position)) {
				selection = nearestEdge;
			}
		}
		return selection;
	}

	public Polygon[] split(Edge edge) {
		if(getEdges().size() == 1)
			return null;
		int cutIndex = vertices.indexOf(edge.getV0());
		if(cutIndex < 0)
			return null;
		
		Polygon first = new Polygon(), second = new Polygon();
		for(int i =0; i <=cutIndex; i++) {
			first.getVertices().add(this.vertices.get(i));
		}
		for(int i =cutIndex+1; i < this.vertices.size(); i++) {
			second.getVertices().add(this.vertices.get(i));
		}
		int j = 0;
		Polygon[] polies = new Polygon[2];
		if(first.getVertices().size() > 1) {
			polies[j] = first;
			j++;
		}
		if(second.getVertices().size() > 1) {
			polies[j] = second;
		}
		
		return polies;
	}
	
	public void fuseCloseVertices() {
		int size = vertices.size();
		boolean end = true;
		for(int i = 0; i < size -1; i++) {
			if(vertices.get(i).dst(vertices.get(i+1)) < 5) {
				vertices.remove(i+1);
				end = false;
				break;
			}
		}
		if(!end)
			fuseCloseVertices();
	}

	public ArrayList<Ray> getBisectors() {
		if(bisectors == null || bisectors.isEmpty()) {
			calculateBisectors();
		}
		calculateBisectors();
		return bisectors;
	}
	
	public void removeDuplicateVertices() {
		boolean clean = false;
		do {
			clean = true;
			for(int i = 0; i < vertices.size()-1; i++) {
				for(int j = i+1; j < vertices.size(); j++) {
					if(vertices.get(j).epsilonEquals(vertices.get(i))) {
						vertices.remove(j);
						clean = false;
						break;
					}
				}
				if(!clean)
					break;
			}
		}while(!clean);
	}
	
	@Override
	public void setWinding(int winding) {
		int currentWinding = MathUtilities.getWinding(vertices.toArray(new Vector2[0]));
		if(winding * currentWinding < 0) {
			ArrayList<Vector2> flippedVertices = new ArrayList<Vector2>();
			for(int i = vertices.size() - 1; i >= 0; i--) {
				flippedVertices.add(vertices.get(i));
			}
			vertices.clear();
			vertices.addAll(flippedVertices);
		}
	}
	
	public static void main(String[] args) {
		Vector2 v0 = new Vector2(0,100), v1 = new Vector2(100,100), v2 = new Vector2(100,0), v3 = new Vector2();
		Polygon poly = new Polygon(v0, v3, v2, v1);
		String winding = poly.getWinding()>0?"ccw": "cw";
		System.out.println("poilygon winding is " + winding);
	}
}
