package gdxapp.object3d;

import java.util.ArrayList;
import java.util.List;

import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.graphics.g3d.model.Node;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import dressing.model.DesignObject3D;
import dressing.model.FaceModel;
import dressing.model.Piece2D;
import dressing.model.PieceSplitManagerImp;
import dressing.model.Space3D;
import dressing.model.Usinage;
import dressing.model.types.Orientation;
import dressing.model.types.PoigneePosition;
import gdxapp.fabs3d.Planche;
import gdxapp.shaders.PbrMaterial;

public class DesignModelFactory {

	private ModelBuilder modelBuilder;
	private static DesignModelFactory factory;

	private DesignModelFactory() {
		modelBuilder = new ModelBuilder();
	}

	public Model createModelFromVertices(KitchenElement object) {
		int materialCounter = 0;
		if (object.getDesignObject() == null)
			return null;
		DesignObject3D design = object.getDesignObject();
		ArrayList<Planche> planches = new ArrayList<Planche>();
		getDesignPlanches(object, design, planches);
		if (modelBuilder == null)
			modelBuilder = new ModelBuilder();

		int counter = 0;
		modelBuilder.begin();
		PbrMaterial material;
		for (Planche planche : planches) {
			try {
				planche.setVertices();
				Node nodeX = modelBuilder.node();
				nodeX.id = "node_" + counter;
				material = planche.getMaterial();
				if (!material.isReady()) {
					material.prepare();
				}
				if (planche.getPiece2D().getMaterialType().isIsColor())
					material.remove(TextureAttribute.Diffuse);
				material.id = "mtl" + ++materialCounter;
				if (planche.getPiece2D() == null) {
					PolygonBuilder.createPolygoneNode(modelBuilder, "planche" + ++counter, planche.getLocalVertices(),
							new ArrayList<Vector2>(), material, planche.getTransform().cpy().inv(),
							new float[] { planche.getThickness() / 2, -planche.getThickness() / 2 });
				} else {
					PolygonBuilder.createPiece3D(modelBuilder, planche);
				}
			}catch (Exception e) {
				e.printStackTrace();
				continue;
			}	
		}

		Model model = modelBuilder.end();

		Node parent = new Node();
		parent.id = "root";
		for (Node child : model.nodes) {
			parent.addChild(child);
		}
		model.nodes.clear();
		model.nodes.add(parent);
		model.calculateTransforms();
		return model;
	}

	public void getDesignPlanches(KitchenElement object, DesignObject3D design, ArrayList<Planche> planches) {
		
		for (DesignObject3D designX : design.getChilds()) {
			if(designX instanceof Usinage ||designX instanceof FaceModel|| !designX.isVisible()) {
				continue;
				
			}
			if (designX instanceof Piece2D)
			{
				if(((Piece2D) designX).isSolid())
				{
					planches.add(new Planche((Piece2D) designX, object));
				}
				
			}
			getDesignPlanches(object, designX, planches);
		}
	}
	

	public Model createSketchFromVertices(KitchenElement object) {
		if (object.getDesignObject() == null)
			return null;
		DesignObject3D design = object.getDesignObject();
		ArrayList<Planche> planches = new ArrayList<Planche>();
		getDesignPlanches(object, design, planches);
		if (modelBuilder == null)
			modelBuilder = new ModelBuilder();
		int counter = 0;
		modelBuilder.begin();
		for (Planche planche : planches) {
			try {
				if (planche.getPiece2D() != null) {
					planche.setVertices();
					Node nodeX = modelBuilder.node();
					nodeX.id = "node_" + counter;
					PolygonBuilder.createPieceSketch(modelBuilder, planche);
				}
			}catch (Exception e) {
				e.printStackTrace();
				continue;
			}
			
		}
		Model model = modelBuilder.end();
		model.calculateTransforms();
		Node root = new Node();
		root.id = "root";
		for (Node node : model.nodes) {
			root.addChild(node);
		}
		model.nodes.clear();
		model.nodes.add(root);
		return model;
	}
	
	public Matrix4 getHandleTransform(DesignObject3D design, Planche planche, PoigneePosition poigneePosition,
			Orientation orientation, Vector3 handleDimension) {
		Matrix4 transform = new Matrix4();
		float verticalPadding = 0.03f, horizentalPadding = 0.03f;
		Vector3 translation = new Vector3(planche.getCenter()).add(0, 0, planche.getHalfExtents().z);
		Quaternion rotation = (orientation == Orientation.HORIZONTAL) ? new Quaternion()
				: new Quaternion(Vector3.Z.cpy(), 90);
		Vector3 dims = new Vector3(handleDimension);
		if (orientation == Orientation.VERTICAL) {
			dims = new Vector3(handleDimension.y, handleDimension.x, handleDimension.z);
		}
		switch (poigneePosition) {
		case TOP_LEFT:
			translation.add(horizentalPadding - planche.getHalfExtents().x + dims.x / 2.0f,
					planche.getHalfExtents().y - verticalPadding - dims.y / 2.0f, 0);
			break;
		case TOP_RIGHT:
			translation.add(-horizentalPadding + planche.getHalfExtents().x - dims.x / 2.0f,
					planche.getHalfExtents().y - verticalPadding - dims.y / 2.0f, 0);
			break;
		case TOP_MIDDLE:
			translation.add(0, planche.getHalfExtents().y - verticalPadding - dims.y / 2.0f, 0);
			break;
		case BOTTOM_LEFT:
			translation.add(horizentalPadding - planche.getHalfExtents().x + dims.x / 2.0f,
					-planche.getHalfExtents().y + verticalPadding + dims.y / 2.0f, 0);
			break;
		case BOTTOM_RIGHT:
			translation.add(-horizentalPadding + planche.getHalfExtents().x - dims.x / 2.0f,
					-planche.getHalfExtents().y + verticalPadding + dims.y / 2.0f, 0);
			break;
		case BOTTOM_MIDDLE:
			translation.add(0, -planche.getHalfExtents().y + verticalPadding + dims.y / 2.0f, 0);
			break;
		case LEFT_MIDDLE:
			translation.add(horizentalPadding - planche.getHalfExtents().x + dims.x / 2.0f, 0, 0);
			break;
		case RIGHT_MIDDLE:
			translation.add(-horizentalPadding + planche.getHalfExtents().x - dims.x / 2.0f, 0, 0);
			break;
		default:
			break;
		}
		transform.set(translation, rotation);
		return transform;
	}


	public static DesignModelFactory getFactory() {
		if (factory == null) {
			factory = new DesignModelFactory();
		}
		return factory;
	}

	public ModelBuilder getModelBuilder() {
		return modelBuilder;
	}
	public boolean isSolid(DesignObject3D solid){
		if(solid instanceof Piece2D)
		{
			Piece2D piece=((Piece2D) solid) ;
			List<Piece2D> parts=PieceSplitManagerImp.getParts(piece);
			return piece.isSolid()&& parts.isEmpty();
		}
//		if(solid instanceof AssemblyContainerModel) {
//			return true;
//		}
		return false;
	}
}
