package dressing.ui.palette;

import java.io.File;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.e4.ui.workbench.modeling.ESelectionService;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.wb.swt.ResourceManager;
import org.eclipse.wb.swt.SWTResourceManager;
import org.osgi.service.event.EventHandler;

import com.badlogic.gdx.math.Vector3;
import com.frs.supercad.dressing.provider.PreviewModelProvider;

import dressing.config.WorkspaceConfiguration;
import dressing.config.persistence.ResourceManagers;
import dressing.ui.util.ImageLoaderCache;
import gdxapp.object3d.ModeledObject;
import gdxapp.object3d.WorldObject;
import param.DesignInstance;
import param.MechanicDesign;
import param.Option;
import param.PieceType;
import param.TypeDef;
import param.TypeDefElement;

/**
 * PaletteComposite is responsible ONLY for:
 * - UI layout and rendering
 * - Navigation between palette groups
 * - Wiring UI interactions (selection, drag, menus)
 *
 * Business logic and model manipulation are delegated
 * to dedicated services.
 */
public class PaletteComposite extends Composite {

    // --------------------------------------------------------------------
    // Constants
    // --------------------------------------------------------------------
    public static final int ITEM_SIZE = 200;
    public static final int ICON_SIZE = 32;
    public static final int TOOLTIP_IMAGE_SIZE = 720;
    private static final int CELL_SPACING = 5;
    public Image FOLDER_ICON;

    public Image DESIGN_ICON;
    private Image ARROW_LEFT;
    private Image ARROW_UP;
    private Image CLOSE_ICON;
    private Image HIDE_ICON;

    // --------------------------------------------------------------------
    // UI State
    // --------------------------------------------------------------------
    private Button btnBack;
    private Button btnUp;
    private FolderNavigationComposite navigationBar;
    private ScrolledComposite scrolledComposite;
    private Composite clientArea;
    private Composite grid;
    private boolean reflowScheduled = false;
    // --------------------------------------------------------------------
    // Navigation / model state
    // --------------------------------------------------------------------
    private PaletteGroup parentGroup;
    private final ConcurrentLinkedDeque<PaletteGroup> previousGroups =
            new ConcurrentLinkedDeque<PaletteGroup>();
    private PaletteProvider paletteProvider;

    // --------------------------------------------------------------------
    // Services
    // --------------------------------------------------------------------
    protected final IEventBroker eventBroker;
    protected final ESelectionService selectionService;
	private Composite buttonBar;
	protected ImageToolTipManager toolTipManager;
	
	// --------------------------------------------------------------------
	// Search
	// --------------------------------------------------------------------
	private org.eclipse.swt.widgets.Text searchText;
	private String currentSearchQuery = "";
	
	// --------------------------------------------------------------------
	// Tag filtering
	// --------------------------------------------------------------------
	private final Set<String> selectedTags = new HashSet<>();
	private Composite tagBar;
	// -----------------------------------------------------
	// Attribute filtering
	// -----------------------------------------------------
	private Shell filterShell;
	private final Map<String, Set<String>> selectedAttributes = new HashMap<>();
	private Button filterButton;
	private Composite filterContent;
	private boolean ignoreNextShellDeactivate = false;
	//preview single thread executor
	private final ExecutorService previewExecutor = Executors.newSingleThreadExecutor( new ThreadFactory() {
		
		@Override
		public Thread newThread(Runnable r) {
			Thread t = new Thread(r);
			t.setDaemon(true);
			t.setPriority(Thread.NORM_PRIORITY + 1);
			return t;
		}
	});
    // --------------------------------------------------------------------
    // Constructor & lifecycle
    // --------------------------------------------------------------------
    public PaletteComposite(Composite parent, int style,
                            IEventBroker eventBroker,
                            ESelectionService selectionService,PaletteProvider paletteProvider) {
        super(parent, style);
        this.eventBroker = eventBroker;
        this.selectionService = selectionService;
        this.paletteProvider =paletteProvider;
        ARROW_LEFT = ResourceManager.getImage(WorkspaceConfiguration.ICONS+File.separator+"icons8-left-24.png") ;
        ARROW_UP = ResourceManager.getImage(WorkspaceConfiguration.ICONS+File.separator+"icons8-up-24.png");
        DESIGN_ICON = getimage(
                WorkspaceConfiguration.getIconsPath()
                        + File.separator + PaletteProvider.DEFAULT_MODEL_ICON,
                ICON_SIZE, ICON_SIZE,true);
        FOLDER_ICON = getimage(
                WorkspaceConfiguration.getIconsPath()
                        + File.separator + PaletteProvider.DEFAULT_FOLDER_ICON,
                ICON_SIZE, ICON_SIZE,true);
        CLOSE_ICON = ResourceManager.getImage(WorkspaceConfiguration.ICONS+File.separator+"cancel_22X22.png");
        HIDE_ICON = ResourceManager.getImage(WorkspaceConfiguration.ICONS+File.separator+"hide24.png");

        toolTipManager = ImageToolTipManager.createImageToolTipManager(getDisplay());
        setLayout(new GridLayout(1, true));
        createHeaderBar();
        createScrolledArea();
       

        // hooks
        hookResizeRefresh(parent);
        hookModelRefresh();

        // initial refresh if needed
        refresh();
    }

    /**
     * Rebuild palette from provider root.
     */
    public void refresh() {
        // recreate the provider project (same as original)
        parentGroup = paletteProvider.createPaletteProject();

        // clear navigation history - match original behavior
        previousGroups.clear();

        // render
        renderGroup(parentGroup);
        if (searchText != null && !searchText.isDisposed()) {
            searchText.setText("");
        }
    }

    // --------------------------------------------------------------------
    // UI creation helpers
    // --------------------------------------------------------------------

    private void createScrolledArea() {
        scrolledComposite = new ScrolledComposite(this, SWT.V_SCROLL|SWT.BORDER);
        scrolledComposite.setExpandHorizontal(true);
        scrolledComposite.setExpandVertical(true);
        scrolledComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        clientArea = new Composite(scrolledComposite, SWT.NONE);
        GridLayout gl = new GridLayout(1, false);
        gl.horizontalSpacing = CELL_SPACING;
        gl.verticalSpacing = CELL_SPACING;
        gl.marginWidth = 0;
        gl.marginHeight = 0;
        clientArea.setLayout(gl);
        
        scrolledComposite.setContent(clientArea);
    }
    
    private void createHeaderBar() {

        // Root header container (2 rows)
        buttonBar = new Composite(this, SWT.NONE);
        GridLayout headerLayout = new GridLayout(1, false);
        headerLayout.marginHeight = 2;
        headerLayout.marginWidth = 2;
        headerLayout.verticalSpacing = 2;
        buttonBar.setLayout(headerLayout);
        buttonBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        createSearchRow(buttonBar);
        createNavigationRow(buttonBar);
    }
    /**
     * First header row: full-width search bar.
     */
    private void createSearchRow(Composite parent) {

        Composite searchRow = new Composite(parent, SWT.NONE);
        GridLayout gl = new GridLayout(1, false);
        gl.marginHeight = 0;
        gl.marginWidth = 0;
        gl.verticalSpacing = 4;
        searchRow.setLayout(gl);
        searchRow.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        // ---- Line 1: text search ----
        Composite textRow = new Composite(searchRow, SWT.NONE);
        GridLayout textLayout = new GridLayout(3, false);
        textLayout.marginHeight = 0;
        textLayout.marginWidth = 0;
        textRow.setLayout(textLayout);
        textRow.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        Label lbl = new Label(textRow, SWT.NONE);
        lbl.setFont(SWTResourceManager.getFont("Segoe UI", 11, SWT.BOLD));
        lbl.setText("Recherche :");

        searchText = new Text(textRow, SWT.SEARCH | SWT.ICON_CANCEL | SWT.BORDER);
        searchText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        searchText.setMessage("Nom ou mot-clé");

        searchText.addListener(SWT.Traverse, e -> {
            if (e.detail == SWT.TRAVERSE_ESCAPE) {
                searchText.setText("");
                e.doit = false;
            }
        });

        searchText.addModifyListener(e -> {
            currentSearchQuery = searchText.getText().trim().toUpperCase();
            applySearchFilter();
        });

        filterButton = new Button(textRow, SWT.TOGGLE);
        filterButton.setText("Filtres \u25BE");
        filterButton.addListener(SWT.FocusIn, e -> {
            ignoreNextShellDeactivate = true;
        });
        filterButton.addListener(SWT.Selection, e -> toggleFilterShell(filterButton));
        // ---- Line 2: tag bar ----
        tagBar = new Composite(searchRow, SWT.NONE);
        RowLayout rowLayout = new RowLayout(SWT.HORIZONTAL);
        rowLayout.marginLeft = 0;
        rowLayout.marginTop = 0;
        rowLayout.spacing = 6;
        tagBar.setLayout(rowLayout);
        tagBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        createFilterShell(filterButton);
    }

    /**
     * Creates the filter shell and its full UI exactly once.
     * The shell is only shown/hidden afterward.
     */
    private void createFilterShell(Control anchor) {

        filterShell = new Shell(anchor.getShell(), SWT.NO_TRIM | SWT.ON_TOP|SWT.NO_FOCUS);
        filterShell.setLayout(new FillLayout());
        filterShell.setVisible(false);

        filterContent = new Composite(filterShell, SWT.NONE);
        filterContent.setLayout(new GridLayout(1, false));

        // Build UI ONCE
        buildAttributeGroups(filterContent);
        createFilterFooter(filterContent);

        filterShell.pack();
        positionShellBelow(anchor, filterShell);

        // Auto-hide behavior
        filterShell.addListener(SWT.Deactivate, e -> {
			if (ignoreNextShellDeactivate) {
				return;
			}


			filterShell.setVisible(false);

			if (!filterButton.isDisposed()) {
				filterButton.setSelection(false);
			}

        });
    }
    
    private void toggleFilterShell(Control anchor) {
        if (filterShell == null || filterShell.isDisposed()) {
            return;
        }
        
        boolean show = !filterShell.isVisible();

        if (show) {
            positionShellBelow(anchor, filterShell);
            filterShell.setVisible(true);

        } else {
            filterShell.setVisible(false);

        }
        Display.getCurrent().asyncExec(() -> ignoreNextShellDeactivate = false);
    }
    
    private void createFilterFooter(Composite parent) {

        Composite footer = new Composite(parent, SWT.NONE);
        footer.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        footer.setLayout(new GridLayout(2, false));

        Button clear = new Button(footer, SWT.PUSH|SWT.CLOSE);
        clear.setText("Effacer les filtres");
        clear.setImage(CLOSE_ICON);
        clear.addListener(SWT.Selection, e -> {

            selectedAttributes.clear();

            // Reset checkboxes
            resetAttributeCheckboxes(filterContent);

            applySearchFilter();
            updateFilterBadge();
        });
        Button close = new Button(footer, SWT.PUSH);
        close.setText("Masquer les filtres");
        close.setImage(HIDE_ICON);

        close.addListener(SWT.Selection, e -> {

        	 filterShell.setVisible(false);
        });
    }
    
    private void resetAttributeCheckboxes(Composite root) {

        for (Control c : root.getChildren()) {

            if (c instanceof Button b && (b.getStyle() & SWT.CHECK) != 0) {
                b.setSelection(false);
            }

            if (c instanceof Composite child) {
                resetAttributeCheckboxes(child);
            }
        }
    }
    
    private void rebuildFilterUI() {
        for (Control c : filterContent.getChildren()) {
            c.dispose();
        }
        buildAttributeGroups(filterContent);
        createFilterFooter(filterContent);
        filterContent.layout(true, true);
    }
    
    private void updateFilterBadge() {

        if (filterButton == null || filterButton.isDisposed()) {
            return;
        }

        int count = getSelectedFilterCount();
        if (count == 0) {
            filterButton.setText("Filtres \u25BE");
        } else {
            filterButton.setText("Filtres (" + count + ")"+"\u25BE");
        }
        filterButton.getParent().layout();
    }
    private int getSelectedFilterCount() {
        return selectedAttributes.values()
                .stream()
                .mapToInt(Set::size)
                .sum();
    }
    
    private void positionShellBelow(Control anchor, Shell shell) {

        Point size = shell.getSize();
        size.y = Math.min(size.y, 420);
        filterShell.setSize(size);
        Point anchorLoc = anchor.toDisplay(0, anchor.getSize().y);

        Rectangle bounds = anchor.getDisplay().getPrimaryMonitor().getClientArea();

        int x = Math.min(anchorLoc.x, bounds.x + bounds.width - size.x);
        int y = Math.min(anchorLoc.y, bounds.y + bounds.height - size.y);

        shell.setLocation(x, y);
    }
    
    private void buildAttributeGroups(Composite parent) {
        
        TypeDef portType=  ResourceManagers.getIntance().getTypeByKey("PORT_TYPE");
        TypeDef portDIRECTION=    ResourceManagers.getIntance().getTypeByKey("PORTE_DIRECTION");
        TypeDef POIGNEEType=   ResourceManagers.getIntance().getTypeByKey("POIGNEE_TYPE");
        TypeDef golaType=   ResourceManagers.getIntance().getTypeByKey("gola_Type");

        Composite content = new Composite(parent, SWT.NONE);
        content.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        content.setLayout(new GridLayout(4, false));

        createAttributeGroup(content, portType);
        createAttributeGroup(content, portDIRECTION);
        createAttributeGroup(content, POIGNEEType);
        createAttributeGroup(content, golaType);
    }
    
    private void createAttributeGroup(Composite parent, TypeDef attribute) {
    	String attributeKey=attribute.getKey().toUpperCase();
    	Set<String> selected = selectedAttributes.computeIfAbsent(attributeKey, k -> new HashSet<>());
        Group group = new Group(parent, SWT.NONE);
        group.setText(attribute.getName());
        group.setLayout(new GridLayout(1, false));
        group.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
        group.setFont(SWTResourceManager.getFont("Times New Roman (Titres CS)", 12, SWT.BOLD) );

        for (TypeDefElement element : attribute.getTypedefelement()) {
        	String elementKey =element.getKey().toUpperCase();
            Button chk = new Button(group, SWT.CHECK);
            chk.setText(element.getName());
            chk.setSelection(selected.contains(elementKey));
            chk.setFont(SWTResourceManager.getFont("Segoe UI", 10, SWT.NORMAL) );

            chk.addListener(SWT.Selection, e -> {
                if (chk.getSelection()) {
                    selected.add(elementKey);
                } else {
                    selected.remove(elementKey);
                }
                applySearchFilter();
            });
        }
    }
    
    /**
     * Second header row: navigation buttons and breadcrumb.
     */
    private void createNavigationRow(Composite parent) {

        Composite navRow = new Composite(parent, SWT.NONE);
        GridLayout gl = new GridLayout(3, false);
        gl.marginHeight = 0;
        gl.marginWidth = 0;
        gl.horizontalSpacing = 4;
        navRow.setLayout(gl);
        navRow.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        btnBack = createNavButton(navRow, ARROW_LEFT, this::navigateBack);
        btnUp   = createNavButton(navRow, ARROW_UP, this::navigateUp);

        navigationBar = new FolderNavigationComposite(navRow, SWT.NONE, null);
        navigationBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        navigationBar.addPaletteGroupSelectionListener(this::renderGroup);
    }

    /**
     * Applies the current search query to the visible palette grid.
     * This method only hides/shows existing cells and never rebuilds the grid.
     */
    private void applySearchFilter() {

        if (grid == null || grid.isDisposed()) {
            return;
        }

        for (Control control : grid.getChildren()) {

            if (!(control instanceof PaletteCellComposite cell)) {
                continue;
            }
            
			PaletteItem item = cell.getItem();
			String name = item != null ? item.getName() : null;
			boolean visible = false;
			if (item instanceof PaletteGroup) {
				visible = matchesText(name);
			} else {
				boolean textMatch = matchesText(name);
				boolean tagMatch = matchesTags(item);
				boolean attributesMatch = matchesAttributes(item);

				visible = textMatch && tagMatch && attributesMatch;
			}

            control.setVisible(visible);

            Object ld = control.getLayoutData();
            if (ld instanceof GridData gd) {
                gd.exclude = !visible;
            }
        }

        // Re-layout grid and update scroll bounds
        grid.layout(true, true);

        int availableWidth = scrolledComposite.getClientArea().width;
        if (availableWidth > 0) {
            Point size = clientArea.computeSize(availableWidth, SWT.DEFAULT, true);
            scrolledComposite.setMinSize(size);
        }
        updateFilterBadge();

    }
    
    private boolean matchesText(String name) {

        if (currentSearchQuery == null || currentSearchQuery.isEmpty()) {
            return true;
        }
        if (name == null || name.isBlank()) {
            return false;
        }

        String upper = name.toUpperCase();
        for (String token : currentSearchQuery.split("\\s+")) {
            if (!upper.contains(token)) {
                return false;
            }
        }
        return true;
    }
    
    private boolean matchesAttributes(PaletteItem item) {

        if (selectedAttributes.isEmpty()) {
            return true;
        }

        Map<String, Set<String>> itemAttrs = item.getAttributes();
        if (itemAttrs == null || itemAttrs.isEmpty()) {
            return true ;
        }

        // AND between attributes
        for (Map.Entry<String, Set<String>> entry : selectedAttributes.entrySet()) {

            Set<String> selectedValues = entry.getValue();
            if (selectedValues.isEmpty()) {
                continue;
            }

            Set<String> itemValues = itemAttrs.get(entry.getKey());
            if (itemValues == null || Collections.disjoint(itemValues, selectedValues)) {
                return false;
            }
        }
        return true;
    }
    
    private boolean matchesTags(PaletteItem item) {

        if (selectedTags.isEmpty()) {
            return true;
        }
        if (item == null || item.getTags() == null) {
            return false;
        }

        // AND logic: item must contain all selected tags
        return item.getTags().containsAll(selectedTags);
    }
    
    private Button createNavButton(Composite parent, Image icon, Runnable action) {
        Button btn = new Button(parent, SWT.PUSH);
        GridData gd = new GridData();
        gd.verticalAlignment=SWT.TOP;
        btn.setLayoutData(gd);
        // keep using the resource manager used by previous code
        btn.setImage(icon); 
        btn.setSize(24, 24);
        btn.addListener(SWT.Selection, e -> action.run());
        return btn;
    }

    // --------------------------------------------------------------------
    // Navigation
    // --------------------------------------------------------------------
    private void navigateBack() {
    	if(previousGroups.isEmpty()) {
    		return;
    	}
        PaletteGroup previous = previousGroups.pop();
        if (previous != null) {
            renderGroup(previous);
            previousGroups.pop();
            btnBack.setEnabled(!previousGroups.isEmpty());

        }
    }

    private void navigateUp() {
        if (parentGroup != null && parentGroup.getParent() != null) {
            renderGroup(parentGroup.getParent());
        }
    }

    // --------------------------------------------------------------------
    // Rendering / lifecycle
    // --------------------------------------------------------------------

    private void renderGroup(PaletteGroup group) {
        if (group == null) {
            return;
        }

        // dispose old grid
        disposeGrid();

        // update state
        if(group!=parentGroup)
        {
        	previousGroups.push(parentGroup);
        }
        btnBack.setEnabled(!previousGroups.isEmpty());
        parentGroup = group;
        btnUp.setEnabled(parentGroup.getParent()!=null);
        navigationBar.setCurrentGroup(group);
//        buttonBar.pack();
        // create new grid
        grid = new Composite(clientArea, SWT.NONE);
        grid.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        int availableWidth = scrolledComposite.getClientArea().width -scrolledComposite.getVerticalBar().getSize().x-scrolledComposite.getBorderWidth()*2;
        int columns = computeColumnCount(availableWidth);
        GridLayout gl = new GridLayout(columns, true); // column count will be updated later
        gl.horizontalSpacing = CELL_SPACING;
        gl.verticalSpacing = CELL_SPACING;
        gl.marginWidth = CELL_SPACING;
        gl.marginHeight = CELL_SPACING;
        grid.setLayout(gl);

        // items
        for (PaletteItem item : group.getItems()) {
            createItemCell(grid, item);
        }

        // subgroups
        for (PaletteGroup subgroup : group.getPalettegroupes()) {
            createGroupCell(grid, subgroup);
        }

        // keep the same parent handler behaviour
        new ParentHandler(grid);
        
        // layout and scrolled settings
        // 🔴 SINGLE POINT OF TRUTH
        refreshScrollToTop();;
        // Reapply search filter if active
//        if (!currentSearchQuery.isEmpty()) {
            applySearchFilter();
//        }
        updateTagBar(group);

    }
    /**
     * Rebuilds the tag bar based on visible group items.
     */
    private void updateTagBar(PaletteGroup group) {

        if (tagBar == null || tagBar.isDisposed()) {
            return;
        }

        for (Control c : tagBar.getChildren()) {
            c.dispose();
        }

        Set<String> tags = new TreeSet<>();

        for (PaletteItem item : group.getItems()) {
            if (item.getTags() != null) {
                tags.addAll(item.getTags());
            }
        }

        selectedTags.clear();

        for (String tag : tags) {
            createTagToggle(tagBar, tag);
        }

        tagBar.layout(true, true);
    }
    
    /**
     * Creates a toggle button used as a tag filter.
     */
    private void createTagToggle(Composite parent, String tag) {

        Button btn = new Button(parent, SWT.TOGGLE);
        btn.setText(tag);
        btn.setFont(SWTResourceManager.getFont("Segoe UI", 9, SWT.NORMAL));

        btn.addListener(SWT.Selection, e -> {
            if (btn.getSelection()) {
                selectedTags.add(tag);
            } else {
                selectedTags.remove(tag);
            }
            applySearchFilter();
        });
    }
    
    private int computeColumnCount(int availableWidth) {
        if (availableWidth <= 0) {
            return 1;
        }
        int columnWidth = ITEM_SIZE + CELL_SPACING;
        return Math.max(1, availableWidth / columnWidth);
    }
    
    private void disposeGrid() {
        if (grid != null && !grid.isDisposed()) {
            grid.dispose();
            grid = null;
        }
    }


    /**
     * Recompute scroll size from the grid ONLY and reset scroll position.
     * This is the single authoritative scroll-sizing method.
     */
    private void refreshScrollToTop() {
        if (scrolledComposite == null || scrolledComposite.isDisposed()) {
            return;
        }
        if (grid == null || grid.isDisposed()) {
            return;
        }

        // Ensure grid computes its real size
        grid.pack(true);

        int availableWidth = scrolledComposite.getClientArea().width -scrolledComposite.getVerticalBar().getSize().x-scrolledComposite.getBorderWidth()*2;
        if (availableWidth <= 0) {
            availableWidth = SWT.DEFAULT;
        }

        Point gridSize = clientArea.computeSize(availableWidth, SWT.DEFAULT, true);

        scrolledComposite.setMinSize(gridSize);
        scrolledComposite.setOrigin(0, 0);

        scrolledComposite.layout(true, true);
        this.layout();
    }
    
    // --------------------------------------------------------------------
    // Cell creation
    // --------------------------------------------------------------------

    private void createItemCell(Composite parent, PaletteItem item) {
        PaletteCellComposite cell =
            new PaletteCellComposite(parent, ITEM_SIZE, item,PaletteCellComposite.IS_CELL_DESIGN);

        Button btn = cell.getButton();
        Button btnIcon = cell.getIconButton();

        if (item.getImage() != null) {
            Image img = getimage(item, ITEM_SIZE, ITEM_SIZE, ImageLoaderCache.isCacheImage);
            cell.setImage(img);
            attachToolTip(btnIcon, item);
        }

        cell.addOverlayPainter(DESIGN_ICON, ITEM_SIZE, ICON_SIZE);
        hookSelection(btn, item.getData());
        PaletteDragService.attach(btn, item);
        addMenuToItem(item, btn);
        cell.getPreviewButton().addListener(SWT.MouseDown, e -> {previewExecutor.submit(() -> previewModel(item));});
    }

    private void createGroupCell(Composite parent, PaletteGroup group) {
    	boolean isDesign = group.getData() instanceof MechanicDesign;
        PaletteCellComposite cell =
            new PaletteCellComposite(parent, ITEM_SIZE, group,isDesign);

        Button btn = cell.getButton();
        Button btnIcon = cell.getIconButton();

        if (group.getImage() != null) {
            Image img = getimage(group, ITEM_SIZE, ITEM_SIZE, ImageLoaderCache.isCacheImage);
            cell.setImage(img);

            if (isDesign) {
                attachToolTip(btnIcon, group);
                cell.getPreviewButton().addListener(SWT.MouseDown, e -> {previewExecutor.submit(() -> previewModel(group));});
            }
        }

        cell.addOverlayPainter(FOLDER_ICON, ITEM_SIZE, ICON_SIZE);

        btn.addListener(SWT.MouseDoubleClick, e -> {
           
            renderGroup(group);
            selectionService.setSelection(group.getData());
        });

        hookSelection(btn, group.getData());
        addMenuToItemGroup(group, btn);
    }

    // --------------------------------------------------------------------
    // Event / UI helpers
    // --------------------------------------------------------------------

    protected void hookSelection(Button button, Object selectionData) {
        if (selectionData == null) {
            return;
        }
        button.addListener(SWT.MouseDown, e -> selectionService.setSelection(selectionData));
        button.addListener(SWT.MouseDoubleClick, e -> selectionService.setSelection(selectionData));
    }

    /**
     * Attach the shared Image tooltip for this display.
     */
    protected void attachToolTip(Button button, PaletteItem item) {
    	Image image = getimage(item, TOOLTIP_IMAGE_SIZE, TOOLTIP_IMAGE_SIZE, !ImageLoaderCache.isCacheImage);
		button.addListener(SWT.MouseHover, e -> {
			
			toolTipManager.show(button, image,item.getName());
		});
        button.addListener(SWT.MouseExit, e -> toolTipManager.hide());
        button.addListener(SWT.MouseDown, e -> toolTipManager.hide());
        button.addListener(SWT.Dispose, e -> {
        	toolTipManager.hide();
        	if(image!=null && !image.isDisposed())
        	{
        		image.dispose();
        	}
        });
      
    }

    // --------------------------------------------------------------------
    // Context menus (placeholders)
    // --------------------------------------------------------------------
    /**
     * Add context menu for an item. Reintroduce your original menu code here.
     */
    private void addMenuToItem(PaletteItem item, Control control) {
        // Placeholder: migrate your original addMenuToItem(...) code here.
        // The previous implementation was present in the original class.
        // For example: create MenuManager, add actions, attach to control.
    }

    /**
     * Add context menu for a group. Reintroduce your original group-menu code here.
     */
    private void addMenuToItemGroup(PaletteGroup group, Control control) {
        // Placeholder: migrate your original addMenuToItemGroup(...) code here.
    }

    // --------------------------------------------------------------------
    // Utilities
    // --------------------------------------------------------------------

    /**
     * Load image via ImageLoaderCache (same contract as original).
     */
    public static Image getimage(PaletteItem item, int width, int height,boolean cache) {
    	if(item==null) {
    		return null;
    	}
    	if(item.getData()!=null && item.getData() instanceof ModeledObject ) {
    		if(item.getImage()!=null && !item.getImage().isEmpty()) {
    			return getimage(item.getImage(), width, height, cache, !item.getImage().contains(WorkspaceConfiguration.getIconsPath()));
    		}
    	}
    	String path =item.getImage();
    	return getimage(path, width, height, cache);
    }
    public static Image getimage(String path, int width, int height,boolean cache) {
    	return getimage(path, width, height, cache,!ImageLoaderCache.isResizedImageSameFolder);
    }
    public static Image getimage(String path, int width, int height,boolean cache, boolean isResizedSameFolder) {
        if (path != null) {
            File image = new File(path);
            if (image.exists()) {
                return ImageLoaderCache.getInstance().loadImage(path, width, height,cache,isResizedSameFolder);
            }
        }
        return null;
    }
    // convenience wrapper
    public static org.eclipse.swt.graphics.Color getSystemColor(int colorIndex) {
        return Display.getCurrent().getSystemColor(colorIndex);
    }

    // --------------------------------------------------------------------
    // Hooks
    // --------------------------------------------------------------------
    
    private void hookResizeRefresh(Composite parent) {
        ControlAdapter listener = new ControlAdapter() {
            @Override
            public void controlResized(ControlEvent e) {
                scheduleReflow();
            }
        };

        addControlListener(listener);

        Shell shell = getShell();
        if (shell != null && !shell.isDisposed()) {
            shell.addControlListener(listener);
        }
    }
    
    private void scheduleReflow() {
        if (reflowScheduled) {
            return;
        }
        reflowScheduled = true;

        Display display = getDisplay();
        if (display == null || display.isDisposed()) {
            return;
        }

        display.asyncExec(() -> {
            if (isDisposed()) {
                return;
            }
            scrolledComposite.setRedraw(false);
            performReflow();
            scrolledComposite.setRedraw(true);
            reflowScheduled = false;
        });
    }
    
    private void performReflow() {
        if (scrolledComposite == null || scrolledComposite.isDisposed()) {
            return;
        }
        if (clientArea == null || clientArea.isDisposed()) {
            return;
        }
        if (grid == null || grid.isDisposed()) {
            return;
        }

        int availableWidth = scrolledComposite.getClientArea().width;
        if (availableWidth <= 0) {
            return;
        }

        // Compute number of columns deterministically
        int columns = computeColumnCount(availableWidth);

        GridLayout layout = (GridLayout) grid.getLayout();
        Control[] gridItems= grid.getChildren();
        columns =Math.min(columns, gridItems.length);
        if (layout.numColumns != columns) {
            layout.numColumns = columns;
            
            grid.layout(true);
            
        }
//        if (!currentSearchQuery.isEmpty()) {
//            applySearchFilter();
//        }
        // Recompute scroll bounds once
        Point size = clientArea.computeSize(availableWidth, SWT.DEFAULT, true);
        scrolledComposite.setMinSize(size);
    }
    
    protected void hookModelRefresh() {
        eventBroker.subscribe("update_DataModel", new EventHandler() {
            @Override
            public void handleEvent(org.osgi.service.event.Event event) {
                // run on UI thread
                Display.getDefault().asyncExec(() -> {
                    try {
                        refresh();
                    } catch (Exception ex) {
                        ex.printStackTrace();
                    }
                });
            }
        });
    }
    
	public void previewModel(PaletteItem paletteItem) {
		Object selection =paletteItem.getData();
		if (selection != null) {
			
			MechanicDesign designCopy=MechanicDesignFactory.createCopy(selection); 
			if(designCopy!=null) {
				selection = designCopy;
			}
			Map<String, Object> eventData = new HashMap<String, Object>();
			if (selection instanceof MechanicDesign) {
				MechanicDesign design = (MechanicDesign) selection;
				
				Map info = PreviewModelProvider.getProvider().getOrCreatePreviewPath(design);
				info.put("has_addons", true);
				eventData.putAll(info);
			} else if (selection instanceof WorldObject) {
				WorldObject worldObject = (WorldObject) selection;
				var info = worldObject.getModel().getInfo();
				String path = (String) info.getProperties().get("path_to_LP");
				if (path != null) {
					eventData.put("value", path);
					Vector3 dims = worldObject.getRealWorldDimension().xyz();
					eventData.put("size", new float[] { dims.x, dims.y, dims.z });
					eventData.put("has_addons", false);

				}
			}
			if(!eventData.isEmpty())
			{
				eventBroker.post("PREVIEW_MODEL", eventData);
			}
		}
	}
	
    @Override
    public void dispose() {
    	if (filterShell != null && !filterShell.isDisposed()) {
            filterShell.dispose();
        }
    	super.dispose();
    	navigationBar.dispose();
    	scrolledComposite.dispose();
    }
    @Override
    protected void checkSubclass() {
        // Allow subclassing of SWT components
    }
}
