package dressing.ui.palette;

import java.io.File;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.wb.swt.ResourceManager;
import org.eclipse.wb.swt.SWTResourceManager;

import dressing.config.WorkspaceConfiguration;

/**
 * Visual cell used by the palette grid.
 * Encapsulates button + label layout and icon overlay support.
 */
public class PaletteCellComposite extends Composite {
	private static final int CELL_MARGIN = 4;
	private static final int MAX_LABEL_LINES = 2;
	public static final int ICON_SIZE = 24;
	public static final boolean IS_CELL_DESIGN = true;
	private final int labelHeight;
    private final Button button;
    private final Label label;
    private PaletteItem item;
	private Button iconButton;
	private Button previewButton;
	private boolean isDesign;
    /**
     * Create a palette cell with a fixed visual structure.
     *
     * @param parent parent composite
     * @param itemSize square size of the main button
     * @param item label text (may be null)
     * @param isDesign 
     * @param icon 
     */
    public PaletteCellComposite(Composite parent, int itemSize, PaletteItem item, boolean isDesign) {
        super(parent, SWT.BORDER);
        this.item = item;
        this.isDesign = isDesign;
        labelHeight = computeLabelHeight(this, MAX_LABEL_LINES);
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 2;
        layout.marginHeight = 2;
        setLayout(layout);

        GridData cellData = new GridData(SWT.FILL, SWT.TOP, true, false);
        cellData.widthHint = itemSize;
        cellData.heightHint = itemSize + labelHeight + CELL_MARGIN;
        if(isDesign) {
        	cellData.heightHint += CELL_MARGIN*2+ICON_SIZE;
        }
        setLayoutData(cellData);
        if(this.isDesign) {
            Composite cellHeader = new Composite(this, SWT.NONE);
            GridLayout headerLayout = new GridLayout(2, false);
            headerLayout.marginHeight = 0;
            headerLayout.marginWidth = 0;
            headerLayout.verticalSpacing = 0;
            headerLayout.marginRight  = 0;

            cellHeader.setLayout(headerLayout);
            GridData cellHeaderData = new GridData(SWT.RIGHT, SWT.TOP, true, false);
            cellHeaderData.heightHint = ICON_SIZE+CELL_MARGIN;
            cellHeader.setLayoutData(cellHeaderData);
            
        	previewButton = new Button(cellHeader, SWT.PUSH);
            
            GridData previewButtonData=  new GridData(SWT.RIGHT, SWT.TOP, true, false);
            previewButtonData.heightHint = ICON_SIZE+CELL_MARGIN;
            previewButton.setLayoutData(previewButtonData);
            previewButton.setImage(ResourceManager.getImage(WorkspaceConfiguration.ICONS+File.separator+"icons8-visible-24.png"));
            previewButton.setText("Visualiser");
            iconButton = new Button(cellHeader, SWT.PUSH);
            
            GridData iconButtonData=  new GridData(SWT.RIGHT, SWT.CENTER, true, false);
            iconButtonData.widthHint=ICON_SIZE;
            iconButtonData.heightHint = ICON_SIZE+CELL_MARGIN;
            iconButton.setLayoutData(iconButtonData);
            iconButton.setImage(ResourceManager.getImage(WorkspaceConfiguration.ICONS+File.separator+"info_alt_obj.png"));
            iconButton.setToolTipText("Visualiseur Rapide");
        }

        button = new Button(this, SWT.NONE);
        GridData buttonData = new GridData(SWT.FILL, SWT.TOP, true, true);
        buttonData.widthHint = itemSize;
        buttonData.heightHint = itemSize;
        button.setLayoutData(buttonData);

        label = new Label(this, SWT.WRAP | SWT.CENTER);
        label.setAlignment(SWT.CENTER);
        label.setText(item == null ? "" : item.getName());
        label.setFont(SWTResourceManager.getFont("Times New Roman (Titres CS)", 10, SWT.BOLD) );
        GridData labelData = new GridData(SWT.FILL, SWT.CENTER, true, false);
        labelData.heightHint = labelHeight;
        label.setLayoutData(labelData);
    }
    
    /**
     * Computes the label height based on current font metrics
     * and a fixed maximum number of lines.
     */
    private static int computeLabelHeight(Control control, int maxLines) {
        GC gc = new GC(control);
        try {
            gc.setFont(SWTResourceManager.getFont("Times New Roman (Titres CS)", 10, SWT.BOLD) );
            FontMetrics metrics = gc.getFontMetrics();

            int lineHeight = metrics.getHeight();
            int verticalPadding = 6; // visual breathing space

            return (lineHeight * maxLines) + verticalPadding;
        } finally {
            gc.dispose();
        }
    }
    
    public PaletteItem getItem() {
		return item;
	}

	public void setItem(PaletteItem item) {
		this.item = item;
	}

	/**
     * @return the main clickable button
     */
    public Button getButton() {
        return button;
    }

    public Button getIconButton() {
		return iconButton;
	}

	public Button getPreviewButton() {
		return previewButton;
	}

	/**
     * Set the button image.
     */
    public void setImage(Image image) {
        button.setImage(image);
    }

    /**
     * Set overlay icon painter.
     */
    public void addOverlayPainter(Image overlay, int itemSize, int iconSize) {
    	
        button.addPaintListener(e ->
        	
            {
            	int width = getClientArea().width;
                if (width <= 0) {
                    return;
                }

                int x = width - iconSize - 10;
                int y = 6;
                if(overlay!=null && !overlay.isDisposed())
            	{
                	e.gc.drawImage(overlay,x,y);
            	}
            }
        );
    }

    @Override
    protected void checkSubclass() {
        // Allow subclassing
    }
}