package dressing.ui.palette;

import org.eclipse.emf.ecore.util.EcoreUtil;

import dressing.model.ProjectManager;
import param.DesignInstance;
import param.MechanicDesign;
import param.MechanicPublicParam;
import param.Option;

/**
 * Factory responsible for creating safe, ready-to-use copies
 * of MechanicDesign instances for drag-and-drop operations.
 */
public final class MechanicDesignFactory {

    private MechanicDesignFactory() {
        // Utility class
    }

    /**
     * Create a deep copy of a design represented by palette data.
     *
     * @param data palette item data
     * @return copied and prepared MechanicDesign, or null
     */
    public static MechanicDesign createCopy(Object data) {
        MechanicDesign design = null;

        if (data instanceof MechanicDesign) {
            design = EcoreUtil.copy((MechanicDesign) data);
        }
        else if (data instanceof DesignInstance instance && instance.getDesign() != null) {
            design = EcoreUtil.copy(instance.getDesign());

            if (instance.getOptions() != null) {
                try {
                    design.injectOption(instance.getOptions().getSelectedOption());
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

            design.setName(instance.getName());

            if (!instance.getAccessoires().getAccessoires().isEmpty()) {
                design.setAccessoires(EcoreUtil.copy(instance.getAccessoires()));
            }
            if (!instance.getQuincailleries().getQuincailleries().isEmpty()) {
                design.setQuincailleries(EcoreUtil.copy(instance.getQuincailleries()));
            }
        }
        else if (data instanceof Option option) {
            design = EcoreUtil.copy(option.getDesign());
            try {
                design.injectOption(option);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if (design != null) {
            adjustAltitude(design);
        }

        return design;
    }

    /**
     * Apply altitude correction based on scene preferences.
     */
    private static void adjustAltitude(MechanicDesign design) {
        try {
            MechanicPublicParam ref = design.getPublicParam("element_YReference");
            MechanicPublicParam altitude = design.getPublicParam("element.positionY");

            if (altitude == null || altitude.getDefaultvalue() == null || altitude.getDefaultvalue().isEmpty()) {
                return;
            }

            if (ref != null && ref.getTypedefelement() != null
                    && "TOP".equals(ref.getTypedefelement().getKey())) {

                float wallHeight = 2500f;
                if (ProjectManager.getManager().getCurrentScene() != null) {
                    wallHeight =
                        ProjectManager.getManager().getCurrentScene()
                            .getPreferences()
                            .getWallHeight() * 1000;
                }

                int y = Math.round(wallHeight)
						- Float.valueOf(altitude.getDefaultvalue()).intValue();

                altitude.setDefaultvalue(String.valueOf(y));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
