package dressing.ui.palette;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.wb.swt.ResourceManager;
import org.eclipse.wb.swt.SWTResourceManager;

import dressing.config.WorkspaceConfiguration;

/**
 * Composite responsable de la navigation par dossiers (breadcrumb + sous-dossiers).
 * 
 * Ce composant est totalement autonome :
 * - Gestion du dossier courant
 * - Affichage des boutons de navigation
 * - Popup des sous-dossiers
 */
public class FolderNavigationComposite extends Composite {

	private PaletteGroup currentGroup;

	private Shell popup;
	private PaletteGroup popupGroup;

    private final Image arrowDown;
    private final Image arrowRight;
    private final Image homeIcon;
    private final List<PaletteGroupSelectionListener> selectionListeners = new ArrayList<>();
    private org.eclipse.swt.widgets.Listener popupAutoCloseListener;
	private boolean ignoreNextShellDeactivate = false;
	private boolean shellDeactivateflag = false;

    private List<Button> arrowButtons=new ArrayList<Button>();
    /**
     * Crée le composite de navigation.
     *
     * @param parent le composite parent
     * @param style  style SWT
     * @param initialGroup groupe de départ
     */
    public FolderNavigationComposite(Composite parent, int style, PaletteGroup initialGroup) {
        super(parent, style);
        this.currentGroup = initialGroup;
        RowLayout rL=new RowLayout(SWT.HORIZONTAL);
        rL.marginBottom=0;
        rL.marginLeft = 0;
        rL.marginRight=0;
        rL.marginTop=0;
        rL.marginHeight =0;
        rL.marginWidth=0;
        setLayout(rL);

        arrowDown = ResourceManager.getImage(
                WorkspaceConfiguration.ICONS + "/down-arrow_24.png");
        arrowRight = ResourceManager.getImage(
                WorkspaceConfiguration.ICONS + "/right-arrow_24.png");
        homeIcon = ResourceManager.getImage(
                WorkspaceConfiguration.ICONS + "/house-24.png");
        rebuildNavigation();
        getShell().addListener(SWT.Move, e -> closePopup());
        getShell().addListener(SWT.Resize, e -> closePopup());
        getShell().addListener(SWT.FocusOut, e -> {
        	closePopup();
        	if(shellDeactivateflag)
        	{
        		ignoreNextShellDeactivate =true;
        	}
        });
    }

    /**
     * Reconstruit complètement la barre de navigation.
     */
    private void rebuildNavigation() {
		disposeChildren();
		createGroupButtons(currentGroup);
		layout(true, true);
		getParent().requestLayout();
		pack();
    }

    /**
     * Crée récursivement les boutons représentant le chemin hiérarchique.
     */
    private void createGroupButtons(PaletteGroup group) {
    	if(group==null) {
    		return;
    	}
        if (group.getParent() != null) {
            createGroupButtons(group.getParent());
        }

        Button groupButton = new Button(this, SWT.PUSH);
        groupButton.setText(resolveGroupName(group));
        groupButton.addListener(SWT.Selection, e -> {
        	updateNavigation(group);
        });
        groupButton.setFont(SWTResourceManager.getFont("Segoe UI", 10, SWT.BOLD) );
        if(group.getParent()==null) {
        	groupButton.setImage(homeIcon);
        	groupButton.setSize(24, 24);

        }
        if(group!=currentGroup) {
        	Button arrowButton = new Button(this, SWT.PUSH);
            arrowButton.setImage(arrowRight);
            arrowButton.setSize(24, 24);

            arrowButton.addListener(SWT.Selection, e -> {
                if (isPopupVisibleFor(group) || ignoreNextShellDeactivate) {
                    closePopup();
                    arrowButton.setImage(arrowRight);
                } else {
                    
                    showChildGroups(group, arrowButton);
                    arrowButton.setImage(arrowDown);
                }
                ignoreNextShellDeactivate=false;
                shellDeactivateflag=true;
            });
            arrowButtons.add(arrowButton);
        }
    }

    /**
     * Affiche la popup contenant les sous-groupes.
     */
    private void showChildGroups(PaletteGroup group, Button anchor) {
        closePopup();

        popupGroup = group;
        popup = new Shell(getShell(), SWT.NO_TRIM | SWT.ON_TOP);
        popup.setBackground(SWTResourceManager.getColor(SWT.COLOR_WHITE));
        popup.setLayout(new RowLayout(SWT.VERTICAL));

        Point displayLocation = anchor.toDisplay(0, anchor.getSize().y);
        popup.setLocation(displayLocation.x, displayLocation.y + 5);

        ScrolledComposite scrolled = new ScrolledComposite(popup, SWT.V_SCROLL);
        scrolled.setExpandHorizontal(true);

        Composite content = new Composite(scrolled, SWT.NONE);
        content.setLayout(new RowLayout(SWT.VERTICAL));
        content.setBackground(SWTResourceManager.getColor(SWT.COLOR_WHITE));

        List<PaletteGroup> children = group.getPalettegroupes();
        if (children != null) {
            for (PaletteGroup child : children) {
                Button btn = new Button(content, SWT.PUSH);
                btn.setText(resolveGroupName(child));
                btn.addListener(SWT.Selection, e -> {
                	updateNavigation(child);
                });
            }
        }

        content.pack();
        scrolled.setContent(content);
        scrolled.setMinSize(content.computeSize(SWT.DEFAULT, SWT.DEFAULT));
        Point scrolledSize= scrolled.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        
        popup.pack();
        popup.setSize(scrolledSize.x, Math.min(scrolledSize.y, 600));
        clampPopupLocation(anchor);
        popup.open();
        installPopupAutoClose();
    }
    /**
     * Positionne la popup sous l’ancre tout en la maintenant
     * à l’intérieur du shell parent.
     */
    private void clampPopupLocation(Button anchor) {
        Shell parentShell = getShell();

        Point anchorDisplay = anchor.toDisplay(0, anchor.getSize().y + 5);
        Point popupSize = popup.getSize();
        Point shellOrigin = parentShell.toDisplay(0, 0);
        Point shellSize = parentShell.getSize();

        int x = anchorDisplay.x;
        int y = anchorDisplay.y;

        int shellRight = shellOrigin.x + shellSize.x;
        int shellBottom = shellOrigin.y + shellSize.y;

        // Clamp horizontal
        if (x + popupSize.x > shellRight) {
            x = shellRight - popupSize.x;
        }
        if (x < shellOrigin.x) {
            x = shellOrigin.x;
        }

        // Clamp vertical
        if (y + popupSize.y > shellBottom) {
            y = shellBottom - popupSize.y;
        }
        if (y < shellOrigin.y) {
            y = shellOrigin.y;
        }

        popup.setLocation(x, y);
    }
    
    /**
     * Installe un listener global pour fermer la popup
     * lorsqu’un clic a lieu en dehors.
     */
    private void installPopupAutoClose() {
    	if (popup == null || popup.isDisposed() || !popup.isVisible()) {
    	    return;
    	}
        Display display = getDisplay();

        popupAutoCloseListener = event -> {
            if (popup == null || popup.isDisposed()) {
                return;
            }

            if (event.type == SWT.MouseDown) {
                // Si le clic n’est ni dans la popup ni dans le composite
                if (!isEventInside(event, popup) && !isEventInside(event, this)) {
                    closePopup();
                }
            }
        };

        display.addFilter(SWT.MouseDown, popupAutoCloseListener);
    }
    
    /**
     * Vérifie si un événement souris est contenu dans un contrôle donné.
     */
    private boolean isEventInside(org.eclipse.swt.widgets.Event event, Composite control) {
        Point displayPoint = new Point(event.x, event.y);
        displayPoint= ((Control) event.widget).toDisplay(displayPoint);
        Point controlOrigin = control.toDisplay(0, 0);
        Point size = control.getSize();
        boolean isInside=displayPoint.x >= controlOrigin.x 
                && displayPoint.x <= controlOrigin.x + size.x
                && displayPoint.y >= controlOrigin.y 
                && displayPoint.y <= controlOrigin.y + size.y;
        return isInside;
    }
    
    public PaletteGroup getCurrentGroup() {
		return currentGroup;
	}

    /**
     * method for external notifiers to update the current group
     * @param group
     */
	public void setCurrentGroup(PaletteGroup group) {
		if(group!= this.currentGroup)
		{
			this.currentGroup = group;
			closePopup();
	        rebuildNavigation();
	        
		}
	}
	/**
	 * method for internal notifiers to update the current group
	 * @param group
	 */
	private void updateNavigation(PaletteGroup group) {
		if (group != this.currentGroup) {
			this.currentGroup = group;
			closePopup();
			rebuildNavigation();
			fireGroupSelected(group);
		}
	}
	/**
     * Ferme la popup si elle existe.
     */
    private void closePopup() {
        if (popup != null && !popup.isDisposed()) {
            popup.dispose();
        }
        popup = null;
        popupGroup = null;
        if (popupAutoCloseListener != null) {
            getDisplay().removeFilter(SWT.MouseDown, popupAutoCloseListener);
            popupAutoCloseListener = null;
        }
		arrowButtons.forEach(button -> {
			if (button != null && !button.isDisposed()) {
				button.setImage(arrowRight);
			}
		});
    }

    /**
     * Vérifie si la popup actuelle correspond déjà au groupe donné.
     */
    private boolean isPopupVisibleFor(PaletteGroup group) {
        return popup != null && !popup.isDisposed()
                && popupGroup != null
                && popupGroup == group;
    }

    /**
     * Supprime tous les enfants du composite.
     */
    private void disposeChildren() {
        for (var child : getChildren()) {
            child.dispose();
        }
        arrowButtons.clear();
    }
    /**
     * Résout le nom affichable d’un groupe.
     */
    private String resolveGroupName(PaletteGroup group) {
        return group.getName() != null ? group.getName() : "";
    }
    
    /**
     * Ajoute un listener de sélection de groupe.
     */
    public void addPaletteGroupSelectionListener(PaletteGroupSelectionListener listener) {
        if (listener != null && !selectionListeners.contains(listener)) {
            selectionListeners.add(listener);
        }
    }

    private void removeAllListeners() {
    	 selectionListeners.clear();
	}
    
    /**
     * Notifie les listeners d’un changement de groupe.
     */
    private void fireGroupSelected(PaletteGroup group) {
        for (PaletteGroupSelectionListener listener : selectionListeners) {
            listener.groupSelected(group);
        }
    }
    
    @Override
    public void dispose() {
//        arrowDown.dispose();
//        arrowRight.dispose();
//        homeIcon.dispose();
        removeAllListeners();
        if(popup!=null && popup.isDisposed())
        {
        	popup.dispose();
        }
        super.dispose();
    }

    /* -------------------------- Test standalone -------------------------- */

    public static void main(String[] args) {
        Display display = new Display();
        Shell shell = new Shell(display);
        shell.setText("Navigation palette");
        shell.setSize(800, 600);
        shell.setLayout(new FillLayout());

        PaletteGroup root = PaletteProvider.getInstance().getPaletteRoot(); // helper de test
        new FolderNavigationComposite(shell, SWT.NONE, root);

        shell.open();
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch()) {
                display.sleep();
            }
        }
        display.dispose();
    }
}
