package dressing.ui.components;

import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Set;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.nebula.jface.tablecomboviewer.TableComboViewer;
import org.eclipse.nebula.widgets.tablecombo.TableCombo;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;

import dressing.ui.engine3d.SceneTexture;
import dressing.ui.util.SingleImageItemLabelProvider;
import param.BaseObject;

public class TableComboViewerSearch extends TableComboViewer {
	private static final int TIME_OUT = 300;
	private static final Set<Object> IGNORED_KEYS = Set.of(SWT.ARROW_UP, SWT.ARROW_DOWN, SWT.CR, SWT.LF, SWT.ESC,
			SWT.KEYPAD_CR);

	final int[] debounceId = { 0 };

	public TableComboViewerSearch(Composite parent, int style) {
		super(parent, style);
		initDynamicSearch();
		addResetFilterListener();

	}

	public TableComboViewerSearch(Composite parent) {
		super(parent);
		initDynamicSearch();
		addResetFilterListener();
	}

	public TableComboViewerSearch(TableCombo tableCombo) {
		super(tableCombo);
		initDynamicSearch();
		addResetFilterListener();
	}

	private void initDynamicSearch() {

		TableCombo tableCombo = getTableCombo();

		tableCombo.getTextControl().addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(KeyEvent e) {
				if (IGNORED_KEYS.contains(e.keyCode)) {
					return;
				}

				int id = ++debounceId[0];

				Display.getCurrent().timerExec(TIME_OUT, () -> {
					if (debounceId[0] == id) {

						tableCombo.setTableVisible(true);
						String text = tableCombo.getText();
                        resetFilters();

                        if (text == null || text.trim().isEmpty()) {
                            refresh();
                            return;
                        }

                        // Precompute tokens ONCE (major speed boost)
                        final String[] tokens = text.toUpperCase().split("\\s+");

						addFilter(new ViewerFilter() {
							@Override
							public boolean select(Viewer viewer, Object parentElement, Object element) {
								String label = null;

								if (element instanceof BaseObject) {
									label = ((BaseObject) element).getSearchLabel();
								} else if (element instanceof SceneTexture) {
									String path = ((SceneTexture) element).getPath();
									label = SingleImageItemLabelProvider.getInstance()
											.readFileLabelProperty(Paths.get(path));
								}

								if (label == null) {
									return false;
								}

								// Pre-uppercase once
								String upper = label.toUpperCase();

								// Use fast loop instead of streams
								for (String t : tokens) {
									if (!upper.contains(t)) {
										return false;
									}
								}

								return true;
							}
						});

						refresh();
						tableCombo.setVisible(true);
						tableCombo.setTableVisible(true);
						tableCombo.setVisibleItemCount(10);

					}
				});
			}

		});
	}

	private void addResetFilterListener() {
		addSelectionChangedListener(new ISelectionChangedListener() {

			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				resetFilters();
			}
		});
	}

}
