package dressing.config.persistence;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import dressing.config.WorkspaceConfiguration;
import param.Application;
import param.ModelRoot;
import param.TypeDef;
import param.TypeDefElement;
import param.util.ParamResourceImpl;

import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.Collections;
import java.util.Timer;
import java.util.TimerTask;
import java.nio.file.StandardWatchEventKinds;
/**
 * Manager that loads and maintains a cache of TypeDef objects (.type files),
 * and monitors the dynamic types folder for live changes.
 */
public class DynamicTypeManager {

    /** Background thread for folder watching */
    private Thread watchThread;

    /** WatchService to detect file system changes */
    private WatchService watchService;

    /** Flag to stop watching when needed */
    private volatile boolean running = false;

    private Application application;
    public static boolean isSaving=false;
    private Timer refreshTimer;
    public DynamicTypeManager(Application application) {
    	this.application = application;
    }
    
    public void init() {
    	attachAutoSaveListener();
		scheduleWatchTypeDirectory();
        Runtime.getRuntime().addShutdownHook(new Thread(this::stopWatching));	
    }
    /**
     * Saves all dynamic TypeDef instances currently registered in the application
     * into individual .type files under the dynamic types directory.
     */
    public void saveDynamicTypes() {
        if (getApplication().getTypes() == null || 
        		getApplication().getTypes().getTypes().isEmpty()) {
            System.out.println("No dynamic types to save.");
            return;
        }
        stopWatching();
        // Save each TypeDef to its own file
        for (TypeDef type : getApplication().getTypes().getTypes()) {
            saveDynamicType(type);
        }
        scheduleWatchTypeDirectory();
    }


	/**
	 * @param dirPath
	 * @param dir
	 * @param type
	 */
	public void saveDynamicType(TypeDef type) {
		if (type == null) return;
		isSaving=true;
		try {

	        // Ensure dynamic types directory exists
	        String dirPath = WorkspaceConfiguration.getDynamicTypesPath();
	        File dir = new File(dirPath);
	        if (!dir.exists()) {
	            dir.mkdirs();
	        }
	        
		    // Construct file name: key + modelId + ".type"
		    String safeKey = (type.getKey() != null) ? type.getKey() : "Type";
		    String safeModelId = (type.getModelId() != null) ? type.getModelId().toString() : "";
		    String fileName = safeKey+"_" + safeModelId + ".type";
		    String fullPath = dirPath + File.separator + fileName;

		    File file = new File(fullPath);
		    URI uri = URI.createFileURI(file.getAbsolutePath());
		    Resource oldResource = type.eResource();
		    // Create or retrieve resource from the shared ResourceSet
		    Resource resource = ResourceManagers.getIntance().getResource(uri);
		    resource.getContents().clear();
		    resource.getContents().add(type);

		    // Save the resource to disk
		    resource.save(ResourceManagers.saveOptions);
		    System.out.println("✅ Saved dynamic type: " + fileName);
		    // If the TypeDef had an old resource with a different URI, check and delete it
		    if (oldResource != null && !oldResource.getURI().equals(uri)) {
		        URI oldUri = oldResource.getURI();

		        if (oldUri.isFile()) {
		            File oldFile = new File(oldUri.toFileString());

		            // Check that the old file is in the dynamic types directory
		            try {
		                String dirCanonical = dir.getCanonicalPath();
		                String oldCanonical = oldFile.getCanonicalPath();

		                if (oldCanonical.startsWith(dirCanonical + File.separator)) {
		                    if (oldFile.exists()) {
		                        try {
		                            oldResource.delete(Collections.EMPTY_MAP);
		                        } catch (Exception e) {
		                            e.printStackTrace();
		                        }
		                        boolean deleted = oldFile.delete();
		                        if (deleted) {
		                            System.out.println("🗑️ Ancien fichier supprimé : " + oldFile.getName());
		                        } else {
		                            System.err.println("⚠️ Impossible de supprimer l'ancien fichier : " + oldFile.getName());
		                        }
		                    }
		                }
		            } catch (IOException e) {
		                System.err.println("⚠️ Erreur lors de la vérification du dossier du type : " + e.getMessage());
		            }
		        }
		    }
		} catch (Exception e) {
		    System.err.println("❌ Failed to save dynamic type: " + type.getKey());
		    e.printStackTrace();
		}
		isSaving=false;
	}  

	public void copyDynamicTypes() {
		for (ModelRoot root : ResourceManagers.getIntance().getModelroots()) {
			for (TypeDef type : root.getCategorie().getTypedef()) {
				copyDynamicType(type);
			}
		}
		
	}
	
	public void copyDynamicType(TypeDef type) {
		
		TypeDef copy = EcoreUtil.copy(type);
		saveDynamicType(copy);
		ParamResourceImpl copyResource = (ParamResourceImpl) copy.eResource();
		ParamResourceImpl typeResource = (ParamResourceImpl) type.eResource();
		setEmfCopyObjectOriginId(type, copy, copyResource, typeResource);
		for(TypeDefElement element:type.getTypedefelement()) {
			setEmfCopyObjectOriginId(element, copy.getElement(element.getKey()), copyResource, typeResource);
		}
		saveDynamicType(copy);
	}
    // --------------------------------------------------------
    // 🔄 Watch Service: Auto-reload dynamic types
    // --------------------------------------------------------

	/**
	 * @param type
	 * @param copy
	 * @param copyResource
	 * @param typeResource
	 */
	public void setEmfCopyObjectOriginId(EObject type, EObject copy, ParamResourceImpl copyResource,
			ParamResourceImpl typeResource) {
		String id = typeResource.getID(type);
		if(id !=null) {
			copyResource.getIDToEObjectMap().put(id, copy);
			copyResource.getEObjectToIDMap().put(copy, id);
			copyResource.setID(copy, id);
			copyResource.getIntrinsicIDToEObjectMap().put(id, copy);
		}
		copyResource.attachedHelper(copy);
	}

    /**
     * Starts a background thread to watch the dynamic types directory.
     * Reloads only the modified or created file, and removes deleted ones.
     */
    private void startWatching() {
        if (running) return;

        try {
            Path dir = Paths.get(WorkspaceConfiguration.getDynamicTypesPath());
            watchService = FileSystems.getDefault().newWatchService();
            dir.register(watchService,
                    StandardWatchEventKinds.ENTRY_CREATE,
                    StandardWatchEventKinds.ENTRY_MODIFY,
                    StandardWatchEventKinds.ENTRY_DELETE);

            running = true;

            watchThread = new Thread(() -> {
                System.out.println("👁️  Surveillance des fichiers .type démarrée dans : " + dir);
                while (running) {
                    try {
                        WatchKey key = watchService.take();
                        for (WatchEvent<?> event : key.pollEvents()) {
                            WatchEvent.Kind<?> kind = event.kind();
                            Path changed = (Path) event.context();

                            if (!changed.toString().toLowerCase().endsWith(".type")) continue;
                            
                            if(!isSaving) {
                            	File changedFile = dir.resolve(changed).toFile();
                                
                                if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                                    removeDynamicTypeByFileName(changedFile.getAbsolutePath());
                                    System.out.println("🗑️ Type supprimé : " + changedFile.getName());
                                } else {
                                    TypeDef reloaded = ResourceManagers.getIntance().loadDynamicType(changedFile.getAbsolutePath(), false);
                                    if (reloaded != null) {
                                        replaceOrAddDynamicType(reloaded);
                                        System.out.println("🔄 Type rechargé : " + changedFile.getName());
                                    }
                                }
                            }
                            
                        }
                        key.reset();
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        break;
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                System.out.println("👁️  Surveillance des fichiers .type arrêtée.");
            });

            watchThread.setDaemon(true);
            watchThread.start();

        } catch (IOException e) {
            System.err.println("❌ Erreur lors de l'initialisation du WatchService : " + e.getMessage());
        }
    }
    /** Stops the file watching thread safely. */
    public void stopWatching() {
        running = false;
        try {
            if (watchService != null) watchService.close();
        } catch (IOException ignored) { ignored.printStackTrace();}
    }
    
	public void scheduleWatchTypeDirectory() {
//		if (refreshTimer != null) refreshTimer.cancel();
//	    refreshTimer = new Timer();
//	    refreshTimer.schedule(new TimerTask() {
//	        @Override
//	        public void run() {
//				startWatching();
//	        }
//	    }, 30000); // delay 300 ms after last change
	}
    /** Automatically saves when the in-memory model changes. */
    private void attachAutoSaveListener() {
        if (getApplication().getTypes() == null) return;

        getApplication().getTypes().eAdapters().add(new AdapterImpl() {
            @Override
            public void notifyChanged(Notification msg) {
            	if (msg.getEventType() != Notification.SET && msg.getEventType() != Notification.REMOVE) {
            		System.out.println("💾 Auto-saving due to model change...");
            		saveDynamicTypes();
            	}
				
				
				if (msg.getEventType() == Notification.ADD) {
					if (msg.getNewValue() != null && msg.getNewValue() instanceof TypeDef) {
						TypeDef type = (TypeDef) msg.getNewValue();
						addTypeChangeListener(type);
					}
				}
			}

			
        });
    }
    
    /**
	 * @param type
	 */
	public void addTypeChangeListener(TypeDef type) {
		type.eAdapters().add(new AdapterImpl() {
			public void notifyChanged(Notification msg) {
				stopWatching();
				saveDynamicType(type);
				scheduleWatchTypeDirectory();
			}
		});
	}
    /**
     * Replace existing TypeDef in memory by key or add it if not present.
     */
    private void replaceOrAddDynamicType(TypeDef reloaded) {
        EList<TypeDef> list = getApplication().getTypes().getTypes();
        for (int i = 0; i < list.size(); i++) {
            TypeDef existing = list.get(i);
            if (existing.getKey() != null && existing.getKey().equals(reloaded.getKey())) {
                list.set(i, reloaded);
                return;
            }
        }
        list.add(reloaded);
    }

    /**
     * Remove a TypeDef when its file is deleted.
     */
    private void removeDynamicTypeByFileName(String fileName) {
        EList<TypeDef> list = getApplication().getTypes().getTypes();
        list.removeIf(t -> fileName.contentEquals(t.eResource().getURI().toFileString()));
    }

	public Application getApplication() {
		return application;
	}

	public void setApplication(Application application) {
		this.application = application;
	}
    
    
}
