package utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.emf.ecore.util.EcoreUtil;

import com.badlogic.gdx.math.Vector3;

import dressing.model.DesignObject3D;
import dressing.model.Intervale;
import dressing.model.Piece2D;
import dressing.model.types.CaissonType;
import dressing.model.types.Orientation;
import dressing.model.types.PoigneePosition;
import dressing.model.types.PoigneeType;
import dressing.model.types.PortType;
import dressing.model.types.PorteDirection;
import dressing.model.types.PositionReferenceType;
import geometry.Box;
import param.Cavity;
import param.CondtionalEquation;
import param.Equation;
import param.MechanicDesign;
import param.MechanicDesignElment;
import param.MechanicPrivateParam;
import param.MechanicPublicParam;
import param.ParamFactory;
import param.PieceType;
import param.PrivateParamGroup;
import param.PublicParamGroup;
import param.Rainure;
import param.UsinageNode;
import param.UsinageTrou;

public class EmfUtils {

	public static final HashSet<String> unchagableParams = new HashSet<String>(Arrays.asList(
    	    "PORT_TYPE", "POIGNEE_POSITION", "POIGNEE_ORIENTATION", "PORTE_DIRECTION", "POIGNEE_TYPE", "gola_Type","PORTE_MAXWIDTH","global.p","global.l","global.h"
    	));
	public static final HashSet<String> unchagableGloabalParams = new HashSet<String>(Arrays.asList(
    	    "global.ep", "global.epface", "global.epback", "DOS_TYPE", "CAISSON_TYPE", "POSITION_TYPE"
    	));
		public static final String DebordementHaut_Key = "Debordement.haut";
		public static final String DebordementBas_Key = "Debordement.bas";
		public static final String DebordementArriere_Key = "Debordement.Arriere";
		public static final String DebordementAvant_Key = "Debordement.Avant";
		public static final String DebordementGauche_Key = "Debordement.gauche";
		public static final String DebordementDroite_Key = "Debordement.droite";
		
		public static final String DebordementHaut_Equation_Key = "@"+DebordementHaut_Key+"@";
		public static final String DebordementBas_Equation_Key = "@"+DebordementBas_Key+"@";
		public static final String DebordementArriere_Equation_Key = "@"+DebordementArriere_Key+"@";
		public static final String DebordementAvant_Equation_Key ="@"+DebordementAvant_Key+"@";
		public static final String DebordementGauche_Equation_Key = "@"+DebordementGauche_Key+"@";
		public static final String DebordementDroite_Equation_Key = "@"+DebordementDroite_Key+"@";
		
		public static final String DebordementHaut_Label = "Débordement haut";
		public static final String DebordementBas_Label = "Débordement bas";
		public static final String DebordementArriere_Label = "Débordement Arrière";
		public static final String DebordementAvant_Label = "Débordement Avant";
		public static final String DebordementGauche_Label = "Débordement gauche";
		public static final String DebordementDroite_Label = "Débordement droite";
	
	public EmfUtils() {
		
	}
	public static Set<String> extractVariables(String condition) {
        Set<String> variables = new HashSet<>();
        if(condition==null|| condition.isEmpty()) {
        	return variables;
        }
        // Remove method calls like PARENT.getLongeurext() entirely
        String cleaned = condition.replaceAll("\\b\\w+\\.\\w+\\s*\\([^\\)]*\\)", "");

        // Also remove string literals to avoid matching words inside quotes
        cleaned = cleaned.replaceAll("'[^']*'", "");
        
        // Regex to match variable names (identifiers) not in quotes, not part of method calls
        Pattern pattern = Pattern.compile("\\b[A-Za-z_]\\w*\\b");
        Matcher matcher = pattern.matcher(cleaned);

        // Keywords and literals to ignore
        Set<String> ignore = new HashSet<>(Arrays.asList(
        	    "true", "false", "null", "if", "else", "&&", "==", "||",
        	    "<", ">", "<=", ">=", "!="
        	));

        while (matcher.find()) {
            String token = matcher.group();
            // Ignore operators, literals, and known keywords
            if (!ignore.contains(token) &&
                !token.matches("'[^']*'|\\d+(\\.\\d+)?")
            ) {
                variables.add(token);
            }
        }

        return variables;
    }

	public static Set<String> parseDependencies(String equation) {
		Set<String> dependencies = new HashSet<>();
		String regex = "@\\w*\\.\\w*@";
		Matcher matcher = Pattern.compile(regex).matcher(equation);
		while (matcher.find()) {
			dependencies.add(equation.substring(matcher.start(), matcher.end()));
		}
		return dependencies;
	}

	public static boolean transformPieceElementToMechanicDesign(MechanicDesign design, MechanicDesignElment element) {
		PublicParamGroup publicParamGroup = design.getPublicparamgroup();
		PrivateParamGroup privateParamGroup = design.getPrivateparamgroup();

		List<MechanicPublicParam> publicParams = new ArrayList<MechanicPublicParam>();
		List<MechanicPrivateParam> privateParams = new ArrayList<MechanicPrivateParam>();

		Set<String> dependencies = new HashSet<String>();
		List<Equation> pieceEquations = element.getEquation();
		dependencies.addAll(extractVariables(element.getExistexpression()));
		addEquationsDependencies(dependencies, pieceEquations);
		for (UsinageNode node : element.getUsinagenode()) {
			addUsinageNodeDependencies(dependencies, node);
		}
		for(String key:unchagableGloabalParams) {
			MechanicPublicParam param =	findParamFlexible(publicParamGroup, key);
			if(param!=null) {
				param.setChangeable(false);
			}
		}
		addDependenciesToParams(publicParamGroup, privateParamGroup, publicParams, privateParams, dependencies, new HashSet<>());
		
		MechanicDesign pieceDesign = createPieceMechanicDesign(design, element, publicParams, privateParams);
		design.getMechanicdesign().add(pieceDesign);
		return true;
	}
	private static MechanicDesign createPieceMechanicDesign(MechanicDesign design, MechanicDesignElment element,
			List<MechanicPublicParam> publicParams, List<MechanicPrivateParam> privateParams) {
		MechanicDesign pieceDesign = ParamFactory.eINSTANCE.createMechanicDesign();
		pieceDesign.setDeleteable(false);
		pieceDesign.setName(element.getName());
		pieceDesign.setCode(element.getName());
		pieceDesign.setPublic(true);
		pieceDesign.setType(PieceType.PIECE2D);
		pieceDesign.setVersion("1.0");
		
		pieceDesign.setMechanicelementgroup(ParamFactory.eINSTANCE.createMechanicElementGroup());
		MechanicDesignElment pieceElement = element;
		dressing.model.types.PieceType type = getPieceType(pieceElement);
		String caissonType = getMechanicDesignCaissonType(design);
		//ajouter debordement to facade
		if(type.equals(dressing.model.types.PieceType.PORTE_GAUCHE)|| type.equals(dressing.model.types.PieceType.PORTE_DROITE)) {
			addDebordementToFacade(publicParams, pieceElement);
		}
		else if(!EmfUtils.isElementCoin(caissonType) &&
				(type.equals(dressing.model.types.PieceType.COTE_GAUCHE)|| type.equals(dressing.model.types.PieceType.COTE_DROITE))) {
			addDebordementToCote(publicParams, pieceElement);
		}
		pieceDesign.setPublicparamgroup(ParamFactory.eINSTANCE.createPublicParamGroup());
		pieceDesign.getPublicparamgroup().getMechanicpublicparam().addAll(publicParams);
		pieceDesign.setPrivateparamgroup(ParamFactory.eINSTANCE.createPrivateParamGroup());
		pieceDesign.getPrivateparamgroup().getMechanicprivateparam().addAll(privateParams);
		pieceDesign.getMechanicelementgroup().getMechanicdesignelment().add(pieceElement);
		pieceDesign.setPrincipalmaterial(design.getPrincipalmaterial());
		pieceDesign.setFacadeMaterial(design.getFacadeMaterial());
		pieceDesign.setSecondmaterial(design.getSecondmaterial());
		pieceDesign.setResultDelimeters(ParamFactory.eINSTANCE.createOuterDelimeterGroup());
		return pieceDesign;
	}

	public static void addDebordementToFacade(List<MechanicPublicParam> publicParams,
			MechanicDesignElment pieceElement) {
		MechanicPublicParam debordementHaut = createValuePublicParam(DebordementHaut_Label, DebordementHaut_Key, 0);
		publicParams.add(debordementHaut);
		MechanicPublicParam debordementBas = createValuePublicParam(DebordementBas_Label,DebordementBas_Key,0);
		publicParams.add(debordementBas);
		MechanicPublicParam debordementGauche = createValuePublicParam(DebordementGauche_Label, DebordementGauche_Key, 0);
		
		MechanicPublicParam debordementDroite = createValuePublicParam(DebordementDroite_Label,DebordementDroite_Key,0);
		dressing.model.types.PieceType type = getPieceType(pieceElement);

		for(Equation eq : pieceElement.getEquation()) {
			if(eq.getKey().strip().equals("this.h"))
			{
				String add = " + (" + DebordementBas_Equation_Key + ")"
	                       + " + (" + DebordementHaut_Equation_Key + ")";
	            appendToAll(eq, add);
			}
			
			if(eq.getKey().strip().equals("this.posy"))
			{
				String add = " - (" + DebordementBas_Equation_Key + ")";
	            appendToAll(eq, add);
			}
			if(type.equals(dressing.model.types.PieceType.PORTE_GAUCHE)) {
				publicParams.add(debordementDroite);
				if(eq.getKey().strip().equals("this.l"))
				{
					String add = " + (" + DebordementDroite_Equation_Key + ")";
		            appendToExpression(eq, add);
		            for (CondtionalEquation cond : eq.getCondtionalequation()) {
		            	String exp = cond.getWhenexpression();
		            	if(!exp.replaceAll(" ", "").contains("PORT_TYPE=='FRANCAISE2V'"))
		    	        {
		            		cond.setExpression(cond.getExpression() + add);
		    	        }
		    	    }
				}
				
			}else if(type.equals(dressing.model.types.PieceType.PORTE_DROITE)){
				publicParams.add(debordementGauche);
				if(eq.getKey().strip().equals("this.l"))
				{
					String add = " + (" + DebordementGauche_Equation_Key + ")";
		            appendToExpression(eq, add);
		            for (CondtionalEquation cond : eq.getCondtionalequation()) {
		            	String exp = cond.getWhenexpression();
		            	if(!exp.replaceAll(" ", "").contains("PORT_TYPE=='FRANCAISE2V'"))
		    	        {
		            		cond.setExpression(cond.getExpression() + add);
		    	        }
		    	    }
				}
				if(eq.getKey().strip().equals("this.posx"))
				{
					String add = " - (" + DebordementGauche_Equation_Key + ")";
		            appendToExpression(eq, add);
		            for (CondtionalEquation cond : eq.getCondtionalequation()) {
		            	String exp = cond.getWhenexpression();
		            	if(!exp.replaceAll(" ", "").contains("PORT_TYPE=='FRANCAISE2V'"))
		    	        {
		            		cond.setExpression(cond.getExpression() + add);
		    	        }
		    	    }
				}
			}
		}
	}
	public static void addDebordementToCote(List<MechanicPublicParam> publicParams,
			MechanicDesignElment pieceElement) {
		MechanicPublicParam debordementHaut = createValuePublicParam(DebordementHaut_Label, DebordementHaut_Key, 0);
		publicParams.add(debordementHaut);

		MechanicPublicParam debordementBas = createValuePublicParam(DebordementBas_Label, DebordementBas_Key, 0);
		publicParams.add(debordementBas);

		MechanicPublicParam debordementAvant = createValuePublicParam(DebordementAvant_Label, DebordementAvant_Key, 0);
		debordementAvant.setChangeable(false);
		publicParams.add(debordementAvant);

		MechanicPublicParam debordementArriere = createValuePublicParam(DebordementArriere_Label,
				DebordementArriere_Key, 0);
		publicParams.add(debordementArriere);

		for(Equation eq : pieceElement.getEquation()) {
			if(eq.getKey().strip().equals("this.h"))
			{
				 String add = " + (" + DebordementBas_Equation_Key + ")"
	                       + " + (" + DebordementHaut_Equation_Key + ")";
				 appendToAll(eq, add);
			}
			
			if(eq.getKey().strip().equals("this.posy"))
			{
				 String add = " - (" + DebordementBas_Equation_Key + ")";
		         appendToAll(eq, add);
			}
			if(eq.getKey().strip().equals("this.p"))
			{
				 String add = " + (" + DebordementArriere_Equation_Key + ")"
	                       + " + (" + DebordementAvant_Equation_Key + ")";
				 appendToAll(eq, add);
			}
			
			if(eq.getKey().strip().equals("this.posz"))
			{
				 String add = " - (" + DebordementArriere_Equation_Key + ")";
		         appendToAll(eq, add);
			}
		}
	}
	private static boolean hasConditionals(Equation eq) {
	    return eq.getCondtionalequation() != null && !eq.getCondtionalequation().isEmpty();
	}

	private static void appendToExpression(Equation eq, String addition) {
	    eq.setExpression(eq.getExpression() + addition);
	}

	private static void appendToConditionalExpressions(Equation eq, String addition) {
	    if (!hasConditionals(eq)) return;

	    for (CondtionalEquation cond : eq.getCondtionalequation()) {
	        cond.setExpression(cond.getExpression() + addition);
	    }
	}

	/**
	 * Appends to both the main expression and all conditional expressions.
	 */
	private static void appendToAll(Equation eq, String addition) {
	    appendToExpression(eq, addition);
	    appendToConditionalExpressions(eq, addition);
	}
	
	public static MechanicPublicParam createValuePublicParam(String name, String key, double defaultValue) {
		MechanicPublicParam debordementBas =ParamFactory.eINSTANCE.createMechanicPublicParam();
		debordementBas.setChangeable(true);
		debordementBas.setUserlabel(name);
		debordementBas.setName(name);
		debordementBas.setDefaultvalue(String.valueOf(defaultValue));
		debordementBas.setKey(key);
		return debordementBas;
	}
	private static dressing.model.types.PieceType getPieceType(MechanicDesignElment pieceElement) {
		dressing.model.types.PieceType type = dressing.model.types.PieceType.GENERIC;
		if (pieceElement.getTypedef() != null && pieceElement.getTypedef()!=null && pieceElement.getTypedef().getKey().contentEquals("PieceType")
				&& pieceElement.getTypedefelement() != null && pieceElement.getTypedefelement().getKey() != null
				&& !pieceElement.getTypedefelement().getKey().isEmpty()) {
			String typedefElment = pieceElement.getTypedefelement().getKey();
			try {
				type = dressing.model.types.PieceType.valueOf(typedefElment);
			} catch (Exception e) {
				type = dressing.model.types.PieceType.valueOf(pieceElement.getName());
			}
		} else {

			try {
				type = dressing.model.types.PieceType.valueOf(pieceElement.getName());
			} catch (Exception e) {
			}
		}
		return type;
	}
	private static void addUsinageNodeDependencies(Set<String> dependencies, UsinageNode node) {
		dependencies.addAll(extractVariables(node.getExistexpression()));
		for (Cavity cavity : node.getCavities()) {
			dependencies.addAll(extractVariables(cavity.getExistexpression()));
			addEquationsDependencies(dependencies, cavity.getEquation());
		}
		for (Rainure rainure : node.getUsinageRainure()) {
			dependencies.addAll(extractVariables(rainure.getExistexpression()));
			addEquationsDependencies(dependencies, rainure.getEquation());
		}
		for (UsinageTrou trou : node.getUsinagetrou()) {
			addTrouDependencies(dependencies, trou);
		}
	}
	private static void addTrouDependencies(Set<String> dependencies, UsinageTrou trou) {
		dependencies.addAll(extractVariables(trou.getExistexpression()));
		addEquationsDependencies(dependencies, trou.getEquation());

		dependencies.addAll(parseDependencies(trou.getDiameter()));
		dependencies.addAll(parseDependencies(trou.getOppositeDiameter()));
		dependencies.addAll(parseDependencies(trou.getProfondeur()));
		dependencies.addAll(parseDependencies(trou.getOppositeProfondeur()));

		dependencies.addAll(parseDependencies(trou.getX()));
		dependencies.addAll(parseDependencies(trou.getY()));
		dependencies.addAll(parseDependencies(trou.getZ()));

		dependencies.addAll(extractVariables(trou.getHaveOppositeExpression()));
	}
	
	private static void addDependenciesToParams(PublicParamGroup publicParamGroup, PrivateParamGroup privateParamGroup,
			List<MechanicPublicParam> publicParams, List<MechanicPrivateParam> privateParams, Set<String> dependencies,
			Set<String> visited) {
		for (String var : dependencies) {
			// Clean and normalize dependency name
			String dependString = var.replace("@", "").trim();
			String normalized = normalizeName(dependString);
			// Avoid processing the same dependency more than once
			if (!visited.add(normalized)) {
				continue;
			}

			// Try to find public param (both forms: dot and underscore)
			MechanicPublicParam publicParam = findParamFlexible(publicParamGroup, dependString);
			if (publicParam != null) {
				MechanicPublicParam copiedPublicParam = EcoreUtil.copy(publicParam);
				copiedPublicParam.setImport(true);
				if(unchagableParams.contains(copiedPublicParam.getKey().trim())) {
					copiedPublicParam.setChangeable(false);
				}
				// Only add if not already in list
				if (publicParams.stream().noneMatch(p -> dependString.equals(p.getName()))) {
					publicParams.add(copiedPublicParam);
				}

				String defaultValue = copiedPublicParam.getDefaultvalue();
				if (defaultValue != null && !defaultValue.isEmpty()) {

					Set<String> paramDependencies = parseDependencies(defaultValue);
					if (!paramDependencies.isEmpty()) {
						addDependenciesToParams(publicParamGroup, privateParamGroup, publicParams, privateParams,
								paramDependencies, visited);
					}
				}
				continue;
			}
			// Try to find private param (both forms)
			MechanicPrivateParam privateParam = null;
			try {
				privateParam = findParamFlexible(privateParamGroup, dependString);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			if (privateParam != null) {
				MechanicPrivateParam copiedPrivateParam = EcoreUtil.copy(privateParam);
				copiedPrivateParam.setImport(true);
				// Only add if not already in list
				if (privateParams.stream().noneMatch(p -> namesEqual(p.getName(), dependString))) {
					privateParams.add(copiedPrivateParam);
				}

				String value = copiedPrivateParam.getValue();
				if (value != null && !value.isEmpty()) {
					Set<String> paramDependencies = parseDependencies(value);
					if (!paramDependencies.isEmpty()) {
						addDependenciesToParams(publicParamGroup, privateParamGroup, publicParams, privateParams,
								paramDependencies, visited);
					}
				}
			}

		}
	}
	
	/**
	 * Try to find a param by either its exact name or normalized name
	 * (replacing '.' with '_' and vice versa).
	 */
	public static MechanicPublicParam findParamFlexible(PublicParamGroup group, String name) {
	    if (group == null) return null;
	    String normalized = normalizeName(name);
	    MechanicPublicParam p = group.getParam(name);
	    if (p == null) {
	        p = group.getParam(name.replace('.', '_'));
	    }
	    if (p == null) {
	        p = group.getParam(name.replace('_', '.'));
	    }
	    if (p == null) {
	        p = group.getParam(normalized);
	    }
	    return p;
	}
	
	public static MechanicPrivateParam findParamFlexible(PrivateParamGroup group, String name) throws Exception {
	    if (group == null) return null;
	    String normalized = normalizeName(name);
	    MechanicPrivateParam p = group.getParam(name);
	    if (p == null) {
	        p = group.getParam(name.replace('.', '_'));
	    }
	    if (p == null) {
	        p = group.getParam(name.replace('_', '.'));
	    }
	    if (p == null) {
	        p = group.getParam(normalized);
	    }
	    return p;
	}
	
	/** Normalize names by replacing '.' and '_' uniformly with '_' */
	private static String normalizeName(String name) {
	    return name == null ? "" : name.replace('.', '_');
	}

	/** Check if two names are equivalent (dot/underscore-insensitive) */
	private static boolean namesEqual(String n1, String n2) {
	    return normalizeName(n1).equalsIgnoreCase(normalizeName(n2));
	}

	public static boolean isElementCoin(String caissonType) {

		if (caissonType != null && !caissonType.isEmpty()) {
			switch (caissonType) {
			case "BAS_COINS_L":
			case "HAUT_COINS_L":
			case "HAUT_COINS_ANGLE":
			case "BAS_COINS_ANGLE":
				return true;
			default:
				return false;
			}
		}

		return false;
	}
	
	private static void addEquationsDependencies(Set<String> dependencies, List<Equation> pieceEquations) {
		for(Equation eq:pieceEquations) {
			dependencies.addAll(parseDependencies(eq.getExpression()));
			for(CondtionalEquation CondtionalEquation:eq.getCondtionalequation()) {
				dependencies.addAll(parseDependencies(CondtionalEquation.getExpression()));
				dependencies.addAll(extractVariables(CondtionalEquation.getWhenexpression()));
			}
		}
	}
	
	public static double safeParseParam(PublicParamGroup group, String key, double fallback) throws NumberFormatException {
		try {
			if (group == null) {
				System.err.println("PublicParamGroup is null. Using fallback for: " + key);
				return fallback;
			}

			MechanicPublicParam param = group.getParam(key);
			if (param == null || param.getDefaultvalue() == null) {
				System.err.println("Param '" + key + "' is missing or null. Using fallback: " + fallback);
				return fallback;
			}

			return Double.parseDouble(param.getDefaultvalue());
		} catch (NumberFormatException ex) {
			System.err.println("Invalid number format for param '" + key + "': " + ex.getMessage()
					+ ". Using fallback: " + fallback);
			return fallback;
		}
	}
	public static String resolveParam(DesignObject3D design,String key) {
		String value = null;
		try {
			MechanicPublicParam publicParam = design.getMechanicDesignDefinition().getPublicParam(key);
			if (publicParam != null) {
				value = publicParam.getDefaultvalue();
			} else {
				MechanicPrivateParam privateParam = design.getMechanicDesignDefinition().getPrivateParam(key);
				if (privateParam != null)
					value = privateParam.getValue();
			}

		} catch (Exception e) {
			e.printStackTrace();
			value = null;
		}
		return value;

	}
	// getters for public Params of the designDifition for designPurposes
	public static MechanicPublicParam getElementPositionType(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("POSITION_TYPE");
	}

	public static CaissonType getPositionType(DesignObject3D design) {
		try {
			MechanicPublicParam param = getElementPositionType(design);
			if (param == null) {
				return CaissonType.MONO_CAISSON;
			}
			String type = param.getTypedefelement().getKey();
			switch (type) {
			case "DROITE":
				return CaissonType.EXTREME_DROITE;

			case "GAUCHE":
				return CaissonType.EXTREME_GAUCHE;

			case "MILIEU":
				return CaissonType.MILIEU;

			case "MONO":
				return CaissonType.MONO_CAISSON;

			default:
				return CaissonType.MONO_CAISSON;
			}
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}
	

	public static MechanicPublicParam getElementPorteDirection(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("PORTE_DIRECTION");
	}

	public static PorteDirection getPorteDirection(DesignObject3D design) {
		MechanicPublicParam param;

		try {
			param = getElementPorteDirection(design);
			if (param == null) {
				return PorteDirection.LEFT;
			}
			String type = param.getTypedefelement().getKey();

			switch (type) {
			case "LEFT":
				return PorteDirection.LEFT;

			case "RIGHT":
				return PorteDirection.RIGHT;

			default:
				return PorteDirection.LEFT;

			}
		} catch (Exception e) {

			e.printStackTrace();
			return null;
		}
	}
	

	public static MechanicPublicParam getElementPoigneeType(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("POIGNEE_TYPE");
	}

	public static PoigneeType getPoigneeType(DesignObject3D design) {

		MechanicPublicParam param;
		try {
			param = getElementPoigneeType(design);
			if (param == null) {
				return PoigneeType.AVEC_POIGNEE;
			}
			String type = param.getTypedefelement().getKey();

			switch (type) {
			case "AVEC_POIGNEE":
				return PoigneeType.AVEC_POIGNEE;

			case "SANS_POIGNEE":
				return PoigneeType.SANS_POIGNEE;

			default:
				return PoigneeType.AVEC_POIGNEE;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return null;
		}
	}

	public static MechanicPublicParam getElementPorteType(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("PORT_TYPE");
	}

	public static PortType getPorteType(DesignObject3D design) {
		try {
			MechanicPublicParam param = getElementPorteType(design);
			if (param == null) {
				return PortType.FRANCAISE;
			}
			String type = param.getTypedefelement().getKey();

			switch (type) {
			case "FRANCAISE":
				return PortType.FRANCAISE;

			case "FRANCAISE2V":
				return PortType.FRANCAISE2V;

			case "BASCULANT":
				return PortType.BASCULANT;

			case "BASCULANT2H":
				return PortType.BASCULANT2H;

			case "COLISSANT":
				return PortType.COLISSANT;

			case "FRANCAISE2H":
				return PortType.FRANCAISE2H;
			case "BASCULANT2H2P":
				return PortType.BASCULANT2H2P;
			default:
				return PortType.FRANCAISE;
			}
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	public static MechanicPublicParam getElementPoigneePosition(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("POIGNEE_POSITION");
	}

	public static PoigneePosition getPoigneePosition(DesignObject3D design) {
		try {
			MechanicPublicParam param = getElementPoigneePosition(design);
			if (param == null) {
				return PoigneePosition.TOP_LEFT;
			}
			String type = param.getTypedefelement().getKey();

			switch (type) {
			case "BOTTOM_LEFT":
				return PoigneePosition.BOTTOM_LEFT;

			case "BOTTOM_MIDDLE":
				return PoigneePosition.BOTTOM_MIDDLE;

			case "BOTTOM_RIGHT":
				return PoigneePosition.BOTTOM_RIGHT;

			case "LEFT_MIDDLE":
				return PoigneePosition.LEFT_MIDDLE;

			case "RIGHT_MIDDLE":
				return PoigneePosition.RIGHT_MIDDLE;

			case "TOP_LEFT":
				return PoigneePosition.TOP_LEFT;
			case "TOP_MIDDLE":
				return PoigneePosition.TOP_MIDDLE;

			case "TOP_RIGHT":
				return PoigneePosition.TOP_RIGHT;

			case "CENTER":
				return PoigneePosition.CENTER;
			default:
				return PoigneePosition.TOP_LEFT;
			}
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	public static boolean setPoigneePosition(DesignObject3D design,PoigneePosition position) {
		try {
			String pos = "TOP_RIGHT";
			switch (position) {
			case BOTTOM_LEFT:
				pos = "BOTTOM_LEFT";
				break;
			case BOTTOM_MIDDLE:
				pos = "BOTTOM_MIDDLE";
				break;
			case BOTTOM_RIGHT:
				pos = "BOTTOM_RIGHT";
				break;
			case LEFT_MIDDLE:
				pos = "LEFT_MIDDLE";
				break;
			case RIGHT_MIDDLE:
				pos = "RIGHT_MIDDLE";
				break;
			case TOP_LEFT:
				pos = "TOP_LEFT";
				break;
			case TOP_MIDDLE:
				pos = "TOP_MIDDLE";
				break;
			case TOP_RIGHT:
				pos = "TOP_RIGHT";
				break;
			case CENTER:
				pos = "CENTER";
				break;
			}

			design.getMechanicDesignDefinition().getPublicparamgroup().setParamType("POIGNEE_POSITION", pos);

		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	public static MechanicPublicParam getElementPoigneeOrientation(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("POIGNEE_ORIENTATION");
	}

	public static Orientation getPoigneeOrientation(DesignObject3D design) {

		MechanicPublicParam param;

		try {
			param = getElementPoigneeOrientation(design);
			if (param == null) {
				return Orientation.HORIZONTAL;
			}
			String type = param.getTypedefelement().getKey();

			switch (type) {
			case "HORIZONTAL":
				return Orientation.HORIZONTAL;

			case "VERTICAL":
				return Orientation.VERTICAL;

			default:
				return Orientation.HORIZONTAL;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return null;
		}
	}

	public static boolean setPoigneeOrientation(DesignObject3D design,Orientation position) {
		try {
			String pos = "HORIZONTAL";
			switch (position) {
			case HORIZONTAL:
				pos = "HORIZONTAL";
				break;
			case VERTICAL:
				pos = "VERTICAL";
				break;
			default:
				pos = "HORIZONTAL";
				break;

			}

			design.getMechanicDesignDefinition().getPublicparamgroup().setParamType("POIGNEE_ORIENTATION", pos);

		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	public static double getDesignLongeur(DesignObject3D design) throws Exception {
		MechanicPublicParam param = design.getMechanicDesignDefinition().getPublicParam("global.l");
		String value = param.getDefaultvalue();
		double longeur = Double.valueOf(value);

		return longeur;

	}

	public static double getDesignHauteur(DesignObject3D design) throws Exception {
		MechanicPublicParam param = design.getMechanicDesignDefinition().getPublicParam("global.h");
		String value = param.getDefaultvalue();
		double longeur = Double.valueOf(value);

		return longeur;

	}

	public static double getDesignMaxPorteLongeur(DesignObject3D design) {
		MechanicPublicParam param;
		double longeur = 600.0;
		try {
			param = design.getMechanicDesignDefinition().getPublicParam("PORTE_MAXWIDTH");
			if(param!=null)
			{
				String value = param.getDefaultvalue();
				longeur = Double.valueOf(value);

			}

		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return longeur;

	}

	public static double getDesignProfondeur(DesignObject3D design) throws Exception {
		MechanicPublicParam param = design.getMechanicDesignDefinition().getPublicParam("global.p");
		String value = param.getDefaultvalue();
		double longeur = Double.valueOf(value);

		return longeur;

	}

	public static boolean getDesignContainSocle(DesignObject3D design) {
		String value = "";
		MechanicPublicParam param;
		try {
			param = design.getMechanicDesignDefinition().getPublicParam("contains_socle");
			value = param.getTypedefelement().getKey();
			return Boolean.valueOf(value);
		} catch (Exception e) {
		}
		return false;

	}

	public static float getDesignAltitude(DesignObject3D design) {
		float altitude = 0;
		String value = null;
		MechanicPublicParam param;
		try {
			param = design.getMechanicDesignDefinition().getPublicParam("element.positionY");
			value = param.getDefaultvalue();
			altitude = value != null ? Float.valueOf(value) : 0;
		} catch (Exception e) {
		}
		return altitude;
	}


	public static PositionReferenceType getDesignAltitudeReference(DesignObject3D design) {
		PositionReferenceType reference = PositionReferenceType.BAS;
		String value = "";
		MechanicPublicParam param;
		try {
			param = design.getMechanicDesignDefinition().getPublicParam("element_YReference");
			value = param.getTypedefelement().getKey();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		if (value != null && value.toUpperCase().contentEquals("TOP")) {
			reference = PositionReferenceType.HAUT;
		} else {
			reference = PositionReferenceType.BAS;
		}
		return reference;
	}

	public static String getDesignCaissonType(DesignObject3D design) {
		String value = "";
		MechanicPublicParam param;
		try {
			param = design.getMechanicDesignDefinition().getPublicParam("CAISSON_TYPE");
			if(param!=null && param.getTypedefelement()!=null)
			{
				value = param.getTypedefelement().getKey();
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return value;

	}
	public static String getMechanicDesignCaissonType(MechanicDesign design) {
		String value = "";
		MechanicPublicParam param;
		try {
			param = design.getPublicParam("CAISSON_TYPE");
			value = param.getTypedefelement().getKey();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return value;

	}
	public static int getNbTiroir(DesignObject3D design) {
		String value = null;
		MechanicPublicParam param;
		try {
			param = design.getMechanicDesignDefinition().getPublicParam("NB_TIROIR");
			value = param.getDefaultvalue();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return value != null ? Integer.valueOf(value) : -1;

	}

	//
	public static MechanicPublicParam getElementNombrEtagere(DesignObject3D design) throws Exception {
		return design.getMechanicDesignDefinition().getPublicParam("ETAGERE_COUNT");
	}

	public static boolean isFausseFacade(DesignObject3D design) {
		return EmfUtils.getDesignCaissonType(design).contentEquals("FAUSSE_FACADE");
	}

	
	public static List<Box> getArrangmentSpaces(DesignObject3D design){
		ArrayList<Box> boxes = new ArrayList<Box>();
		List<Piece2D> fixedShelfs = design.getListPieces().stream()
				.filter(p -> p.getPiecetype().isShelf()).toList();
		if(fixedShelfs.size() >2) {
			fixedShelfs = new ArrayList<Piece2D>(fixedShelfs);
			fixedShelfs.sort((p1,p2)-> (int) (p1.getCenter().y - p2.getCenter().y));
			List<Piece2D> facades = design.getListPieces().stream().filter(p -> p.getPiecetype().isFacade()).toList();
			for(int i = 0; i < fixedShelfs.size()-1; i++) {
				Piece2D current = fixedShelfs.get(i);
				Piece2D next = fixedShelfs.get(i + 1);
				float yMin = current.getCenter().y;
				float yMax = next.getCenter().y;
				boolean isCached = false;
				for(var facade: facades) {
					var center = facade.getCenter().y;
					var facadeYmin = facade.getPosition().y;
					var facadeYmax = facade.getPosition().y+facade.getSize().y;
					Intervale facadeInterval=new Intervale(facadeYmin, true, facadeYmax, true);
					Intervale shelfInterval=new Intervale(yMin, true, yMax, true);
					Intervale intersection =facadeInterval.getintersection(shelfInterval);
					if(intersection!=null && intersection.getlong()>current.getSize().y*2) {
						isCached = true;
						break;
					}
				}
				if(!isCached) {
					Vector3 postion = current.getCenter().cpy().add(next.getCenter()).scl(0.5f);
					Vector3 size = current.getSize().add(30,0,0);
					size.y = yMax - yMin;
					size.z=(float) design.getProfondeurext();
					postion.z=(float) (design.getProfondeurext()/2);
					boxes.add(new Box(postion, size));
				}
			}
		}
		return boxes;
	}

	//
	public static Box getOvenSpace(DesignObject3D design) {
		try {
			String caissonType = getDesignCaissonType(design);
			if (caissonType != null && !caissonType.isEmpty()) {
				if ((caissonType.contentEquals("COLONNE_FOUR")
						|| caissonType.contentEquals("COLONNE_FOUR_MICROONDE"))) {
					List<Box> boxes = EmfUtils.getArrangmentSpaces(design);
					if(!boxes.isEmpty()) {
						boxes.sort((b1, b2) -> Math.round((b2.getDimensions().y - b1.getDimensions().y)));
						return boxes.get(0);
					}
				} else if (caissonType.contentEquals("BAS_FOUR")) {
					return new Box(design.getPosition().add(design.getSize().scl(0.5f)), design.getSize());
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}

		return null;
	}

	public static Box getMicrowaveSpace(DesignObject3D design) {
		try {
			String caissonType = getDesignCaissonType(design);
			if (caissonType != null && !caissonType.isEmpty() && (caissonType.contentEquals("COLONNE_FOUR_MICROONDE")
					|| caissonType.contentEquals("COLONNE_MICROONDE"))) {
				List<Box> boxes = getArrangmentSpaces(design);
				if(!boxes.isEmpty()) {
					boxes.sort((b1, b2) -> Math.round((b2.getDimensions().y - b1.getDimensions().y)));
					int index = Math.min(1, boxes.size()-1);
					return boxes.get(index);//second largest
				}
			}
		} catch (Exception e) {

			e.printStackTrace();
			return null;
		}

		return null;
	}

	public Vector3 getFrigoPosition(DesignObject3D design) {
		try {
			String caissonType = getDesignCaissonType(design);
			if (caissonType != null && !caissonType.isEmpty() && caissonType.contentEquals("COLONNE_FRIGO")) {
				MechanicPrivateParam paramsep1Position = design.getMechanicDesignDefinition()
						.getPrivateParam("separator.position");
				String valueSep1Position = paramsep1Position.getValue();
				float sep1Position = Float.valueOf(valueSep1Position);
				float x = (float) (design.getLongeurext() / 2000f);
				float y = (sep1Position + 0.009f) / 2000f;
				float z = (float) design.getProfondeurext() / 2000f + 0.026f;
				Vector3 pos = new Vector3(x, y, z);
				return pos;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return null;
	}

	public Vector3 getFrigoDimention(DesignObject3D design) {
		try {
			String caissonType = getDesignCaissonType(design);
			if (caissonType != null && !caissonType.isEmpty() && caissonType.contentEquals("COLONNE_FRIGO")) {
				MechanicPrivateParam paramsep1Position = design.getMechanicDesignDefinition()
						.getPrivateParam("separator.position");
				String valueSep1Position = paramsep1Position.getValue();
				float sep1Position = Float.valueOf(valueSep1Position);
				float x = (float) (design.getLongeurext() / 1000f) + 0.036f;
				float y = sep1Position / 1000f - 0.027f;
				float z = (float) design.getProfondeurext() / 1000f + 0.052f;
				Vector3 pos = new Vector3(x, y, z);
				return pos;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return null;
	}

}
