package gdxapp.object3d;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.GL30;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.VertexAttribute;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.VertexAttributes.Usage;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder.VertexInfo;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.g3d.utils.BaseAnimationController.Transform;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.mathutils.EarClipper;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Surface;
import dressing.mathutils.Triangle;
import dressing.model.Piece2D;
import dressing.model.ProjectManager;
import dressing.model.types.PieceType;
import gdxapp.assets.AssetsTextures;
import gdxapp.fabs3d.FacadeCreator;
import gdxapp.fabs3d.Planche;
import gdxapp.fabs3d.WoodCovering;
import gdxapp.shaders.PbrMaterial;
import geometry.CompoundObject;
import geometry.Polygon;

public class PolygonBuilder {

	private static ModelBuilder modelBuilder;

	private static PolygonBuilder builder = new PolygonBuilder();

	private PolygonBuilder() {}

	public static Model create3DPolygonModel(ArrayList<Vector2> outerPoly, Matrix4 transform, float[] depthLevels,
			Material material) {
		MathUtilities.setWinding(outerPoly, 1);
		if (modelBuilder == null)
			modelBuilder = new ModelBuilder();
		modelBuilder.begin();
		PolygonBuilder.createPolygoneNode(modelBuilder, "poly", outerPoly, new ArrayList<Vector2>(), material,
				transform, depthLevels);
		return modelBuilder.end();
	}

	public static Model create3DPolygonModel(ArrayList<Vector2> outerPoly, ArrayList<Vector2> hole, Matrix4 transform,
			float[] depthLevels, Material material) {
		if (modelBuilder == null)
			modelBuilder = new ModelBuilder();
		modelBuilder.begin();
		PolygonBuilder.createPolygoneNode(modelBuilder, "poly", outerPoly, hole, material, transform, depthLevels);
		return modelBuilder.end();
	}
	
	public static Model createModel(CompoundObject compoundObject, float ymin, float ymax) {
		if (modelBuilder == null)
			modelBuilder = new ModelBuilder();
		modelBuilder.begin();
		try {
			PbrMaterial exteriorMaterial = (PbrMaterial) compoundObject.getBorder().getMaterial();
			if(!exteriorMaterial.isReady()) {
				exteriorMaterial.load();
				exteriorMaterial.prepare();
			}
			PolygonBuilder.createPolygoneNode(modelBuilder, "exterior", compoundObject.getBorder().getVertices(), compoundObject.getExtrusionVertices(),
					exteriorMaterial, compoundObject.getWorldTransform(), new float[] {ymax, ymin});
			if(compoundObject.hasHole() && !compoundObject.isPerfored()) {
				ArrayList<Vector2> glassPartVertices = new ArrayList<Vector2>();
	    		int size = compoundObject.getExtrusionVertices().size();
	    		for(int i = 0; i < size; i++ ) {
	    			glassPartVertices.add(compoundObject.getExtrusionVertices().get(size - 1 - i).cpy());
	    		}
	    		PbrMaterial glass = (PbrMaterial) compoundObject.getExtrusion().getMaterial();
	    		PolygonBuilder.createPolygoneNode(modelBuilder, "marble", glassPartVertices, new ArrayList<Vector2>(),
	    				glass, compoundObject.getWorldTransform(), new float[] {ymax , ymin});
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		return modelBuilder.end();
	}
    		
	public static void createPiece3D(ModelBuilder modelBuilder, Planche planche ) {
		if(planche.getPiece2D().getPiecetype().isFacade()) {
			FacadeCreator.createFacade(planche, modelBuilder);
		}
		Material frameMaterial = null;
		PbrMaterial plancheVitreMaterial=planche.getVitreMaterial();
		PbrMaterial plancheMaterial=planche.getMaterial();
		if(plancheVitreMaterial!= null)
		{
			plancheVitreMaterial.prepare();
			plancheMaterial.toGdxMaterial();
		}
		if(plancheMaterial != null)
		{
			if(!plancheMaterial.isReady())
				plancheMaterial.prepare();
			plancheMaterial.toGdxMaterial();
		}
		frameMaterial = plancheMaterial;
		frameMaterial.id = planche.getPiece2D().getName() + "frame";
		
		PolygonBuilder.createPolygoneNode(modelBuilder, "planche" , planche.getLocalVertices(), planche.getExtrusionVertices(), frameMaterial, planche.getTransform().cpy().inv(), new float[] {planche.getThickness()/2,
				-planche.getThickness()/2});
		boolean isfacade=planche.getPiece2D().getPiecetype().isFacade();
		boolean isaddVitre=isfacade;
		List<Piece2D> pieces=planche.getPiece2D().getListPieces();
		if(pieces.size()>0&& pieces.stream().filter(p->p.getPiecetype().equals(PieceType.VITRE)).count()>0) {
			isaddVitre=false;
		}
		if(planche.getExtrusionVertices().size() > 0 && isaddVitre) {
			Material glassMaterial = plancheVitreMaterial;
			glassMaterial.id = planche.getPiece2D().getName() + "extrusion";

			ArrayList<Vector2> glassPartVertices = new ArrayList<Vector2>();
			int size = planche.getExtrusionVertices().size();
			for(int i = 0; i < size; i++ ) {
				glassPartVertices.add(planche.getExtrusionVertices().get(size - 1 - i).cpy());
			}
			PolygonBuilder.createPolygoneNode(modelBuilder, "planche" , glassPartVertices, new ArrayList<Vector2>(), glassMaterial, planche.getTransform().cpy().inv(), new float[] {planche.getThickness()/2 - 0.006f,
					-planche.getThickness()/2f + 0.006f } );
		}
	}

	public static void createPolygoneNode(ModelBuilder modelBuilder, String partID, ArrayList<Vector2> vertices,
			ArrayList<Vector2> holes, Material material, Matrix4 worldTransform, float[] depthLevels) {
		ArrayList<Triangle> faceTriangles = EarClipper.triangulate(vertices, holes);
		ArrayList<Vector3> faceVertices = new ArrayList<>();
		ArrayList<Vector3> backVertices = new ArrayList<>();
		for (Vector2 vertex : vertices) {
			faceVertices.add(new Vector3(vertex, depthLevels[0]).mul(worldTransform));
			backVertices.add(new Vector3(vertex, depthLevels[1]).mul(worldTransform));
		}
		Matrix4 faceTransform = calculateTSMatrix(faceVertices);
		Matrix4 backTransform = calculateTSMatrix(backVertices);
		ArrayList<Vector3> faceVerticesHole = new ArrayList<>();
		ArrayList<Vector3> backVerticesHole = new ArrayList<>();
		for (Vector2 vertex : holes) {
			faceVerticesHole.add(new Vector3(vertex,  depthLevels[0]).mul(worldTransform));
			backVerticesHole.add(new Vector3(vertex, depthLevels[1]).mul(worldTransform));
		}
		ArrayList<Triangle> sideTriangles = calculateSideTriangles(faceVertices, backVertices);
		ArrayList<Triangle> holesSides = calculateSideTriangles(faceVerticesHole, backVerticesHole);

		int attr = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal
				| VertexAttributes.Usage.TextureCoordinates;
		MeshPartBuilder meshPartBuilder = modelBuilder.part(partID, GL20.GL_TRIANGLES, attr, material);
		
		for (Triangle triangle : faceTriangles) {
			VertexInfo[] infosF = getVertices(triangle, faceTransform, worldTransform, depthLevels[0], null);
			meshPartBuilder.triangle(infosF[0], infosF[1], infosF[2]);
			Vector3 backNormal = infosF[0].normal.cpy().scl(-1);
			VertexInfo[] infosB = getVertices(triangle, backTransform, worldTransform, depthLevels[1], backNormal);
			meshPartBuilder.triangle(infosB[0], infosB[2], infosB[1]);
		}
		float uvOffset = 0;
		float height = Math.abs(depthLevels[1] - depthLevels[0]);
		for (int i = 0; i < sideTriangles.size(); i += 2) {
			VertexInfo[] infos = getVertices(sideTriangles, i, uvOffset,height);
			meshPartBuilder.triangle(infos[0], infos[1], infos[2]);			
			meshPartBuilder.triangle(infos[2], infos[3], infos[0]);
			if ((i/2)  + 1 < faceVertices.size()) {
				ArrayList<Vector2> uvs = new ArrayList<>();
				for(int c = 0; c < 4; c++) {
					uvs.add(infos[c].uv);
				}
				Vector2[] bounds = MathUtilities.getBoundariesPlane(uvs);
				uvOffset += bounds[1].sub(bounds[0]).x;
			}
		}
		uvOffset = 0;
		for (int i = 0; i < holesSides.size(); i += 2) {
			VertexInfo[] infos = getVertices(holesSides, i, uvOffset,height);
			meshPartBuilder.triangle(infos[0], infos[2], infos[1]);
			meshPartBuilder.triangle(infos[0], infos[3], infos[2]);
			
			meshPartBuilder.triangle(infos[0], infos[1], infos[2]);
			meshPartBuilder.triangle(infos[0], infos[2], infos[3]);
			if ((i / 2) + 1 < faceVerticesHole.size())
				uvOffset += 0;//faceVerticesHole.get((i / 2) + 1).dst(faceVerticesHole.get(i / 2));
		}
	}
	
	public static void createPieceSketch(ModelBuilder modelBuilder, Planche planche ) {
		Material material = new Material(ColorAttribute.createDiffuse(Color.BLACK));
		
		ArrayList<Vector3> faceVertices = new ArrayList<Vector3>(), backVertices = new ArrayList<Vector3>();
		ArrayList<Vector2> localVertices = planche.getLocalVertices();
		Matrix4 transform = planche.getTransform().cpy().inv();
		for(int i =0; i < localVertices.size(); i++) {
			faceVertices.add(new Vector3(localVertices.get(i), planche.getThickness()/2.0f).mul(transform));
			backVertices.add(new Vector3(localVertices.get(i), -planche.getThickness()/2.0f).mul(transform));
		}
		createLineNode(modelBuilder, "croquis", faceVertices, backVertices, material);
		
		
		if(planche.getExtrusionVertices().size() > 0) {
			ArrayList<Vector3> faceEVertices = new ArrayList<Vector3>(), backEVertices = new ArrayList<Vector3>();
			ArrayList<Vector2> extrusionLVertices = planche.getExtrusionVertices();
			for(int i =0; i < extrusionLVertices.size(); i++) {
				faceEVertices.add(new Vector3(localVertices.get(i), planche.getThickness()/2.0f).mul(transform));
				backEVertices.add(new Vector3(localVertices.get(i), -planche.getThickness()/2.0f).mul(transform));
			}
			createLineNode(modelBuilder, "croquis extrusion", faceEVertices, backEVertices, material);
		}
	}
	
	public static void createLineNode(ModelBuilder modelBuilder, String partID, ArrayList<Vector3> faceVertices,
			 ArrayList<Vector3> backVertices, Material material) {

		int attr = VertexAttributes.Usage.Position;
		MeshPartBuilder meshPartBuilder = modelBuilder.part(partID, GL20.GL_LINES, attr, material);
		for(int i = 0; i < faceVertices.size(); i++) {
			meshPartBuilder.line(faceVertices.get(i), faceVertices.get((i+1)%faceVertices.size()));
			meshPartBuilder.line(backVertices.get(i), backVertices.get((i+1)%backVertices.size()));
			meshPartBuilder.line(faceVertices.get(i), backVertices.get(i));
		}
	}

	public static void createFaceAndBackNode(ModelBuilder modelBuilder, String partID, ArrayList<Vector2> vertices,
			ArrayList<Vector2> holes, Material material, Matrix4 worldTransform, float[] depthLevels) {
		ArrayList<Triangle> faceTriangles = EarClipper.triangulate(vertices, holes);
		ArrayList<Vector3> faceVertices = new ArrayList<>();
		ArrayList<Vector3> backVertices = new ArrayList<>();
		for (Vector2 vertex : vertices) {
			faceVertices.add(new Vector3(vertex, depthLevels[0]).mul(worldTransform));
			backVertices.add(new Vector3(vertex, depthLevels[1]).mul(worldTransform));
		}
		ArrayList<Vector3> faceVerticesHole = new ArrayList<>();
		ArrayList<Vector3> backVerticesHole = new ArrayList<>();
		for (Vector2 vertex : holes) {
			faceVerticesHole.add(new Vector3(vertex, depthLevels[0]).mul(worldTransform));
			backVerticesHole.add(new Vector3(vertex, depthLevels[1]).mul(worldTransform));
		}
		
		Matrix4 faceTransform = calculateTSMatrix(faceVertices);
		Matrix4 backTransform = calculateTSMatrix(backVertices);

		int attr = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal
				| VertexAttributes.Usage.TextureCoordinates;
		MeshPartBuilder meshPartBuilder = modelBuilder.part(partID, GL20.GL_TRIANGLES, attr, material);
		for (Triangle triangle : faceTriangles) {
			Vector3 normal = new Vector3(0,0,1).mul(new Matrix3().set(worldTransform));
			VertexInfo[] infosF = getVertices(triangle, faceTransform, worldTransform, depthLevels[0], normal);
			meshPartBuilder.triangle(infosF[0], infosF[1], infosF[2]);
			VertexInfo[] infosB = getVertices(triangle, backTransform, worldTransform, depthLevels[1], normal.scl(-1));
			meshPartBuilder.triangle(infosB[0], infosB[2], infosB[1]);
		}
	}

	private static VertexInfo[] getVertices(ArrayList<Triangle> sideTriangles, int index, float offset, float height) {
		
		Vector2 uv = new Vector2();
		Triangle first = sideTriangles.get(index);
		Triangle second = sideTriangles.get(index + 1);
		ArrayList<Vector3> surface = new ArrayList<Vector3>();
		surface.addAll(Arrays.asList(first.getWorldCoords()));
		surface.addAll(Arrays.asList(second.getWorldCoords()));
		Matrix4 TBN = calculateTSMatrix(surface);
		ArrayList<Vector3> tangentVertices = new ArrayList<Vector3>();
		for(int i = 0; i < surface.size(); i++) {
			tangentVertices.add(surface.get(i).cpy().mul(TBN));
		}
		Vector3[] bounds = MathUtilities.getBoundaries(tangentVertices);
		Vector3 range = bounds[1].sub(bounds[0]);
		Vector3 normal = first.getWorldCoords()[1].cpy().sub(first.getWorldCoords()[0])
				.crs(first.getWorldCoords()[2].cpy().sub(first.getWorldCoords()[0])).nor();
		VertexInfo info0 = new VertexInfo();
		info0.setPos(first.getWorldCoords()[0]);
		info0.setNor(normal);
		uv.set(0, 0).add(offset,0);
		info0.setUV(uv.cpy());
		VertexInfo info1 = new VertexInfo();
		info1.setPos(first.getWorldCoords()[1]);
		info1.setNor(normal);
		uv.set(0, height).add(offset,0);
		info1.setUV(uv.cpy());
		VertexInfo info2 = new VertexInfo();
		info2.setPos(first.getWorldCoords()[2]);
		info2.setNor(normal);
		uv.set(range.x + range.y - height, height).add(offset,0);
		info2.setUV(uv.cpy());
		VertexInfo info3 = new VertexInfo();
		info3.setPos(second.getWorldCoords()[1]);
		info3.setNor(normal);
		uv.set(range.x + range.y - height,0).add(offset,0);
		info3.setUV(uv.cpy());
		VertexInfo[] infos = { info0, info1, info2, info3 };
		return infos;
	}

	private static ArrayList<Triangle> calculateSideTriangles(ArrayList<Vector3> faceVertices,
			ArrayList<Vector3> backVertices) {
		ArrayList<Triangle> triangles = new ArrayList<>();
		int nextIndex;
		Vector3 v0, v1, v2, v3;
		for (int i = 0; i < faceVertices.size(); i++) {
			nextIndex = (i + 1) % faceVertices.size();
			v0 = faceVertices.get(i);
			v1 = backVertices.get(i);
			v2 = backVertices.get(nextIndex);
			v3 = faceVertices.get(nextIndex);
			triangles.add(new Triangle(v0, v1, v2));
			triangles.add(new Triangle(v2, v3, v0));
		}
		return triangles;
	}

	private static VertexInfo[] getVertices(Triangle triangle, Matrix4 tangentSpaceTransform, Matrix4 worldTrasform,
			float depth, Vector3 normal) {
		Vector2 uv = new Vector2();
		Vector3[] worldCoords = triangle.getSpaceCoords(worldTrasform, depth);
		if(normal == null)
			normal = worldCoords[1].cpy().sub(worldCoords[0]).crs(worldCoords[2].cpy().sub(worldCoords[0])).nor();
		VertexInfo info0 = new VertexInfo();
		info0.setPos(worldCoords[0]);
		info0.setNor(normal);
		Vector3 tangentCoords = worldCoords[0].cpy().mul(tangentSpaceTransform);
		uv = new Vector2(tangentCoords.x, tangentCoords.y);
		info0.setUV(uv.cpy());
		VertexInfo info1 = new VertexInfo();
		info1.setPos(worldCoords[1]);
		info1.setNor(normal);
		tangentCoords = worldCoords[1].cpy().mul(tangentSpaceTransform);
		uv = new Vector2(tangentCoords.x, tangentCoords.y);
		info1.setUV(uv.cpy());
		VertexInfo info2 = new VertexInfo();
		info2.setPos(worldCoords[2]);
		info2.setNor(normal);
		tangentCoords = worldCoords[2].cpy().mul(tangentSpaceTransform);
		uv = new Vector2(tangentCoords.x, tangentCoords.y);
		info2.setUV(uv.cpy());
		VertexInfo[] infos = { info0, info1, info2 };
		return infos;
	}

	public static float[] calculateTangentAndbitangent(Vector3 pos0, Vector3 pos1, Vector3 pos2, Vector2 uv0,
			Vector2 uv1, Vector2 uv2) {
		Vector3 tangent = new Vector3(), bitangent = new Vector3();
		Vector3 edge1 = pos1.cpy().sub(pos0);
		Vector3 edge2 = pos2.cpy().sub(pos0);
		Vector2 deltaUV1 = uv1.cpy().sub(uv0);
		Vector2 deltaUV2 = uv2.cpy().sub(uv0);

		float f = 1.0f / (deltaUV1.x * deltaUV2.y - deltaUV2.x * deltaUV1.y);

		tangent.x = f * (deltaUV2.y * edge1.x - deltaUV1.y * edge2.x);
		tangent.y = f * (deltaUV2.y * edge1.y - deltaUV1.y * edge2.y);
		tangent.z = f * (deltaUV2.y * edge1.z - deltaUV1.y * edge2.z);

		bitangent.x = f * (-deltaUV2.x * edge1.x + deltaUV1.x * edge2.x);
		bitangent.y = f * (-deltaUV2.x * edge1.y + deltaUV1.x * edge2.y);
		bitangent.z = f * (-deltaUV2.x * edge1.z + deltaUV1.x * edge2.z);

		return new float[] { tangent.x, tangent.y, tangent.z, bitangent.x, bitangent.y, bitangent.z };
	}

	public static ModelBuilder getModelBuilder() {
		if(modelBuilder == null)
			modelBuilder = new ModelBuilder();
		return modelBuilder;
	}

	public static void setModelBuilder(ModelBuilder modelBuilder) {
		PolygonBuilder.modelBuilder = modelBuilder;
	}
	
	
	public static Matrix4 calculateTSMatrix(ArrayList<Vector3> vertices) {
		if(vertices.size() < 3)
			return null;
		Matrix4 transform = MathUtilities.calculateTBN(vertices.get(0), vertices.get(1), vertices.get(2),
				new Vector3());
		ArrayList<Vector3> uvs = new ArrayList<Vector3>();
		Vector2 offset = new Vector2(Float.MAX_VALUE, Float.MAX_VALUE);
		Vector2 max = new Vector2(-Float.MAX_VALUE, -Float.MAX_VALUE);
		for(Vector3 vertex: vertices) {
			Vector3 tmpUVs = vertex.cpy().mul(transform);
			offset.x = Math.min(offset.x, tmpUVs.x);
			offset.y = Math.min(offset.y, tmpUVs.y);
			max.x = Math.max(max.x, tmpUVs.x);
			max.y = Math.max(max.y, tmpUVs.y);
		}
		Vector2 scale = max.cpy().sub(offset);
		Matrix4 scaling = new Matrix4().setToScaling (Math.min(8.0f, 1.0f/scale.x), Math.min(8.0f,1.0f/scale.y), 1.0f);
		Vector3 newOrigin = new Vector3(offset.scl(-1), 0);
		transform.mulLeft(new Matrix4().setToTranslation(newOrigin));
		return transform;// .mulLeft(scaling);
	}
	

	public static Model createWoodCovering(WoodCovering woodCovering) {
		return null;
	}
}
