package dressing.ui.palette;

import java.io.File;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.eclipse.emf.ecore.util.EcoreUtil;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Pixmap;
import com.badlogic.gdx.math.Vector3;

import dressing.config.WorkspaceConfiguration;
import dressing.config.persistence.ResourceManagers;
import dressing.mathutils.Vector4;
import dressing.model.DesignObject3D;
import dressing.model.MechanicDesignCreator;
import dressing.model.MechanicDesignMaterialsHelper;
import dressing.model.ProjectManager;
import dressing.ui.util.ImageLoaderCache;
import gdxapp.ModelViewer;
import gdxapp.Commun.ScreenshotFactory;
import gdxapp.object3d.KitchenElement;
import gdxapp.object3d.WorldObjectFactory;
import gdxapp.shaders.PbrMaterial;
import param.DesignInstance;
import param.MechanicDesign;
import param.MechanicPublicParam;
import param.Option;
import param.PieceType;
import param.util.ParamResourceImpl;

public class DesignPreviewGenerator {
	//  private static final Vector4 previewBackgroud = new Vector4(0.95f, 0.95f, 0.95f, 1);
	private static final Vector4 previewBackgroud = new Vector4(1.0f, 1.0f, 1.0f, 0.0f);
	private static DesignPreviewGenerator instance;
	private ConcurrentLinkedQueue<Object> updatePool=new ConcurrentLinkedQueue<Object>();
	private ExecutorService previewUpdateExecuter;
	private Timer time = new Timer();
	 /**
     * Thread-safe singleton accessor (double-checked locking).
     */
    public static DesignPreviewGenerator getInstance() {
    	synchronized (DesignPreviewGenerator.class) {
			if (instance == null) {
				instance = new DesignPreviewGenerator();
			}
			return instance;
		}
    }
    
    public DesignPreviewGenerator() {
    	createExecutor();
    	
	}
    
	private void createExecutor() {
		if (previewUpdateExecuter != null && !previewUpdateExecuter.isShutdown()) {
	        return;
	    }
		this.previewUpdateExecuter = Executors.newSingleThreadExecutor( r -> {
			Thread t = new Thread(r);
			t.setName("previewCreator-" + t.getId());
			t.setDaemon(true); // optional
			t.setPriority(Thread.NORM_PRIORITY + 1);
			return t;
		});
	}
	public void addItemToUpdate(Object item) {
		 updatePool.add(item);
	}
	
	public void start() {
		 // Start consumer loop
		if(time!=null)
		{
			time.cancel();
			time.purge();
		}
		time = new Timer(); // Instantiate Timer Object
		time.schedule(new TimerTask() {
			
			@Override
			public void run() {
				previewUpdateExecuter.submit(()->DesignPreviewGenerator.getInstance().consumeUpdateQueue());
			}
		}, 60000, 300000);
        
	}
	/**
	 * Continuously consumes items from the queue.
	 * Runs in the background executor.
	 */
	private void consumeUpdateQueue() {
	    while (!updatePool.isEmpty()) {
	        Object designUpdate =  updatePool.poll();
	        if (designUpdate != null) {
	        	try {
					if(designUpdate instanceof MechanicDesign)
					{
						createDesignPreview((MechanicDesign) designUpdate);
					}
					else if(designUpdate instanceof DesignInstance) {
						createDesignPreview((DesignInstance)designUpdate);
					}else if(designUpdate instanceof Option) {
						createDesignPreview((Option)designUpdate);
					}
				} catch (Exception e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
	        }

	        try {
	        	System.gc();
	            // Avoid CPU spinning
	            Thread.sleep(20);
	        } catch (InterruptedException e) {
	            Thread.currentThread().interrupt();
	            return;
	        }
	    }
	    ResourceManagers.getIntance().saveLibraryResource();
	}
	/**
	 * Creates the preview (heavy operation) entirely in executor thread.
	 * Posts only the final GDX update to UI thread.
	 */
	private void createDesignPreview(MechanicDesign design) {
		if(design ==null ||( !design.getType().equals(PieceType.CAISSON) && !design.getType().equals(PieceType.SPACE3D))) {
			return;
		}
		MechanicDesign previewCopy=EcoreUtil.copy(design);
		ParamResourceImpl res=(ParamResourceImpl) design.eResource();
		String id = res.getEObjectToIDMap().get(design);
		if(id==null) {
			id=String.valueOf(design.getModelId());
		}
		String projectDirPath = WorkspaceConfiguration.getDefaultConfDir()
	            + File.separator + "elements_Icons";
	    String previewPath = projectDirPath + File.separator
	            + design.getName().strip().replaceAll(" ", "_") +"_"
	            + id + "_preview.png";
	    if(!ImageLoaderCache.isValidImagePath(previewPath))
	    {
	    	createPreviewImage(previewCopy, previewPath);
	    }
        design.setPreviewImagePath(previewPath);
	}
	/**
	 * Creates the preview (heavy operation) entirely in executor thread.
	 * Posts only the final GDX update to UI thread.
	 */
	private void createDesignPreview(DesignInstance instance) {
		MechanicDesign design=instance.getDesign();
		if(design ==null || !design.getType().equals(PieceType.CAISSON)) {
			return;
		}
		ParamResourceImpl res=(ParamResourceImpl) instance.eResource();
		String id = res.getEObjectToIDMap().get(instance);
		MechanicDesign previewCopy = EcoreUtil.copy(design);
		if (instance.getOptions() != null) {
			try {
				previewCopy.injectOption(instance.getOptions().getSelectedOption());
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		
		String projectDirPath = WorkspaceConfiguration.getDefaultConfDir()
	            + File.separator + "elements_Icons";
	    String previewPath = projectDirPath + File.separator
	            + instance.getName().strip().replaceAll(" ", "_") +"_"
	            + id + "_preview.png";
	    
	    createPreviewImage(previewCopy, previewPath);
	    instance.setPerviewPath(previewPath);
	}
	/**
	 * Creates the preview (heavy operation) entirely in executor thread.
	 * Posts only the final GDX update to UI thread.
	 */
	private void createDesignPreview(Option instance) {
		MechanicDesign design=instance.getDesign();
		ParamResourceImpl res=(ParamResourceImpl) instance.eResource();
		String id = res.getEObjectToIDMap().get(instance);
		if(id==null) {
			id=instance.getModelId().toString();
		}
		MechanicDesign previewCopy = EcoreUtil.copy(design);
		try {
			previewCopy.injectOption(instance);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		
		String projectDirPath = WorkspaceConfiguration.getDefaultConfDir()
	            + File.separator + "elements_Icons";
	    String previewPath = projectDirPath + File.separator
	            + instance.getName().strip().replaceAll(" ", "_") +"_"
	            + id + "_preview.png";
	    
	    createPreviewImage(previewCopy, previewPath);
	    instance.setPreviewImagePath(previewPath);
	}
	/**
	 * @param previewCopy
	 * @param previewPath
	 */
	public void createPreviewImage(MechanicDesign previewCopy, String previewPath) {
		DesignObject3D designObject = null;
	    KitchenElement element = null;
	    
	    try {
	    	if(ProjectManager.getInstance().getCurrentKitchen()!=null) {
	    		MechanicDesignMaterialsHelper.updateMaterialFromKitchen(ProjectManager.getInstance().getCurrentKitchen(), previewCopy);
	    	}
	        designObject = MechanicDesignCreator.getInstance()
	                .constructObject(previewCopy);
			MechanicPublicParam altitudeReference = previewCopy.getPublicParam("element_YReference");
			MechanicPublicParam altitude = previewCopy.getPublicParam("element.positionY");

	    
			if (altitude != null && altitude.getDefaultvalue() != null
					&& !altitude.getDefaultvalue().isEmpty()) {
				if (altitudeReference != null && altitudeReference.getTypedefelement() != null) {
					float kitchenAltitude = 2500f;
					if (ProjectManager.getManager().getCurrentScene() != null) {
						kitchenAltitude = ProjectManager.getManager().getCurrentScene().getPreferences().getWallHeight()
								* 1000;
					}
					if (altitudeReference.getTypedefelement().getKey().equals("TOP")) {
						int y = Math.round(kitchenAltitude)
								- Float.valueOf(altitude.getDefaultvalue()).intValue();
						altitude.setDefaultvalue(y + "");
					}
				}
			}
	        element = WorldObjectFactory.getFactory()
	                .createObjectFromDefinition(designObject);
			if(element.getDoorHandles() != null) {
				PbrMaterial doorHandleMtl = new PbrMaterial();
				doorHandleMtl.setAlbedo(new Vector3());
				doorHandleMtl.setMetalness(0.5f);
				doorHandleMtl.setRoughness(0.1f);
				element.getDoorHandles().values().forEach(handle -> handle.setMaterial(doorHandleMtl));
			}
	    } catch (Exception ex) {
	        ex.printStackTrace();
	        
	    }
	    if(element!=null) {
	    	// Collect data for the GDX thread
		    Vector4 dim4 = element.getRealWorldDimension();
		    Vector3 dims = new Vector3(dim4.x, dim4.y, dim4.z);

		    PreviewRequest request = new PreviewRequest(
		            element, dims, previewPath);

		    // STEP 2: Dispatch only rendering to LibGDX thread
		    Gdx.app.postRunnable(() -> renderOnGdxThread(request));
	    }
	    
	}
	private void renderOnGdxThread(PreviewRequest req) {
	    try {
	    	 
	        // NOW GL is safe
	        ModelViewer viewer = new ModelViewer(720, 1);

	        Pixmap pix = viewer.render(
	                req.element,
	                req.dimensions.cpy()
	                     .scl(1.2f, Math.min(1.2f, 2 / req.dimensions.y), 2f),
	                previewBackgroud
	        );

	        viewer.dispose();

	        File file = new File(req.outputPath);
	        file.getParentFile().mkdirs();
	        file.createNewFile();

	        ScreenshotFactory.savePixmapAsPng(pix, req.outputPath);
	        
	    } catch (Exception ex) {
	        ex.printStackTrace();
	    }
	}
	
	public void dispose() {
		if (previewUpdateExecuter != null && !previewUpdateExecuter.isShutdown()) {
			previewUpdateExecuter.shutdownNow();
	    }
		if(time!=null)
		{
			time.cancel();
			time.purge();
		}
	}
	
	private static class PreviewRequest {
	    final KitchenElement element;
	    final Vector3 dimensions;
	    final String outputPath;

	    PreviewRequest(KitchenElement element, Vector3 dims,
	                   String outputPath) {
	        this.element = element;
	        this.dimensions = dims;
	        this.outputPath = outputPath;
	    }
	}
}
