package dressing.model;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.frs.debitage.engine.core.evalutor.GeometricEngineException;

import dressing.cam.model.Direction;
import dressing.model.types.IntersectionType;
import param.Cavity;
import param.MechanicDesign;
import param.MechanicPrivateParam;
import param.MechanicPublicParam;
import param.PieceType;
import param.Rainure;
import param.UsinageNode;
import param.UsinageTrou;

public class MechanicDesignUsinageManager {
	
	
	public static void createUsinage(DesignObject3D target, boolean cavityOnly) throws Exception {
		target.deleteUsinage();
		List<Usinage> usins= new ArrayList<Usinage>();
		target.createUsinage(usins,cavityOnly);
		MechanicDesignUsinageManager.fixUsinage(target.getRoot(), usins);
	}
	
	public static void fixUsinage(DesignObject3D rootspace, List<Usinage> usins) throws Exception {
		if (usins != null && usins.size() > 0) {
			for (Usinage usin : usins) {
				if (usin instanceof Trou) {
					MechanicDesignUsinageManager.fixTrou(rootspace, (Trou) usin);
				} else if (usin instanceof dressing.model.Rainure) {
					MechanicDesignUsinageManager.fixRainure(rootspace, (dressing.model.Rainure) usin);
				}
			}
		}
	}

	public void fixTrous(DesignObject3D rootspace, List<Trou> trous) throws Exception {
		if (trous != null && trous.size() > 0) {
			for (Trou tr : trous) {
				fixTrou(rootspace, tr);
			}
		}
	}

	public static void fixTrou(DesignObject3D rootspace, Trou tr) throws Exception {
		if (tr != null && rootspace != null) {
			Cylindre cyl = new Cylindre(tr);
			Piece2D pp = (Piece2D) tr.getParentdesign();
			Plan3D plan = new Plan3D(pp);
			double maxInter = 0;
			Piece2D father = null;
			for (Piece2D piece : rootspace.getListPieces()) {
				boolean intersectwithcavity = false;
				List<dressing.model.Cavity> cavities = piece.getPiecesCavities();
				if (cavities != null && !cavities.isEmpty()) {
					for (dressing.model.Cavity cavity : cavities) {
						plan = new Plan3D(cavity);
						Plan3D intersection = plan.getIntersection(cyl);

						if (Plan3D.getPlan3DType(intersection).equals(IntersectionType.INTERSECTION)) {
							intersectwithcavity = true;
						}
					}

				}
				if (!intersectwithcavity) {
					plan = new Plan3D(piece);
					Plan3D intersection = plan.getIntersection(cyl);

					if (Plan3D.getPlan3DType(intersection).equals(IntersectionType.INTERSECTION)) {
						double intersectionSize = intersection.getSize();
						if (intersectionSize > maxInter) {
							maxInter = intersectionSize;
							father = piece;
						}

					}
				}

			}

			if (father != null) {
				pp.deletechild(tr, true);
				father.addElement(tr);
				tr.setParentdesign(father);
				tr.setMother(pp);
				tr.transformtoParentCords();
				tr.setName(pp.getName() + "//" + tr.getName());
			}
		}
	}

	public void fixRainures(DesignObject3D rootspace, List<dressing.model.Rainure> rainures) throws Exception {
		if (rainures != null && rainures.size() > 0) {
			for (dressing.model.Rainure rn : rainures) {
				fixRainure(rootspace, rn);
			}
		}
	}

	public static void fixRainure(DesignObject3D rootspace, dressing.model.Rainure rn) throws Exception {
		Plan3D child = new Plan3D(rn);
		Piece2D pp = (Piece2D) rn.getParentdesign();
		List<Piece2D> pieces = Piece2D.getIntersectedPieces(pp, rootspace);
		Plan3D maxintersection = null;
		Piece2D maxintersectedPiece = null;
		for (Piece2D piece : pieces) {
			if (!pp.equals(piece)) {
				Plan3D plan = new Plan3D(piece);
				Plan3D intersectionRNPiece = plan.getIntersection(child);
				if (Plan3D.getPlan3DType(intersectionRNPiece).equals(IntersectionType.INTERSECTION)) {
					double intersectionSize = intersectionRNPiece.getSize();
					if (intersectionSize > 0
							&& (maxintersection == null || intersectionSize > maxintersection.getSize())) {

						maxintersection = intersectionRNPiece;
						maxintersectedPiece = piece;

					}
				}

			}

		}
		if (maxintersectedPiece != null) {
			pp.deletechild(rn, true);
			maxintersectedPiece.addElement(rn);
			rn.setMother(pp);
			rn.transformtoParentCords();
			Plan3D planParent = new Plan3D(maxintersectedPiece);
			Plan3D intersectedRainureWithParent = planParent.getIntersection(new Plan3D(rn));
			// to avoid the rainure being out of bounds of the new parent we make it take
			// its intersection with the parent as its bounds.
			// Position X and width
			rn.setXpos(intersectedRainureWithParent.getXinter().getMininter() - maxintersectedPiece.getXPosABS());
			rn.setLongeurext(intersectedRainureWithParent.getXinter().getlong());
			// Position Y and Height
			rn.setYpos(intersectedRainureWithParent.getYinter().getMininter() - maxintersectedPiece.getYPosABS());
			rn.setHauteurext(intersectedRainureWithParent.getYinter().getlong());
			// Position Z and depth
			rn.setZpos(intersectedRainureWithParent.getZinter().getMininter() - maxintersectedPiece.getZPosABS());
			rn.setProfondeurext(intersectedRainureWithParent.getZinter().getlong());
		}
	}

	public static List<Usinage> buildUsinageNode(Piece2D piece, UsinageNode node, List<Usinage> usins, boolean cavityOnly)
			throws DesignException, GeometricEngineException {
		MechanicDesign design = piece.getMechanicDesignDefinition();
		List<MechanicPublicParam> pPublicParamGroup = new ArrayList<MechanicPublicParam>();
		if (design != null && design.getPublicparamgroup() != null
				&& design.getPublicparamgroup().getMechanicpublicparam() != null) {
			pPublicParamGroup.addAll(design.getPublicparamgroup().getMechanicpublicparam());
		}
		if (node.getPublicparamgroup() != null && node.getPublicparamgroup().getMechanicpublicparam() != null
				&& !node.getPublicparamgroup().getMechanicpublicparam().isEmpty()) {
			pPublicParamGroup.addAll(node.getPublicparamgroup().getMechanicpublicparam());
		}
		List<MechanicPrivateParam> pPrivateParamGroup = new ArrayList<MechanicPrivateParam>();
		if (design != null && design.getPrivateparamgroup() != null
				&& design.getPrivateparamgroup().getMechanicprivateparam() != null)

		{
			pPrivateParamGroup = design.getPrivateparamgroup().getMechanicprivateparam();
		}
		if (node.getUsinagetrou() != null && !cavityOnly) {
			List<Trou> trous = MechanicDesignUsinageManager.buildTrous(piece, node.getUsinagetrou(), pPrivateParamGroup, pPublicParamGroup);
			usins.addAll(trous);
		}
		boolean buildrainures = true;
		if (cavityOnly && !piece.getMechanicDesignElementDefinition().getType().equals(PieceType.ACCESOIRE)) {
			buildrainures = false;
		}
		if (node.getUsinageRainure() != null && buildrainures) {
			List<dressing.model.Rainure> rainures = MechanicDesignUsinageManager.buildRainures(piece, node.getUsinageRainure(), pPrivateParamGroup,
					pPublicParamGroup);
			usins.addAll(rainures);
		}
		if (node.getCavities() != null) {
			List<dressing.model.Cavity> cavities = MechanicDesignUsinageManager.buildCavities(piece, node.getCavities(), pPrivateParamGroup,
					pPublicParamGroup);
			usins.addAll(cavities);
		}
		return usins;
	}

	public static List<Trou> buildTrous(Piece2D piece, EList<UsinageTrou> trouss, List<MechanicPrivateParam> privateparams,
			List<MechanicPublicParam> publicparams) throws DesignException, GeometricEngineException {
		List<Trou> trous = new ArrayList<Trou>();
		if (trouss != null && trouss.size() > 0) {
			for (UsinageTrou usintrou : trouss) {
				Trou tr = new Trou();
				tr.setName(usintrou.getName());
				tr.setDirection(Direction.valueOf(usintrou.getDirection().getLiteral()));
				tr.setParentdesign(piece);
				tr.setMother(piece);
				if (usintrou.getId() == null) {
					UUID id = UUID.randomUUID();
					usintrou.setId(id);
					tr.setID(id);
				} else {
					tr.setID(usintrou.getId());
				}
				tr.constructGenericDebitage(privateparams, publicparams, usintrou, piece);

				if ((Boolean) tr.get("EXIST")) {
					piece.addElement(tr);
					trous.add(tr);
					MechanicDesignUsinageManager.fixOperations(usintrou, tr);
					boolean hasOpposite = tr.get("HAS_OPPOSITE") != null ? (Boolean) tr.get("HAS_OPPOSITE")
							: usintrou.isHaveopposite();
					if (hasOpposite) {
						Trou tr1 = new Trou();
						tr1.setName(usintrou.getName() + " " + piece.getName());
						tr1.setDirection(tr.getOppositeDirection());
						tr1.setParentdesign(piece);
						tr1.setMother(piece);
						if (usintrou.getId() == null) {
							UUID id = UUID.randomUUID();
							usintrou.setId(id);
							tr1.setID(id);
						} else {
							tr.setID(usintrou.getId());
						}

						tr1.setXpos(tr.getXpos());
						tr1.setYpos(tr.getYpos());
						tr1.setZpos(tr.getZpos());
						tr1.setDepth(tr.getOppositeTrouDepth());
						tr1.setDiameter(tr.getOppositeDiameter());
						piece.addElement(tr1);
						trous.add(tr1);
						MechanicDesignUsinageManager.fixOperations(usintrou, tr1);

					}
				}

			}
		}
		return trous;
	}

	public static List<dressing.model.Rainure> buildRainures(Piece2D piece, EList<Rainure> rainures,
			List<MechanicPrivateParam> privateparams, List<MechanicPublicParam> publicparams)
			throws DesignException, GeometricEngineException {
		List<dressing.model.Rainure> rns = new ArrayList<dressing.model.Rainure>();
		if (rainures != null && rainures.size() > 0) {

			for (Rainure rn : rainures) {
				dressing.model.Rainure rainure = new dressing.model.Rainure();
				rainure.setName(rn.getName());
				rainure.setParentdesign(piece);
				rainure.setMother(piece);
				if (rn.getId() == null) {
					UUID id = UUID.randomUUID();
					rn.setId(id);
					rainure.setID(id);
				} else {
					rainure.setID(rn.getId());
				}
				rainure.constructGenericDebitage(privateparams, publicparams, rn, piece);
				if ((Boolean) rainure.get("EXIST")) {
					piece.addElement(rainure);
					MechanicDesignUsinageManager.fixOperations(rn, rainure);
					if (!rn.isIsparentcontainement()) {
						rns.add(rainure);
					}
				}

				//

			}

		}
		return rns;
	}

	public static List<dressing.model.Cavity> buildCavities(Piece2D piece, EList<Cavity> cavities,
			List<MechanicPrivateParam> privateparams, List<MechanicPublicParam> publicparams)
			throws DesignException, GeometricEngineException {
		List<dressing.model.Cavity> cvs = new ArrayList<dressing.model.Cavity>();
		if (cavities != null && cavities.size() > 0) {
			for (Cavity cv : cavities) {
				dressing.model.Cavity cavity = new dressing.model.Cavity();
				cavity.setName(cv.getName());
				cavity.setParentdesign(piece);
				cavity.setMother(piece);
				if (cv.getId() == null) {
					UUID id = UUID.randomUUID();
					cv.setId(id);
					cavity.setID(id);
				} else {
					cavity.setID(cv.getId());
				}
				cavity.constructGenericDebitage(privateparams, publicparams, cv, piece);
				if ((Boolean) cavity.get("EXIST")) {
					piece.addElement(cavity);
					MechanicDesignUsinageManager.fixOperations(cv, cavity);
				}
			}
		}
		return cvs;
	}

	public static void fixOperations(EObject object, Usinage usin) {
		if (object == null || usin == null) {
			return;
		}
		if (object instanceof Rainure || object instanceof Cavity || object instanceof UsinageTrou) {
			dressing.cam.model.Operation ops = new dressing.cam.model.Operation(usin);
			usin.addOperation(ops);
		}
	}

}
