package dressing.handlers;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.inject.Inject;

import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.di.annotations.CanExecute;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.e4.ui.workbench.modeling.EPartService;
import org.eclipse.e4.ui.workbench.modeling.ESelectionService;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;

import dressing.config.persistence.ResourceManagers;
import dressing.events.EventDriver;
import dressing.model.DesignObject3D;
import dressing.model.FaceModel;
import dressing.model.Kitchen;
import dressing.model.Piece2D;
import dressing.model.ProjectManager;
import dressing.model.ProjectRoot;
import dressing.model.Space3D;
import dressing.model.Space3DFree;
import dressing.model.SuperCadProject;
import dressing.model.UsinageNode;

import com.badlogic.gdx.Gdx;

import dressing.ui.parts.ProjectExplorerPart;
import gdxapp.object3d.KitchenElement;
import gdxapp.scenes.SceneEvent;
import param.MechanicDesign;

public class DeleteHandler {

	@Inject
	ESelectionService selectionService;

	@Inject
	IEclipseContext eclipsecontext;

	private Object getActiveSelection(EPartService partService) {
		MPart part = partService.getActivePart();
		Object activePart = part != null ? part.getObject() : null;

		return (activePart instanceof ProjectExplorerPart)
				? selectionService.getSelection("dressing.part.projectexplorer")
				: selectionService.getSelection("dressing.part.tree");
	}

	@Execute
	public void execute(EPartService partService, Shell shell) {
		Object selection = getActiveSelection(partService);

		if (selection == null || !confirmDeletion(shell))
			return;

		if (selection instanceof DesignObject3D) {
			handleSingleDeletion((DesignObject3D) selection, shell);

		} else if (selection instanceof List<?>) {
			handleMultipleDeletion((List<?>) selection, shell);
		}
	}

	// ===========================
	// === CAN EXECUTE
	// ===========================
	@CanExecute
	public boolean check(EPartService partService) {
		Object selection = getActiveSelection(partService);
		if (selection instanceof Kitchen || selection instanceof SuperCadProject)
			return false;
		return (selection instanceof DesignObject3D) || (selection instanceof List<?>);
	}

	public void removeElementFromKitchen(DesignObject3D space) {
		for (Kitchen kitchen : ProjectManager.getManager().getCurrentProject().getKitchens()) {
			if (kitchen.containsElement(space))
			{
				kitchen.removeElement(space);
				KitchenElement kitchenElement = ProjectManager.getInstance().getCurrentKitchen().getScene()
						.getKitchenElement(space);
				if(kitchenElement!=null)
				EventDriver.getDriver().deliverEvent(SceneEvent.REMOVE_OBJECT_REQUEST, kitchenElement);

			}
		}
	}

	private Kitchen getKitchen(DesignObject3D space) {
		for (Kitchen kitchen : ProjectManager.getManager().getCurrentProject().getKitchens()) {
			if (kitchen.containsElement(space))
				return kitchen;
		}
		return null;
	}

	private void handleSingleDeletion(DesignObject3D element, Shell shell) {
		Kitchen kitchen = getKitchen(element);
		if (kitchen != null) {
			kitchen.removeElement(element);
		} else {
			deleteSingle(element, shell);
		}
	}

	private void handleMultipleDeletion(List<?> selection, Shell shell) {
		if (selection == null || selection.isEmpty())
			return;
		List<?> list = new java.util.ArrayList<>(selection);
		Set<DesignObject3D> kitchenElements = getKitchenElementsFromList(list);
		list.removeAll(kitchenElements);
		list.removeIf(
				elt -> elt instanceof DesignObject3D && kitchenElements.contains(((DesignObject3D) elt).getRoot()));
		for (DesignObject3D element : kitchenElements) {
			removeElementFromKitchen(element);
		}
		deleteMultiple(list, shell);
	}

	private Set<DesignObject3D> getKitchenElementsFromList(List<?> list) {
		Set<DesignObject3D> kitchenElements = new HashSet<DesignObject3D>();
		for (Object element : list) {
			if (!(element instanceof DesignObject3D))
				continue;
			Kitchen kitchen = getKitchen((DesignObject3D) element);
			if (kitchen != null) {
				kitchenElements.add((DesignObject3D) element);
			}
		}
		return kitchenElements;
	}

	// ===========================
	// === SINGLE SELECTION
	// ===========================
	private void deleteSingle(DesignObject3D child, Shell shell) {
		if (child == null || child.getParentdesign() == null)
			return;

		if (!child.isDirectlyDelatable()) {
			MessageDialog.openWarning(shell, "Suppression impossible",
					"Impossible de supprimer le modèle sélectionné.");
			return;
		}

		DesignObject3D parent = child.getParentdesign();

		try {
			if (child instanceof FaceModel && parent instanceof Piece2D) {
				deleteFaceModel((FaceModel) child);
			} else if (child instanceof UsinageNode && parent instanceof Piece2D) {
				deleteUsinageNode((UsinageNode) child);
			} else {
				StringBuilder dependents = new StringBuilder();
				boolean hasDeps = hasDependents(child, parent, dependents);

				if (hasDeps) {
					MessageDialog.openWarning(shell, "Suppression bloquée",
							"Les modèles suivants référencent cet élément :\n" + dependents
									+ "\n\nVeuillez les supprimer avant de supprimer cet élément (P"
									+ (parent.getChilds().indexOf(child) + 1) + " " + child.getName() + ").");
					return;
				}

				withNotificationPaused(new Runnable() {
					@Override
					public void run() {
						try {
							parent.deletechild(child, true);
						} catch (Exception e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
							MessageDialog.openError(shell, "Erreur lors du supprission du model",
									"Erreur lors du suppression du model" + child.getName());
						}
					}
				});
				updateDesignKitchenElement(parent);

			}

		} catch (Exception e) {
			MessageDialog.openError(shell, "Erreur de suppression",
					"Une erreur est survenue lors de la suppression du modèle : " + child.getName());
			e.printStackTrace();
		}
	}

	// ===========================
	// === MULTIPLE SELECTIONS
	// ===========================
	private void deleteMultiple(List<?> list, Shell shell) {

		withNotificationPaused(new Runnable() {
			@Override
			public void run() {
				Set<DesignObject3D> modifiedRoots = new HashSet<DesignObject3D>();
				for (Object element : list) {
					if (!(element instanceof DesignObject3D))
						continue;

					DesignObject3D child = (DesignObject3D) element;
					DesignObject3D parent = child.getParentdesign();
					if (parent == null)
						continue;

					if (!child.isDirectlyDelatable()) {
						MessageDialog.openWarning(shell, "Suppression impossible",
								"Impossible de supprimer le modèle sélectionné.");
						continue;
					}

					try {
						if (child instanceof FaceModel && parent instanceof Piece2D) {
							deleteFaceModel((FaceModel) child);
						} else if (child instanceof UsinageNode && parent instanceof Piece2D) {
							deleteUsinageNode((UsinageNode) child);
						} else {
							StringBuilder deps = new StringBuilder();
							boolean hasDeps = hasDependents(child, parent, deps);
							if (!hasDeps) {
								parent.deletechild(child, true);
								modifiedRoots.add(parent.getRoot());
							} else {

								MessageDialog.openWarning(shell, "Suppression bloquée",
										"Les modèles suivants référencent cet élément :\n" + deps
												+ "\n\nVeuillez les supprimer avant de supprimer cet élément (P"
												+ (parent.getChilds().indexOf(child) + 1) + " " + child.getName()
												+ ").");

							}
						}
					} catch (Exception e) {
						e.printStackTrace();
						MessageDialog.openError(shell, "Erreur de suppression",
								"Erreur lors de la suppression d’un modèle.");
					}
				}
				modifiedRoots.forEach(elt -> updateKitchenElement(elt));
			}
		});
	}

	// ===========================
	// === HELPERS
	// ===========================
	private void deleteFaceModel(FaceModel face) throws Exception {
		if (face.getNode() != null)
			face.delete();
		fireModelDeleted();
	}

	private void deleteUsinageNode(UsinageNode node) throws Exception {
		if (node.getNode() != null)
			node.delete();
		fireModelDeleted();
	}

	private boolean hasDependents(DesignObject3D child, DesignObject3D parent, StringBuilder dependents) {
		if (parent == null || parent.getRootModule() == null)
			return false;
		if (!(parent.getRootModule() instanceof Space3D))
			return false;

		Space3D space = (Space3D) parent.getRootModule();
		if (space.getDependenceController() == null || child.getMechanicDesignDefinition() == null)
			return false;

		List<MechanicDesign> deps = space.getDependenceController().getDependents(child.getMechanicDesignDefinition());
		if (deps == null || deps.isEmpty())
			return false;

		if (!ResourceManagers.getIntance().getPreference().getCalculEnginePrefs().isDeletePieceWithReferences()) {
			for (MechanicDesign dependent : deps) {
				DesignObject3D design = space.getDependenceController().getIntsatanceToDefinition().get(dependent);
				int index = parent.getChilds().indexOf(design);
				dependents.append(" P").append(index + 1).append(" ").append(dependent.getName()).append(",");
			}
			if (dependents.length() > 0)
				dependents.setLength(dependents.length() - 1); // remove last comma
			return true;
		}
		return false;
	}

	private void withNotificationPaused(Runnable action) {
		ProjectRoot root = ProjectManager.getInstance().getRoot();
		if (root == null) {
			action.run();
			return;
		}
		root.setNotificationon(false);
		try {
			action.run();
		} finally {
			root.setNotificationon(true);
			fireModelDeleted();
		}
	}

	private void fireModelDeleted() {
		ProjectRoot root = ProjectManager.getInstance().getRoot();
		if (root != null) {
			root.firePropertyChange("project.child.delete", 1, 2);
		}
	}

	private boolean confirmDeletion(Shell shell) {
		return MessageDialog.openConfirm(shell, "Confirmation de suppression",
				"Voulez-vous supprimer le ou les composants sélectionnés ?");
	}

	private void updateDesignKitchenElement(DesignObject3D element) {
		Gdx.app.postRunnable(new Runnable() {
			@Override
			public void run() {
				updateKitchenElement(element.getRoot());
			}

		});
	}

	private void updateKitchenElement(DesignObject3D rootspace) {
		DesignObject3D element = (rootspace != null) ? rootspace.getRoot() : null;
		if (element == null)
			return;

		KitchenElement kitchenElement = ProjectManager.getInstance().getCurrentKitchen().getScene()
				.getKitchenElement(element);
		if (kitchenElement == null)
		    return;
		kitchenElement.getObjectDimention();
		kitchenElement.setRequireRefrech(true);
		EventDriver.getDriver().deliverEvent(SceneEvent.OBJECT_CHANGED, kitchenElement);

	}
}
