package gdxapp.quotation;

import com.badlogic.gdx.graphics.Color;

import org.frs.svg.Line;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import gdxapp.assets.AssetFont;
import gdxapp.object3d.Object3D;

public class Object3DQuotation {
	
	private Vector3 o0;
	private Vector3 o1;
	private Object3D source;
	//drawing porperties
	private Vector2 p0 = new Vector2();
	private Vector2 p1 = new Vector2();
	private Vector2 slash00 = new Vector2(), slash01 = new Vector2(), slash10 = new Vector2(), slash11 = new Vector2();
	private final float value;
	private Vector3 color = new Vector3();
	private float textRotation = 0;
	private Vector3 textTranslation = new Vector3();

	public Object3DQuotation(Vector3 v0, Vector3 v1, Object3D source) {
		this.o0 = v0;
		this.o1 = v1;
		this.value = v0.dst(v1);
		if(source != null)
			this.source = source;
	}
	
	public Object3DQuotation(Vector3 v0, Vector3 v1, Object3D source, float value) {
		this.o0 = v0;
		this.o1 = v1;
		this.value = value;
		if(source != null)
			this.source = source;
	}
		
	
	public void draw(Batch batch, ShapeRenderer shapeRenderer) {
		shapeRenderer.setColor(new Color(this.color.x, this.color.y, this.color.z, 1.0f));
		shapeRenderer.begin(ShapeType.Filled);
		shapeRenderer.setColor(Color.GREEN);
		shapeRenderer.rectLine(p0, p1, 3);
		shapeRenderer.line(slash00, slash01);
		shapeRenderer.line(slash10, slash11);
//		shapeRenderer.circle(p0.x, p0.y, 4);
//		shapeRenderer.circle(p1.x, p1.y, 4);
//		shapeRenderer.setColor(Color.RED);
//		shapeRenderer.circle((p0.x + p1.x)/2.0f, (p0.y + p1.y)/2.0f, 8);

		shapeRenderer.end();
		Vector3 v = new Vector3(p0.cpy().add(p1).scl(0.5f), 0);
		//calculating the angle
		Vector2 angleVector = new Vector2(p1.x - p0.x, p1.y - p0.y);
		if(angleVector.x < 0) {
			angleVector.scl(-1);
		}				
		float angle = angleVector.angle();
		if(angle > 180)
			angle = angle - 360;
		Matrix4 transform = new Matrix4();
		transform.set(v, new Quaternion(Vector3.Z, angle), new Vector3(0.75f, 0.75f, 0.75f));
		String measure = String.format("%.0f", value * 1000.0f);
		if(batch != null) {
			batch.setTransformMatrix(transform);
			batch.begin();
			AssetFont.getInstance().getFont().draw(batch, "" + measure, -25, 25, 50, 0, false, Color.BLUE);
			batch.end();
		}
	}
	
	public String drawSVG() {
		int r = 2;
		String svg = String.format("<line x1=\"%.0f\" y1=\"%.0f\" x2=\"%.0f\" y2=\"%.0f\" style=\"stroke:blue;stroke-width:2\" />\r\n",
				o0.x, o0.y,o1.x,o1.y);
		
		svg += String.format("<line x1=\"%.0f\" y1=\"%.0f\" x2=\"%.0f\" y2=\"%.0f\" style=\"stroke:blue;stroke-width:2\" />\r\n",
				o0.x-r, o0.y-r,o0.x + r,o0.y + r);
		
		svg += String.format("<line x1=\"%.0f\" y1=\"%.0f\" x2=\"%.0f\" y2=\"%.0f\" style=\"stroke:blue;stroke-width:2\" />\r\n",
				o1.x-r, o1.y-r,o1.x + r,o1.y + r);
		
		String text = """
				 	<text x="0" y="0" text-anchor="middle" fill="black" transform="translate(%.0f %.0f) rotate(%.0f 0 0)">%.0f
				    </text>/r/n
				""";
		Vector3 middle = this.o0.cpy().add(o1).scl(0.5f).add(textTranslation);
		svg += String.format(text,middle.x, middle.y, this.textRotation, this.value * 1000);
		return svg;
	}
	
	public float getLength() {
		return o0.dst(o1);
	}
	

	
	public float getLengthOnScreen() {
		return p0.dst(p1);
	}
	
	public void transformToCameraSpace(Camera camera, int width, int height) {
		this.o0 = camera.project(o0.cpy(), 0, 0, width, height).scl(1,-1,0).add(0, height, 0);
		this.o1 = camera.project(o1.cpy(), 0, 0, width, height).scl(1,-1,0).add(0, height, 0);
	}
	
	public void calculateDrawingPoints(Camera camera) {
		Vector3 v0 = camera.project(o0.cpy());
		Vector3 v1 = camera.project(o1.cpy());
		p0.set(v0.x, v0.y);
		p1.set(v1.x, v1.y);
		slash00.set(p0.cpy().add(6,6));
		slash01.set(p0.cpy().sub(6,6));
		slash10.set(p1.cpy().add(6,6));
		slash11.set(p1.cpy().sub(6,6));
	}

	public Vector3 getCenter() {
		return o0.cpy().add(o1).scl(0.5f);
	}

	public Vector2 getOnScreenCenter() {
		return p1.cpy().add(p0).scl(.5f);
	}
	
	
	public Vector2 getOnScreenDirection(boolean normalized) {
		Vector2 dir = p1.cpy().sub(p0);
		if(normalized)
			dir.nor();
		return dir;
	}
	
	public Vector3 getDirection(boolean normalized) {
		Vector3 dir = o0.cpy().sub(o1);
		if(normalized)
			dir.nor();
		return dir;
	}
	
	public void setX(float x) {
		p0.x = x;
		p1.x = x;
	}
	
	public void setY(float y) {
		p0.y = y;
		p1.y = y;
	}
	
	//supposing both quotation lies on the same line
	public float intersect(Object3DQuotation other) {
		return getOnScreenCenter().dst(other.getOnScreenCenter()) - (getLengthOnScreen() + other.getLengthOnScreen())/2.0f;
	}
	
	public float intersectAxe(Object3DQuotation other, Vector2 axe) {
		float distanceBetweenCenters = Math.abs(  getOnScreenCenter().cpy().sub(other.getOnScreenCenter()).dot(axe));
		float intersectionDistance = (Math.abs(getOnScreenDirection(false).dot(axe)) + Math.abs(other.getOnScreenDirection(false).dot(axe)))/2.0f;
		return distanceBetweenCenters - intersectionDistance;
	}


	public Object3D getSource() {
		return source;
	}


	public void setSource(Object3D source) {
		this.source = source;
	}


	public Vector3 getColor() {
		return color;
	}


	public void setColor(Vector3 color) {
		this.color = color;
	}


	public Vector3 getO0() {
		return o0;
	}


	public void setO0(Vector3 o0) {
		this.o0 = o0;
	}


	public Vector3 getO1() {
		return o1;
	}


	public void setO1(Vector3 o1) {
		this.o1 = o1;
	}


	@Override
	public String toString() {
		return "Object3DQuotation [o0=" + o0 + ", o1=" + o1 + ", source=" + source + ", p0=" + p0 + ", p1=" + p1 + "]";
	}


	public Line getLine() {
		return new Line(o0.cpy(), o1.cpy().sub(o0));
	}

	public float getValue() {
		return value;
	}

	public void translate(Vector3 translation) {
		this.o0.add(translation);
		this.o1.add(translation);
	}

	public float getTextRotation() {
		return textRotation;
	}

	public void setTextRotation(float textRotation) {
		this.textRotation = textRotation;
	}

	public Vector3 getTextTranslation() {
		return textTranslation;
	}

	public void setTextTranslation(Vector3 textTranslation) {
		this.textTranslation = textTranslation;
	}
	
	
	
	
	
}
