package org.frs.svg;

import java.util.Objects;

import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

public class LineSegment {

	private Vector3 v0;
	private Vector3 v1;
	private Vector3 color;
	
	public LineSegment(Vector3 v0, Vector3 v1) {
		super();
		this.v0 = v0;
		this.v1 = v1;
	}
	
	public LineSegment transform(Matrix4 transform) {
		this.v0.mul(transform);
		this.v1.mul(transform);
		return this;
	}
	
	public LineSegment cpy() {
		return new LineSegment(v0.cpy(), v1.cpy());
	}
	
	public Vector3 getDirection(boolean normalized) {
		Vector3 dir = v1.cpy().sub(v0);
		if(normalized)
			dir.nor();
		return dir;
	}
	
	public float len() {
		return v0.dst(v1);
	}
	
	public float getLen2() {
		return v0.dst2(v1);
	}
	
	public Vector3 getDir(boolean normalized) {
		Vector3 dir = v1.cpy().sub(v0);
		if(normalized)
			dir.nor();
		return dir;
	}
	
	public boolean contains(Vector3 point) {
		float dst = Math.abs(point.dst(v0) + point.dst(v1) - v1.dst(v0));
//		System.out.println("distance: " + dst);
		return Math.abs(point.dst(v0) + point.dst(v1) - v1.dst(v0)) < 0.1f;
	}
	
	public boolean contains(LineSegment other) {
		return contains(other.getV0()) && contains(other.getV1());
	}
	
	public Vector3 intersect(Line line) {
		Line lineX = new Line(v0, v1.cpy().sub(v0));
		Object intersection = lineX.intersect(line);
		if(intersection instanceof Vector3) {
			Vector3 cutPoint = (Vector3) intersection;
			return contains(cutPoint)?cutPoint:null;
		}
		return null;
	}
	
	public Vector3 intersect(LineSegment other) {
		Line line = new Line(other.getV0(), other.getV1().cpy().sub(other.getV0()));
		Vector3 intersection = intersect(line);
		if(intersection != null && other.contains(intersection))
			return intersection;
		return null;
		
	}
	
	public float dst(LineSegment other) {
		Vector3 p0 = new Vector3();
		Vector3 p1 = new Vector3();
		float distance = new Line(v0, v1.cpy().sub(v0)).dst(new Line(other.getV0(), other.getV1().cpy().sub(other.getV0())));
		if((contains(p0) && other.contains(p1)) ||(contains(p1) && other.contains(p0))) {
			return distance;
		}else {
			return Math.min(Math.min(v0.dst(other.getV0()), v0.dst(other.getV1())), Math.min(v1.dst(other.getV0()), v1.dst(other.getV1())));
		}
		
		
	}
	
	public boolean startOrEndWith(Vector3 point, float epsilon) {
		return v0.dst(point) < epsilon || v1.dst(point) < epsilon;
	}
	
	public Vector3 getV0() {
		return v0;
	}
	public void setV0(Vector3 v0) {
		this.v0 = v0;
	}
	public Vector3 getV1() {
		return v1;
	}
	public void setV1(Vector3 v1) {
		this.v1 = v1;
	}
	
	public Vector3 getColor() {
		return color;
	}
	public void setColor(Vector3 color) {
		this.color = color;
	}
	
	public boolean equals(LineSegment other) {
		return other.startOrEndWith(v0, 0.01f) && other.startOrEndWith(v1, 0.01f);
	}
	
	

	@Override
	public int hashCode() {
		return Objects.hash(color, v0, v1);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		LineSegment other = (LineSegment) obj;
		return Objects.equals(color, other.color) && Objects.equals(v0, other.v0) && Objects.equals(v1, other.v1);
	}

	@Override
	public String toString() {
		return "LineSegment [v0=" + v0 + ", v1=" + v1 + "]";
	}

	public Vector3 getCenter() {
		return v0.cpy().add(v1).scl(0.5f);
	}

	public LineSegment translate(Vector3 translation) {
		this.v0.add(translation);
		this.v1.add(translation);
		return this;
	}

	



}
