package gdxapp;


import java.util.List;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.PerspectiveCamera;
import com.badlogic.gdx.graphics.Pixmap;
import com.badlogic.gdx.graphics.Pixmap.Format;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelBatch;
import com.badlogic.gdx.graphics.g3d.Renderable;
import com.badlogic.gdx.graphics.g3d.Shader;
import com.badlogic.gdx.graphics.g3d.utils.ShaderProvider;
import com.badlogic.gdx.graphics.glutils.FrameBuffer;
import com.badlogic.gdx.graphics.glutils.GLFrameBuffer.FrameBufferBuilder;
import com.badlogic.gdx.math.Vector3;
import gdxapp.Commun.ScreenshotFactory;
import gdxapp.object3d.DesignModelFactory;
import gdxapp.object3d.KitchenElement;
import gdxapp.object3d.ModelInfo;
import gdxapp.object3d.Object3D;
import gdxapp.object3d.ObjectModel;
import gdxapp.screens.room3d.MouseOnlyCameraController;
import gdxapp.shaders.SceneShader;

public class ModelViewer {
	
	Camera camera;
	MouseOnlyCameraController cameraInputController;
	SceneShader shader;
	ModelBatch batch;
	Object3D model;
	KitchenElement element;
	
	private FrameBuffer frameBuffer;
	public FrameBuffer getFrameBuffer() {
		return frameBuffer;
	}

	int height;
	float aspectRatio = 16.0f/9.0f;

	public ModelViewer(int height, float aspectRatio) {
		this.height = height;
		this.aspectRatio = aspectRatio;
		createCameraAndShaders();
		createFrameBuffer();
	}
	
	private void createFrameBuffer() {
		int width = Math.round(aspectRatio * height); 
		FrameBufferBuilder builder = new FrameBufferBuilder(width, height);
		builder.addBasicColorTextureAttachment(Format.RGBA8888);
		builder.addBasicDepthRenderBuffer();
		frameBuffer = builder.build();
	}
	
	public void createCameraAndShaders() {
		camera = new PerspectiveCamera(63, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
		camera.up.set(Vector3.Y.cpy());
		camera.near = 0.01f;
		camera.far = 10;
		camera.position.set(0, 0, 10);
		camera.lookAt(new Vector3());
		camera.update();
		cameraInputController = new MouseOnlyCameraController(camera);
		cameraInputController.setOrbitCenter(new Vector3());
		shader = new SceneShader();
		shader.init();
		batch = new ModelBatch(new ShaderProvider() {
			@Override
			public Shader getShader(Renderable renderable) {
				return shader;
			}
			@Override
			public void dispose() {
				shader.dispose();
			}
		});
	}
	
	public Pixmap render(KitchenElement element, Vector3 cameraPosition) {
		frameBuffer.begin();
		setElement(element);
		setCameraPosition(cameraPosition);
		cameraInputController.update();
		Gdx.gl.glClearColor(1.0f, 1.0f, 1.0f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT | (Gdx.graphics.getBufferFormat().coverageSampling ? GL20.GL_COVERAGE_BUFFER_BIT_NV : 0));
        if(model != null) {
        	batch.begin(camera);
        	model.render(batch);
        	batch.end();
        }
        Pixmap map = ScreenshotFactory.getScreenshot(0, 0, frameBuffer.getWidth(), frameBuffer.getHeight(), true);
		frameBuffer.end();
        return map;
	}
	public Pixmap render(List<Object3D> renderables, Camera camera) {
		frameBuffer.begin();
		Gdx.gl.glClearColor(1.0f, 1.0f, 1.0f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT | (Gdx.graphics.getBufferFormat().coverageSampling ? GL20.GL_COVERAGE_BUFFER_BIT_NV : 0));
    	batch.begin(camera);
        for(var model: renderables) {
        	model.render(batch);
        }
    	batch.end();
    	Pixmap map = ScreenshotFactory.getScreenshot(0, 0, frameBuffer.getWidth(), frameBuffer.getHeight(), true);
		frameBuffer.end();
    	return map;
	}

	
	public void setCameraPosition(Vector3 position){
		cameraInputController.getCamera().position.set(position);
		cameraInputController.setOrbitCenter(this.model.getPosition());
		cameraInputController.getCamera().up.set(0,1,0);
		cameraInputController.getCamera().lookAt(cameraInputController.getOrbitCenter());
		cameraInputController.getCamera().update();
	}


	public KitchenElement getElement() {
		return element;
	}

	public void setElement(KitchenElement element) {
		if(this.element == null || this.element != element) {
			this.element = element;
			if(element.getModel() == null || element.isRequireRefrech()) {
				Model model = null;
	    		try {
	            	model = DesignModelFactory.getFactory().createModelFromVertices(this.element);
	    		}catch (Exception e) {
	    			try {
						DesignModelFactory.getFactory().getModelBuilder().end();
	    			}catch (Exception ex) {
						System.err.print("attempt to end model builder failed \n" + ex.getMessage() );
					}
				}
	    		ModelInfo info = new ModelInfo();
	    		info.setToDefault();
	    		ObjectModel objectModel = new ObjectModel(model, info);
	    		this.element.setModel(objectModel);
			}
			KitchenElement clone = this.element.clone();
			clone.setRealWorldPosition(new Vector3());
			clone.setRotation(0);
			this.model = new Object3D(clone);
		}
	}
	
	public void dispose() {
		frameBuffer.dispose();
	}



	





}
