package gdxapp.screens.room3d;

import java.util.HashMap;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.InputProcessor;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.config.UserPreference;
import dressing.events.Event;
import dressing.events.EventDriver;
import dressing.events.EventHandler;
import gdx.app.camera.PerspectiveCameraController;
import gdxapp.object3d.KitchenElement;
import gdxapp.object3d.Object3D;
import gdxapp.scenes.SceneEvent;
import gdxapp.screens.ObjectEditor.Object3DRightClickMenu;
import gdxapp.screens.ObjectEditor.ScreenUtilities;


public class MouseOnlyCameraController extends PerspectiveCameraController implements InputProcessor, EventHandler {
	
	private OrthographicCamera orthoCamera;
	private Camera currentCamera;
	private Vector2 tmpLocation = new Vector2();
	private Vector2 tmpDrag = new Vector2();
	private Vector3 tmpTarget = new Vector3();
	private Vector3 realWorldPoint = new Vector3();
	private long lastClickDate;
	private boolean doubleClick;
	
	
	
	private Object3D isolatedObject;

	
	public MouseOnlyCameraController(Camera camera) {
		super();
		subscribe(SceneEvent.ISOLATE_OBJECT.name());
		this.camera = camera;
		orthoCamera = new OrthographicCamera(8, 8);
		orthoCamera.near = 0f;
		orthoCamera.far = 10;
		orthoCamera.up.set(0,1,0);
		orthoCamera.position.set(orbitCenter);
		orthoCamera.lookAt(orthoCamera.position.add(0,0,-1));
		orthoCamera.update();
		currentCamera = camera;
	}
	
	public void resetOrthographicCamera() {
		orthoCamera.viewportWidth = 8;
		orthoCamera.viewportHeight = 8;
		orthoCamera.near = 0f;
		orthoCamera.far = 10;
		orthoCamera.up.set(0,1,0);
		orthoCamera.position.set(orbitCenter);
		orthoCamera.lookAt(orthoCamera.position.add(0,0,-1));
		orthoCamera.update();
	}
	
	
	
	
	
	
	
	public void update() {
		handleKeyDown();
		camera.update();
	}
	
	private void setToOrtho(Vector3 side) {
		if(side.epsilonEquals(Vector3.Z)) {
			orthoCamera.up.set(0,1,0);
			orthoCamera.position.set(orbitCenter.x, 1.25f, 10);
			orthoCamera.lookAt(orbitCenter.x, 1.25f, 0);
			orthoCamera.update();
			currentCamera = orthoCamera;
		}else if(side.epsilonEquals(new Vector3(0,-1,0))) {
			orthoCamera.up.set(0,0,1);
			orthoCamera.position.set(orbitCenter.x, 10.0f, orbitCenter.z);
			orthoCamera.lookAt(orbitCenter.x, 0, orbitCenter.z);
			orthoCamera.update();
			currentCamera = orthoCamera;
		}else if(side.epsilonEquals(Vector3.X)) {
			orthoCamera.up.set(0,1,0);
			orthoCamera.position.set(orbitCenter.x, 1.25f, orbitCenter.z);
			orthoCamera.lookAt(10, 1.25f, orbitCenter.z);
			orthoCamera.update();
			currentCamera = orthoCamera;
		}else if(side.epsilonEquals(Vector3.X.cpy().scl(-1))) {
			orthoCamera.up.set(0,1,0);
			orthoCamera.position.set(orbitCenter.x, 1.25f, orbitCenter.z);
			orthoCamera.lookAt(-10, 1.25f, orbitCenter.z);
			orthoCamera.update();
			currentCamera = orthoCamera;
		}
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MODE_CHANGED.name(), null));
	}
		
	public Camera getCurrentCamera() {
		return currentCamera;
	}

	public void setCurrentCamera(Camera currentCamera) {
		if(currentCamera != this.currentCamera) {
			this.currentCamera = currentCamera;
			resetOrthographicCamera();
			EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MODE_CHANGED.name(), null));
		}
	}

	public void handleKeyDown() {
		if(Gdx.input.isKeyPressed(Input.Keys.UP)) {
			moveVertically(1.0f);
		}
		if(Gdx.input.isKeyPressed(Input.Keys.DOWN)) {
			moveVertically(-1.0f);
		}
		if(Gdx.input.isKeyPressed(Input.Keys.Q)) {
			moveLaterally(1f);
		}
		if(Gdx.input.isKeyPressed(Input.Keys.D)) {
			moveLaterally(-1);
		}
		if(Gdx.input.isKeyPressed(Input.Keys.Z)) {
			moveForward(0.5f);
		}
		if(Gdx.input.isKeyPressed(Input.Keys.S)) {
			moveForward(-0.5f);
		}
	}

	@Override
	public boolean keyDown(int keycode) {
		if(keycode == Input.Keys.MINUS){
			adjustFiledOfView(1.1f);
		}
		if(keycode == Input.Keys.PLUS){
			adjustFiledOfView(0.9f);
		}
		if(keycode == Input.Keys.NUMPAD_8){
			setToOrtho(Vector3.Z);
		}
		if(keycode == Input.Keys.NUMPAD_4){
			setToOrtho(Vector3.X);
		}
		if(keycode == Input.Keys.NUMPAD_6){
			setToOrtho(Vector3.X.cpy().scl(-1));
		}
		if(keycode == Input.Keys.NUMPAD_5){
			setToOrtho(Vector3.Y.cpy().scl(-1));
		}

		return true;
	}

	@Override
	public boolean keyUp(int keycode) {
		// TODO Auto-generated method stub
		return true;
	}

	@Override
	public boolean keyTyped(char character) {
		// TODO Auto-generated method stub
		return true;
	}

	@Override
	public boolean touchDown(int screenX, int screenY, int pointer, int button) {		
		
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CLEAR_KEYBOARD_FOCUS.name(), null));
		if(button == Input.Buttons.LEFT) {
			long instant = System.currentTimeMillis();
			doubleClick = false;
			if(instant - lastClickDate < 400)
				doubleClick = true;
			lastClickDate = instant;
			if(currentCamera == orthoCamera) {
				if(doubleClick) {
					setCurrentCamera(camera);
					EventDriver.getDriver().deliverEvent(new Event(SceneEvent.UNHIDE_ALL.name(), null));
				}
			}
			
		}else if(button == Input.Buttons.RIGHT) {
			if(currentCamera == orthoCamera) {
				HashMap<String, Object> eventInfo = new HashMap<String, Object>();
				eventInfo.put("location", new Vector2(screenX, Gdx.graphics.getHeight() - screenY));
				eventInfo.put("object", isolatedObject);
				eventInfo.put("content", Object3DRightClickMenu.PLUMBING);
				EventDriver.getDriver().deliverEvent(new Event(SceneEvent.OBJECT_RIGHT_CLICK.name(), eventInfo));
			}
		}
		tmpLocation.set(screenX, Gdx.graphics.getHeight() - screenY);
		return true;
	}

	@Override
	public boolean touchUp(int screenX, int screenY, int pointer, int button) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public boolean touchDragged(int screenX, int screenY, int pointer) {
		tmpDrag.set(screenX, Gdx.graphics.getHeight() - screenY);
		Vector2 mvm = tmpLocation.cpy().sub(tmpDrag);

		
		float coef = UserPreference.getPreference().getScenePreferences().getCameraSensitivity()/50.0f;
		
		Vector2  speed = new Vector2(coef * 10, coef);
		
		if(Math.abs(mvm.x) > Math.abs(mvm.y)) {
			if(Gdx.input.isButtonPressed(Input.Buttons.RIGHT)) {
				moveLaterally(speed.x * mvm.x);
			}
			if(Gdx.input.isButtonPressed(Input.Buttons.LEFT)) {
				orbit(speed.x * mvm.x);
			}

		}else {
			if(Gdx.input.isButtonPressed(Input.Buttons.RIGHT)) {
				moveVertically(speed.y * mvm.y);
			}
			if(Gdx.input.isButtonPressed(Input.Buttons.LEFT)) {
				tild(speed.y * mvm.y);
			}
		}

		tmpLocation.set(tmpDrag);
		return true;
	}

	@Override
	public boolean mouseMoved(int screenX, int screenY) {
		realWorldPoint.set(ScreenUtilities.getRealWorldPoint(screenX, screenY, currentCamera));
		return false;
	}

	@Override
	public boolean scrolled(int amount) {
		if(currentCamera == orthoCamera) {
			Vector3 translation = realWorldPoint.cpy().sub(orthoCamera.position).scl(1,1,0.1f).nor().scl(-0.2f * Math.signum(amount));
			orthoCamera.position.add(translation);
			orthoCamera.zoom += Math.signum(amount) * 0.1f;
			orthoCamera.update();
		}else {
			moveForward(-amount);

		}
		return true;
	}








	@Override
	public void handle(Event event) {
		SceneEvent sceneEvent = SceneEvent.valueOf(event.getTopic());
		switch(sceneEvent) {
		case ISOLATE_OBJECT:
			setToProjectionAgainst(event.getData(), false);
			break;
		}
	}

	public void setToProjectionAgainst(Object data, boolean offScreenMode) {
		if(data instanceof Object3D) {
			isolatedObject = (Object3D) data;
			float angle =  isolatedObject.getOptimalVolumeAngle();
			isolatedObject.rotate(Vector3.Y, angle);
			isolatedObject.calculateProperties();
			Vector3 dims = isolatedObject.getDimension();
			isolatedObject.rotate(Vector3.Y, -angle);
			isolatedObject.calculateProperties();
			Vector3 dir = new Vector3();
			float width = Math.max(isolatedObject.getWorldObject().getRealWorldDimension().x, isolatedObject.getWorldObject().getRealWorldDimension().z);
			float height = isolatedObject.getWorldObject().getRealWorldDimension().y;
			if(isolatedObject.getWorldObject() instanceof KitchenElement){
				dir.set(1, 0, 0).rotate(Vector3.Y, isolatedObject.getWorldObject().getRotation());
			}else {
				Vector3 xVect = dims.cpy().scl(1,0,0);
				Vector3 zVect = dims.cpy().scl(0,0,1);
				dir = (xVect.len2() < zVect.len2())? zVect: xVect;
				width = dir.len();
				dir.nor().rotate(Vector3.Y, -angle);
			}
			dir = dir.crs(0, 1, 0);
			Vector3 l = getCurrentCamera().position.cpy().sub(isolatedObject.getPosition());
			if(l.dot(dir) < 0)
				dir.scl(-1);
			dir.scl(2.0f);
			Vector3 cameraPosition = isolatedObject.getPosition().cpy().add(dir);
			float viewportWidth = width * 1.5f;
			if(offScreenMode) {
				orthoCamera.viewportWidth = viewportWidth;
				orthoCamera.viewportHeight = viewportWidth * height/width;
			}
			orthoCamera.up.set(0,1,0);
			orthoCamera.position.set(cameraPosition);
			orthoCamera.lookAt(isolatedObject.getPosition().cpy());
			orthoCamera.update();
			currentCamera = orthoCamera;
			Room3DController.getInstance().isolateObject(isolatedObject);
		}
	}

	public OrthographicCamera getOrthoCamera() {
		return orthoCamera;
	}

	
	
}
