package gdxapp.object3d;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eclipse.swt.widgets.Display;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.Input.Keys;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.Event;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.InputListener;
import com.badlogic.gdx.scenes.scene2d.Touchable;

import dressing.controller.tools.ToolController;
import dressing.events.EventDriver;
import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Vector4;
import dressing.model.DesignObject3D;
import dressing.model.ProjectManager;
import dressing.model.Space3D;
import dressing.ui.ChangeCommand;
import dressing.ui.CommandStack;
import dressing.ui.MoveCommand;
import dressing.ui.RotateCommand;
import dressing.ui.ScaleCommand;
import dressing.ui.parts.GdxPart;
import dressing.ui.util.MesureWidow;
import gdxapp.Commun.AbstractScreen;
import gdxapp.Commun.GroupSelection;
import gdxapp.assets.DrawingHelper;
import gdxapp.object3d.WorldObject.ObjectType;
import gdxapp.scenes.Scene;
import gdxapp.scenes.SceneEvent;
import gdxapp.screens.room.RoomController;
import gdxapp.screens.wall.SurfaceController;
import gdxapp.screens.wall.SurfaceScreen;

public class Object2DEventProcessor extends InputListener {

	private final String TAG = Object2DEventProcessor.class.getName();

	private float xoffset, yoffset, rotation;

	private int oldIndex;

	private Object2D object2D;

	private long lastClickDate = System.currentTimeMillis();
	private int activeButton = -1;

	public Object2DEventProcessor(Object2D object2d) {
		super();
		object2D = object2d;
		object2d.setTouchable(Touchable.enabled);
	}

	@Override
	public boolean handle(Event e) {
		if (!ToolController.getInstance().isSELECT_Objects()) return false;
		if (!(e instanceof InputEvent inputEvent)) return false;
		 // Don't allow group move by dragging if more than one selected
		if (inputEvent.getType() == InputEvent.Type.touchDragged && isGroupSelectionActive()) return false;
		 // Delegate only if object is visible
        if (object2D != null && object2D.getWorldObject() != null && !object2D.getWorldObject().isHidden()) {
            return super.handle(e);// Important: super.handle delegates to the overridden methods like touchDown etc.
        }
        return false;
	}

	@Override
	public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
		setScrollFocus();
		setKeyboardFocus();
		deliverSelectionEvent(event.getStageX(), event.getStageY(), button);
//
		object2D.calculateSurroundingEdges();
		
		saveInitialPointerState(x, y);
		
		if (handleGroupSelection(button)) {
			return true;
		}
		
		if (button == Input.Buttons.LEFT) {
		     activeButton = button;
			if (isDoubleClick()) {
				dressing.events.Event sceneEvent  = new dressing.events.Event(SceneEvent.MODIFY_OBJECT.name(), object2D.getWorldObject());
				EventDriver.getDriver().deliverEvent(sceneEvent);
			} else {
				selectKitchenElementIfApplicable();
				saveOldTransformStates();
			}
		} else if (button == Input.Buttons.RIGHT) {
			toggleSurfaceControllerReference();
		}

		return true;
	}

	@Override
	public void touchUp(InputEvent event, float x, float y, int pointer, int button) {
		activeButton = -1; // reset
		object2D.getWorldObject().setHidden(false);
		object2D.setRotation(rotation);
		if (!ToolController.getInstance().isFREEZE_MOVE() && (button == Input.Buttons.LEFT)
				&& object2D.getWorldObject().isMoveable()) {
			CommandStack command = new CommandStack("change from window submit", "change from window",
					new ArrayList<ChangeCommand>());
			float newRotation = object2D.getWorldObject().getRotation();
			Vector3 newPosition = object2D.getWorldObject().getRealWorldPosition();
			Vector4 newDimention = object2D.getWorldObject().getObjectDimention();
			boolean propertyChanged = false;

			if (newRotation != object2D.getOldRotation()) {
				RotateCommand rotate = new RotateCommand("rotate " + object2D.getWorldObject().getName(), "rotate",
						object2D.getWorldObject(), object2D.getOldRotation(), newRotation);
				command.addCommand(rotate);
				propertyChanged = true;
			}

			if (!newPosition.epsilonEquals(object2D.getOldPosition(), 0.001f)) {
				MoveCommand move = new MoveCommand("move " + object2D.getWorldObject(), "move",
						object2D.getOldPosition(), newPosition, object2D.getWorldObject());
				command.addCommand(move);
				propertyChanged = true;
			}

			if (!newDimention.epsilonEquals(object2D.getOldDimention(), 0.001f)) {
				ScaleCommand scale = new ScaleCommand("move " + object2D.getWorldObject().getName(), "move",
						object2D.getOldDimention(), newDimention, object2D.getWorldObject());
				command.addCommand(scale);
				propertyChanged = true;
			}

			if (propertyChanged) {
				rectifyAndAddCommand(command);
			} else if (!command.getStack().isEmpty()) {
				addCommandToController(command);
			}
		}

		object2D.setZIndex(oldIndex);
	}

	private void rectifyAndAddCommand(CommandStack command) {
		Vector3 oldPos = object2D.getWorldObject().getRealWorldPosition();
		Vector4 oldDim = object2D.getWorldObject().getObjectDimention();
		float oldRot = object2D.getWorldObject().getRotation();
		if (shouldRectify()) {
			((AbstractScreen) Scene.game.getScreen()).getController().rectifyPosition(object2D);
		}

		Vector3 newPos = object2D.getWorldObject().getRealWorldPosition();
		Vector4 newDim = object2D.getWorldObject().getObjectDimention();
		float newRot = object2D.getWorldObject().getRotation();

		if (newRot != oldRot)
			command.addCommand(new RotateCommand("rotate", "rotate", object2D.getWorldObject(), oldRot, newRot));
		if (!newPos.epsilonEquals(oldPos, 0.0000001f))
			command.addCommand(new MoveCommand("move", "move", oldPos, newPos, object2D.getWorldObject()));
		if (!newDim.epsilonEquals(oldDim, 0.0000001f))
			command.addCommand(new ScaleCommand("scale", "scale", oldDim, newDim, object2D.getWorldObject()));

		addCommandToController(command);
	}

	private void addCommandToController(CommandStack command) {
		var controller = ((AbstractScreen) Scene.game.getScreen()).getController();
		if (controller.getChangeCommandController() != null) {
			controller.getChangeCommandController().addCommand(command);
		}
	}

	private boolean shouldRectify() {
		return !ToolController.getInstance().isFREE_MOVE() && !Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT)
				&& !Gdx.input.isKeyPressed(Input.Keys.SHIFT_RIGHT);
	}

	@Override
	public void touchDragged(InputEvent event, float x, float y, int pointer) {
		if (ToolController.getInstance().isFREEZE_MOVE()) {
			return;
		}
		if (activeButton != Input.Buttons.LEFT) {
			return; // block drag for non-left buttons
		}
		float dx = x - xoffset;
		float dy = y - yoffset;

		
		object2D.setRotation(0);

		if (!(Scene.game.getScreen() instanceof SurfaceScreen)) {
			object2D.setZIndex(100);
			if(object2D.getWorldObject().isConstraint())
				ProjectManager.getManager().getCurrentScene().removeConstraint(object2D.getWorldObject());
		}

		Vector2 newPosition = new Vector2(object2D.getX() + dx, object2D.getY() + dy);

		if (object2D.getWorldObject().isMoveable() || Gdx.input.isKeyPressed(Input.Keys.W)
				|| (object2D.getWorldObject() instanceof Wall && ToolController.getInstance().isALLOW_WALL_MOVE())) {
			object2D.setPosition(newPosition.x, newPosition.y);
			
		}

		if (Scene.game.getScreen() instanceof SurfaceScreen) {
			DrawingHelper.drawPositionOnWalls(object2D);
		}
	}

	private Vector2 clampToStageBounds(float x, float y) {
		if (object2D.getStage() == null || object2D.getStage() == RoomController.getInstance().getStage())
			return new Vector2(x, y); // fallback if stage is null

		//make sure only object center is within thz stage boundaries and allow partially visible objects
		float maxX = object2D.getStage().getWidth() - object2D.getWidth()/2.0f;
		float maxY = object2D.getStage().getHeight() - object2D.getHeight()/2.0f;

		
		float clampedX = Math.max(-object2D.getWidth()/2.0f, Math.min(maxX, x));
		float clampedY = Math.max(-object2D.getHeight()/2.0f, Math.min(maxY, y));

		return new Vector2(clampedX, clampedY);
	}

	@Override
	public void enter(InputEvent event, float x, float y, int pointer, Actor fromActor) {

	}

	@Override
	public void exit(InputEvent event, float x, float y, int pointer, Actor toActor) {
		long newClickDate = System.currentTimeMillis();
		if (object2D != null && object2D.getStage() != null && object2D.equals(object2D.getStage().getScrollFocus())
				&& newClickDate - lastClickDate > 2000) {
			object2D.getStage().setScrollFocus(null);
		}
	}

	@Override
	public boolean scrolled(InputEvent event, float x, float y, int amount) {
		if (((AbstractScreen) Scene.game.getScreen()).getController() instanceof SurfaceController
				|| object2D.getWorldObject().getType() == ObjectType.POLY
				|| Gdx.input.isKeyPressed(Input.Keys.CONTROL_LEFT) || Gdx.input.isKeyPressed(Input.Keys.CONTROL_RIGHT)
				|| ToolController.getInstance().isFREEZE_MOVE()) {
			return false;
		}
		if (!object2D.getWorldObject().isMoveable()) {
			return false;
		}
		
		if(Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT)) {
			float oldRotation = object2D.getWorldObject().getRotation();
			float rotation = 15 * (amount / Math.abs(amount));
			object2D.rotateBy(rotation);
			rotation = object2D.getRotation();
			rotation = rotation % 360;
			object2D.getWorldObject().setRotation(rotation);
			object2D.positionChanged();
			float newRotation = object2D.getWorldObject().getRotation();
			RotateCommand rotate = new RotateCommand("rotate " + object2D.getWorldObject().getName(), "rotate",
					object2D.getWorldObject(), oldRotation, newRotation);
			if (((AbstractScreen) Scene.game.getScreen()).getController().getChangeCommandController() != null) {
				((AbstractScreen) Scene.game.getScreen()).getController().getChangeCommandController().addCommand(rotate);
			}
			return true;

		}
		return false;

	}

	public Edge detectEdgeSelection(Vector3 point) {
		Edge selectedEdge = null;
		List<Edge> edges = object2D.calculateEdges();
		for (Edge edge : edges) {
			if (selectedEdge == null) {
				selectedEdge = edge;
				continue;
			}
			float dist1 = MathUtilities.lineToPointDistance(point.x, point.y, edge.getV0().x, edge.getV0().y,
					edge.getV1().x, edge.getV1().y);
			float dist2 = MathUtilities.lineToPointDistance(point.x, point.y, selectedEdge.getV0().x,
					selectedEdge.getV0().y, selectedEdge.getV1().x, selectedEdge.getV1().y);
			if (dist1 < dist2)
				selectedEdge = edge;
		}

		for (Edge edge : edges) {
			edge.setSelected(false);
		}
		if (selectedEdge != null) {
			float dist2 = MathUtilities.lineToPointDistance(point.x, point.y, selectedEdge.getV0().x,
					selectedEdge.getV0().y, selectedEdge.getV1().x, selectedEdge.getV1().y);
			if (dist2 < 100) {
				selectedEdge.setSelected(true);
				return selectedEdge;
			}
		}

		return null;
	}

	@Override
	public boolean keyUp(InputEvent event, int keycode) {
		if (Gdx.input.isKeyPressed(Keys.SHIFT_LEFT) || Gdx.input.isKeyPressed(Keys.SHIFT_RIGHT)) {
			if (keycode == Keys.R) {
				object2D.adjustRotation();
				return true;
			} else if (keycode == Keys.P) {
				object2D.adjustPosition();
				return true;
			}
		} else if (keycode == Keys.FORWARD_DEL) {
			EventDriver.getDriver().deliverEvent(SceneEvent.REMOVE_OBJECT_REQUEST, object2D.getWorldObject());
		}
		return false;
	}

	// handlers of group selection
	public boolean touchDownGrp(InputEvent event, float x, float y, int pointer, int button) {
//		Gdx.app.debug(TAG, "touch down group at X: " + x + "Y: " + y);
		xoffset = x;
		yoffset = y;
		rotation = object2D.getRotation();

		if (Gdx.input.isKeyPressed(Keys.CONTROL_LEFT)) {
			GroupSelection.getInstance().toggleSelection(this.object2D);
		}
		long newClickDate = System.currentTimeMillis();
		if (object2D != null && object2D.getWorldObject() != null && button == Input.Buttons.LEFT) {

			if (newClickDate - lastClickDate < 300) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {

						MesureWidow window = MesureWidow.getInstance();
						window.setKitchenElemtn(object2D.getWorldObject());

						window.show();
					}
				});
			} else {
				if (object2D.getWorldObject() instanceof KitchenElement
						&& ((KitchenElement) object2D.getWorldObject()).getDesignObject() instanceof DesignObject3D) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							GdxPart.setSelectedElement(
									(Space3D) ((KitchenElement) object2D.getWorldObject()).getDesignObject());

						}
					});
				}
				//
				object2D.setOldRotation(object2D.getWorldObject().getRotation());
				object2D.setOldPosition(object2D.getWorldObject().getRealWorldPosition());
				object2D.setOldDimention(object2D.getWorldObject().getObjectDimention());

			}
		}
		if (object2D != null && object2D.getWorldObject() != null && button == Input.Buttons.RIGHT) {
			if (((AbstractScreen) Scene.game.getScreen()).getController() instanceof SurfaceController) {
				SurfaceController controller = ((SurfaceController) ((AbstractScreen) Scene.game.getScreen())
						.getController());
				if (controller.getReference() == object2D) {
					controller.setReference(null);
				} else {
					controller.setReference(object2D);
				}
			}

		}

		lastClickDate = newClickDate;

		return true;
	}

	private boolean isGroupSelectionActive() {
		return GroupSelection.getInstance().getSelection() != null
				&& GroupSelection.getInstance().getSelection().size() > 1;
	}

	private boolean handleGroupSelection(int button) {
		if (Gdx.input.isKeyPressed(Keys.CONTROL_LEFT)) {
			GroupSelection.getInstance().toggleSelection(object2D);
			return true;
		} else {
			GroupSelection.getInstance().clearSelection();
			GroupSelection.getInstance().addObject(object2D);
		}
		return false;
	}

	private void setScrollFocus() {
		if (object2D != null && object2D.getStage() != null) {
			object2D.getStage().setScrollFocus(object2D);
		}
	}

	private void setKeyboardFocus() {
		if (object2D != null && object2D.getStage() != null) {
			object2D.getStage().setKeyboardFocus(object2D);
		}
	}

	private void deliverSelectionEvent(float x, float y, int button) {
		HashMap<String, Object> eventData = new HashMap<>();
		
		AbstractScreen screen = (AbstractScreen) Scene.game.getScreen();
		Vector3 location = screen.toWorldCoordiantes(new Vector3(x,y,0));
		eventData.put("location", location);
		eventData.put("source", object2D.getWorldObject());
		String eventName = button == Input.Buttons.RIGHT ? SceneEvent.OBJECT_RIGHT_CLICK.name()
				: SceneEvent.OBJECT_SELECTED.name();
		EventDriver.getDriver().deliverEvent(new dressing.events.Event(eventName, eventData));
	}

	private void saveInitialPointerState(float x, float y) {
		xoffset = x;
		yoffset = y;
		rotation = object2D.getRotation();
		oldIndex = object2D.getZIndex();
	}

	private boolean isDoubleClick() {
		long now = System.currentTimeMillis();
		boolean doubleClick = (now - lastClickDate) < 250;
		lastClickDate = now;
		return doubleClick;
	}

	private void selectKitchenElementIfApplicable() {
		if (object2D.getWorldObject() instanceof KitchenElement
				&& ((KitchenElement) object2D.getWorldObject()).getDesignObject() instanceof DesignObject3D) {
			Display.getDefault().asyncExec(() -> GdxPart
					.setSelectedElement((Space3D) ((KitchenElement) object2D.getWorldObject()).getDesignObject()));
		}
	}

	private void saveOldTransformStates() {
		object2D.setOldRotation(object2D.getWorldObject().getRotation());
		object2D.setOldPosition(object2D.getWorldObject().getRealWorldPosition());
		object2D.setOldDimention(object2D.getWorldObject().getObjectDimention());
	}

	private void toggleSurfaceControllerReference() {
		if (Scene.game.getScreen() instanceof AbstractScreen screen
				&& screen.getController() instanceof SurfaceController controller) {
			if (controller.getReference() == object2D) {
				controller.setReference(null);
			} else {
				controller.setReference(object2D);
			}
		}
	}

}
