package gdxapp.object3d;

import java.util.ArrayList;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.Touchable;

import dressing.mathutils.EarClipper;
import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import gdxapp.Commun.ActionProcessor;
import gdxapp.assets.AssetsTextures;
import gdxapp.assets.DrawingHelper;
import gdxapp.scenes.Scene;
import gdxapp.screens.room.Room;
import gdxapp.screens.room.RoomController;
import gdxapp.screens.room.RoomScreen;
import gdxapp.screens.wall.Wall2D;
import geometry.CompoundObject;
import geometry.CompoundShape;
import geometry.Shape;

public class GeometryObject2D extends Object2D {
	
	
	private GeometryObject geometryObject;
	private ArrayList<Triangle> triangles;
	private float[] mesh;
	private short[] indices;

	public GeometryObject2D(GeometryObject geometryObject) {
		super(geometryObject);
		this.geometryObject = geometryObject;
		transformVertices();
		setColor(1f,1f,1f,0.5f);
	}
	
	private void transformVertices() {
		if(!geometryObject.getGeometry().getWorldTransform().equals(RoomController.getInstance().getSceneToWorldTransform())) {
			// transforming the border vertices to real world coords
			CompoundShape border = geometryObject.getGeometry().getBorder();
			//border.set
			for(Shape shape: border.getNodes()) {
				for(Vector2 vertex: shape.getVertices()) {
					Vector3 v = new Vector3(vertex, 1.0f).mul(border.getWorldTransform());
					v.mul(RoomController.getInstance().getWorldToSceneTransform());
					vertex.set(v.x, v.y);
				}
			}
			border.setWorldTransform(RoomController.getInstance().getSceneToWorldTransform());
			//updating transform and transforming vertices
			if(geometryObject.getGeometry().hasHole()) {
				CompoundShape hole = geometryObject.getGeometry().getExtrusion();
				for(Shape shape: hole.getNodes()) {
					for(Vector2 vertex: shape.getVertices()) {
						Vector3 v = new Vector3(vertex, 1.0f).mul(hole.getWorldTransform());
						v.mul(RoomController.getInstance().getWorldToSceneTransform());
						vertex.set(v.x, v.y);
					}
				}
				hole.setWorldTransform(RoomController.getInstance().getSceneToWorldTransform());
			}
			geometryObject.getGeometry().setWorldTransform(RoomController.getInstance().getSceneToWorldTransform());
		}
	}
	
	
	public void calculateVerticesData() {
		if (((GeometryObject) getWorldObject()).getGeometry() != null) {
			CompoundObject geometry = ((GeometryObject) getWorldObject()).getGeometry();
			Vector2[] boundaries = MathUtilities.getBoundariesPlane(geometry.getBorder().getVertices());
			float width = boundaries[1].x - boundaries[0].x;
			float height = boundaries[1].y - boundaries[0].y;
			setWidth(width);
			setHeight(height);
			setPosition(boundaries[0].x, boundaries[0].y);
			Vector2 origin = boundaries[0].cpy().add(boundaries[1]).scl(0.5f);
			setOrigin(origin.x, origin.y);
			calculateTriangles();
			indices = new short[triangles.size() * 3];
			float[] uvRange = { textureRegion().getU(), textureRegion().getU2(), textureRegion().getV(),
					textureRegion().getV2() };
			mesh = MathUtilities.indexMesh(triangles, this.indices, uvRange,
					Color.toFloatBits(getColor().r, getColor().g, getColor().b, getColor().a));
//			mesh = MathUtilities.indexMesh(triangles, this.indices, uvRange,
//					Color.toFloatBits(Color.GREEN.r, Color.GREEN.g, Color.GREEN.b, getColor().a));

		}
	}
	
	private void drawAsPoly(Batch batch) {
		if (mesh == null)
			calculateVerticesData();
		
		var color = getColor();
		if(getWorldObject().isSelected())
			color = Color.GREEN;
		batch.end();
		
		var oldColor = RoomController.getPolyBatch().getColor();
		
		RoomController.getPolyBatch().setProjectionMatrix(batch.getProjectionMatrix().cpy());
		RoomController.getPolyBatch().setBlendFunction(GL20.GL_BLEND_SRC_ALPHA, GL20.GL_ONE_MINUS_SRC_ALPHA);
		RoomController.getPolyBatch().enableBlending();
		RoomController.getPolyBatch().begin();
		RoomController.getPolyBatch().setColor(color);

		RoomController.getPolyBatch().draw(textureRegion().getTexture(), mesh, 0, mesh.length, indices, 0,
				indices.length);
		RoomController.getPolyBatch().end();
		RoomController.getPolyBatch().setColor(oldColor);
		batch.begin();
	}

	public void calculateTriangles() {
		CompoundObject geometry = ((GeometryObject) getWorldObject()).getGeometry();
		
		ArrayList<Vector2> border = geometry.getBorder().getVertices();
		ArrayList<Vector2> extrusion = geometry.getExtrusionVertices();
		MathUtilities.setWinding(border, 1);
		MathUtilities.setWinding(extrusion, -1);
		this.triangles = EarClipper.triangulate(border, extrusion);
	}
	
	//exclude walls
	public boolean contains(Object2D other)
	{
		if(other instanceof Wall2D)
			return false;
		if(!other.overlap(this))
			return false;
		for(Vector2 vertex: other.getRotatedVertices()) {
			if(!occupyPoint(vertex))
				return false;
		}
		return true;
	}
	
	public boolean occupyPoint(Vector2 point) {
		for(Triangle triangle: getTriangles()) {
			if(triangle.contains(point))
				return true;
		}
		return false;
	}
	
//overriden methods
	
	
	
	
	
	@Override
	public synchronized void calculateRotatedVertices() {
		rotatedVertices.clear();
		if (geometryObject.getGeometry().getBorder() != null) {
			rotatedVertices.addAll(((GeometryObject) getWorldObject()).getGeometry().getBorder().getVertices());
		}
	}


	@Override
	public void adjustTextureAndColors() {
		this.textureRegion = geometryObject.getGeometry().getBorder().getTexture().getPath();
	}



	@Override
	protected TextureRegion textureRegion() {
		TextureRegion region;
		try {
			region = AssetsTextures.getInstance().getTextureRegion(((GeometryObject) worldObject).getGeometry().getBorder().getTexture()
					.getPath());
		}catch (Exception e) {
			region = AssetsTextures.getInstance().getDefaultTextureRegion();
		}
		 
		return region;
	}

	@Override
	public void draw(Batch batch, float parentAlpha) {
		if (!getWorldObject().isHidden()) {
			if (Scene.game.getScreen() == RoomScreen.getInstance()) {
				drawTopView(batch);
			} else {
				drawWallView(batch);
			}
		} else {
			remove();
		}
	}

	@Override
	protected void positionChanged() {
		super.positionChanged();
		calculateVerticesData();
	}

	@Override
	public <T extends WorldObject> T getWorldObject() {
		return (T) geometryObject;
	}

	@Override
	public Actor hit(float x, float y, boolean touchable) {
		if (touchable && this.getTouchable() != Touchable.enabled) return null;
		if (!isVisible()) return null;
		Actor hit = null;
		Vector2 point = new Vector2(x,y).add(getX(), getY());
		for(Triangle triangle: triangles) {
			if(triangle.contains(point)) {
				hit = this;
				break;
			}
		}
		return hit;
	}

	@Override
	protected void drawWallView(Batch batch) {
		if (!worldObject.isHidden()) {
			calculateRotatedVertices();
			drawBorder(batch);
		}

		if (worldObject.isSelected || getWorldObject().isDrawInnerQuotations()
				|| ActionProcessor.getProcessor().isShowInnerQuotations()) {
			DrawingHelper.drawQuotations(this, batch);
		}
	}

	@Override
	protected void drawTopView(Batch batch) {
		if (getWorldObject().isSelected()) {
			batch.end();
			DrawingHelper.drawDistanceLines(this, batch);
			batch.begin();
		}
		if (!worldObject.isHidden()) {
			drawAsPoly(batch);
		}
	}
	
	@Override
	public ArrayList<Edge> calculateEdges() {
		edges.clear();
		edges.addAll(geometryObject.getGeometry().getBorder().getEdges());		
		return edges;
	}
	
	
	public void geometryChanged() {
		geometryObject.getGeometry().getBorder().calculateProperties();
		setWidth(geometryObject.getGeometry().getBorder().getWidth());
		setHeight(geometryObject.getGeometry().getBorder().getHeight());
		setPosition(geometryObject.getGeometry().getBorder().getX(), geometryObject.getGeometry().getBorder().getY());
		getWorldObject().setRequireRefrech(true);
	}
	
//getters and setters



	public ArrayList<Triangle> getTriangles() {
		if(triangles == null) {
			calculateTriangles();
		}
		return triangles;
	}

	public void setTriangles(ArrayList<Triangle> triangles) {
		this.triangles = triangles;
	}
	


	public GeometryObject getGeometryObject() {
		return geometryObject;
	}

	public void setGeometryObject(GeometryObject geometryObject) {
		this.geometryObject = geometryObject;
	}
	
	
}
