package gdxapp.Commun;

import java.util.ArrayList;
import java.util.List;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import gdxapp.assets.AssetFont;
import gdxapp.assets.DrawingHelper;
import gdxapp.object3d.FullScreenActor;
import gdxapp.ui.CursorProvider;
import com.badlogic.gdx.scenes.scene2d.Event;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.InputListener;


public class MeasuresTaker extends FullScreenActor {
		
	private Vector2[] trajectory;
	private ArrayList<Vector2[]> measures = new ArrayList<Vector2[]>();
	private int index;
	private int capacity = 4;
	private ShapeRenderer shapeRenderer;
	private static MeasuresTaker instance;
	private Matrix4 stageToWorldTransform;
	private Vector2 tmpLocation;
	private boolean draw = false;
	private float scale = 10000;
	private List<Vector2> referencePoints;
	private Vector2 hRefPnt;
	private Vector2 vRefPnt;
	private Vector2 mouseOver;
	private Vector2 mouseLocation = new Vector2();
	
	
	public MeasuresTaker() {
		super();
		shapeRenderer = new ShapeRenderer();
		addListener(new MeasureTakerEventProcessor());
	}
	
	public static MeasuresTaker getInstance() {
		if(instance == null)
			instance = new MeasuresTaker();
		return instance;
	}
	
	
	@Override
	public void draw(Batch batch, float parentAlpha) {
		if(!draw)
			return;
		batch.flush();
		shapeRenderer.setProjectionMatrix(camera.combined);
		shapeRenderer.setColor(Color.GREEN);
		shapeRenderer.begin(ShapeType.Filled);
		for(var measure: measures)
			drawMeasure(measure);
		
		float pixel = getPixelWidth();

		for(int i =0; i < index; i++) {
			shapeRenderer.circle(trajectory[i].x, trajectory[i].y, pixel * 3);
			if(index > i+1) {
				shapeRenderer.rectLine(trajectory[i], trajectory[i+1], pixel * 3);
			}
			
		}

		if(tmpLocation != null) {
			shapeRenderer.setColor(Color.RED);
			shapeRenderer.rectLine(trajectory[index-1], tmpLocation, pixel * 2);
		}
		
		if(referencePoints != null) {
			shapeRenderer.setColor(Color.YELLOW);
			Vector2 tmp = new Vector2(pixel , pixel).scl(5);
			for(Vector2 point: referencePoints) {
				shapeRenderer.rectLine(point.cpy().add(tmp.cpy().scl(1,1) ), point.cpy().add(tmp.cpy().scl(-1,-1)), 2 * pixel);
				shapeRenderer.rectLine(point.cpy().add(tmp.cpy().scl(-1,1)), point.cpy().add(tmp.cpy().scl(1,-1)), 2 * pixel);
			}
		}
		shapeRenderer.setColor(Color.BLUE);
		if(mouseOver != null) {
			shapeRenderer.circle(mouseOver.x, mouseOver.y, 6 * pixel);
		}
		
		if(hRefPnt != null) {
			Vector2 tmpPoint = alignPoint(mouseLocation);
			DrawingHelper.drawCutLine(tmpPoint, hRefPnt, batch, Color.BLUE, pixel, 12 * pixel, 4 * pixel);
			
		}
		if(vRefPnt != null) {
			Vector2 tmpPoint = alignPoint(mouseLocation);
			DrawingHelper.drawCutLine(tmpPoint, vRefPnt, batch, Color.BLUE, pixel, 12 * pixel, 4 * pixel);
		}
		
		shapeRenderer.end();
		
		drawQuotations(batch);
	}
	
	public void drawMeasure(Vector2[] measure) {
		float pixel = getPixelWidth();
		shapeRenderer.setColor(Color.GREEN);
		for(int i =0; i < measure.length; i++) {
			shapeRenderer.circle(measure[i].x, measure[i].y, pixel * 3);
			if(measure.length > i+1) {
				shapeRenderer.rectLine(measure[i], measure[i+1], pixel * 1);
			}
		}
	}
	
	public void drawMeasureQuotation(Batch batch, Vector2[] measure) {
		if(measure.length < 2)
			return;
		for(int i = 0; i < measure.length - 1; i++) {
			Vector2 current = measure[i];
			Vector2 next = measure[i+1];
			Vector2 center = current.cpy().add(next).scl(0.5f);
			float length = next.dst(current)/scale;
			AssetFont.getInstance().getSmallFont().drawTransformed(batch, String.format("%d", Math.round(1000 * length)), center.x, center.y, 
					getPixelWidth() * 24,0, 0, 0, Color.BLACK);
		}
	}
	
	private void drawQuotations(Batch batch) {
		for(var measure: measures) {
			drawMeasureQuotation(batch, measure);
		}
		if(index < 2)
			return;
		for(int i = 0; i < index - 1; i++) {
			Vector2 current = trajectory[i];
			Vector2 next = trajectory[i+1];
			Vector2 center = current.cpy().add(next).scl(0.5f);
			float length = next.dst(current)/scale;
			AssetFont.getInstance().getSmallFont().drawTransformed(batch, String.format("%d", Math.round(1000 * length)), center.x, center.y, 
					getPixelWidth() * 24,0, 0, 0, Color.BLACK);
		}
	}
	
	private Vector2 alignPoint(Vector2 point) {
		if(vRefPnt == null && hRefPnt == null)
			return point.cpy();
		Vector2 aligned = new Vector2(point);

		if(hRefPnt != null) {
			aligned.x = hRefPnt.x;
		}
		if(vRefPnt != null) {
			aligned.y = vRefPnt.y;
		}
		
		return aligned;
	}
	
	
	
	public void begin() {
		camera = (OrthographicCamera) getStage().getCamera();
		onCameraChanged();
		index = 0;
		capacity = 4;
		trajectory = new Vector2[capacity];
		tmpLocation = null;
		draw = true;
		hRefPnt = null;
		vRefPnt = null;
		getStage().setKeyboardFocus(this);

	}
	
	public void end() {
		draw = false;
		CursorProvider.getInstance().setDefault();
		remove();
	}
	
	public void appendPoint(Vector2 location) {
		if(index > capacity-1)
			extend();
		
		trajectory[index++] = location;
 	}
	
	
	private void extend() {
		capacity *= 2;
		Vector2[] newBuffer = new Vector2[capacity];
		for(int i = 0; i < index; i++)
			newBuffer[i] = this.trajectory[i];
		this.trajectory = newBuffer;
	}
	
	public void setTmpLocation(Vector2 location) {
		if(index > 0) {
			tmpLocation = location;
		}
	}

	public void setReferencePoints(List<Vector2> referencePoints) {
		this.referencePoints = referencePoints;
	}
	
	
	//this function is adapative to the camera zoom
	private void findReferencePoint() {
		hRefPnt = null;
		vRefPnt = null;
		mouseOver = null;
		float closestDstX = Float.MAX_VALUE;
		float closestDstY = Float.MAX_VALUE;
		Vector2 closestPointX = null;
		Vector2 closestPointY = null;
		ArrayList<Vector2> copy = new ArrayList<Vector2>(referencePoints);
		for(int i = 0; i < index; i++)
			copy.add(trajectory[i]);
		for(Vector2 point: copy) {
			Vector2 translation = point.cpy().sub(mouseLocation);
			float dstX = Math.abs(translation.x);
			float dstY = Math.abs(translation.y);
			if(Math.abs(dstX - closestDstX) < 1e-6f) { // almost equals
				if(point.dst(mouseLocation) < closestPointX.dst(mouseLocation)) {
					closestDstX = dstX;
					closestPointX = point;
				}
			}else if(dstX < closestDstX) {
				closestDstX = dstX;
				closestPointX = point;
			}
			
			if(Math.abs(dstY - closestDstY) < 1e-6f) { // almost equals
				if(point.dst(mouseLocation) < closestPointY.dst(mouseLocation)) {
					closestDstY = dstY;
					closestPointY = point;
				}
			}else if(dstY < closestDstY) {
				closestDstY = dstY;
				closestPointY = point;
			}
			
			
		}
		
		if(closestDstX < 20 * getPixelWidth()) {
			hRefPnt = closestPointX;
			if(hRefPnt.dst(mouseLocation) < 10 * getPixelWidth())
				mouseOver = hRefPnt;
		}
		if(closestDstY < 20 * getPixelWidth()) {
			vRefPnt = closestPointY;
			if(vRefPnt.dst(mouseLocation) < 10 * getPixelWidth()) {
				if(mouseOver == null) {
					mouseOver = vRefPnt;
				}else {
					mouseOver = mouseOver.dst(mouseLocation)<vRefPnt.dst(mouseLocation)?mouseOver:vRefPnt;
				}
			}
		}
	}



	class MeasureTakerEventProcessor extends InputListener{

		
		int selectedIndex = -1;
		
		public boolean selectAt(Vector2 location) {
			selectedIndex = -1;
			float closestDst = Float.MAX_VALUE;
			float thresh = 0.01f * scale;
			for(int i = 0 ; i < index; i++) {
				float dst = location.dst(trajectory[i]);
				if( dst < thresh && dst < closestDst) {
					closestDst = dst;
					selectedIndex = i;
				}
			}
			//if none found search among reference points
			if(selectedIndex == 0) {
				
			}
			return selectedIndex >= 0;
		}
		
		@Override
		public boolean handle(Event e) {
			// TODO Auto-generated method stub
			return super.handle(e);
		}

		@Override
		public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
			if(selectedIndex < 0) {
				Vector2 point = null;
				if(Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT)) {
					point = mouseLocation.cpy();
				}else {
					if(mouseOver != null) {
						point = mouseOver.cpy();
					}else {
						point = alignPoint(mouseLocation);
					}
				}
				appendPoint(point);
			}
			return true;
		}


		@Override
		public void touchDragged(InputEvent event, float x, float y, int pointer) {
			// TODO Auto-generated method stub
			super.touchDragged(event, x, y, pointer);
		}

		@Override
		public boolean mouseMoved(InputEvent event, float x, float y) {
			mouseLocation.set( localToStageCoordinates(new Vector2(x, y)));
			findReferencePoint();
			//
			
			if( mouseOver != null) {
				CursorProvider.getInstance().useHand();
			}else {
				CursorProvider.getInstance().useRuler();
			}
			
			if(selectedIndex < 0)
				setTmpLocation(mouseLocation);
			return true;
		}


		@Override
		public boolean keyUp(InputEvent event, int keycode) {
			if(keycode == Input.Keys.ESCAPE) {
				end();
				return true;
			}else if(keycode == Input.Keys.ENTER) {
				if(index > 0) {
					Vector2[] points = new Vector2[index];
					for(int i = 0; i < index; i++)
						points[i] = trajectory[i];
					measures.add(points);
					if(shapeRenderer.isDrawing())
						shapeRenderer.end();
					begin();
				}
				return true;
			}
			return false;
		}
	}













	






}
