package gdx.app.camera;

import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.math.Vector3;
import dressing.events.Event;
import dressing.events.EventDriver;
import gdxapp.scenes.SceneEvent;

public class PerspectiveCameraController {
	
	protected Camera camera;
	
	protected Vector3 orbitCenter;
	
	public PerspectiveCameraController() {
		orbitCenter = new Vector3();
	}
	
	public void reset() {
		camera.viewportWidth = 1;
		camera.viewportHeight = 1;
		camera.position.set(4,0,10);
		camera.up.set(0,1,0);
		camera.lookAt(4, 0, 4);
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void moveForward(float amount) {
		amount *= orbitCenter.dst(camera.position)/ camera.far;
		amount = Math.signum(amount) * Math.max(0.1f, Math.abs(amount));
		camera.position.add(camera.direction.cpy().scl(1,0,1).nor().scl(amount));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void moveLaterally(float amount) {
		amount *= orbitCenter.dst(camera.position)/ camera.far;
		camera.position.add(camera.up.cpy().nor().crs(camera.direction.cpy().nor()).scl(amount));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void moveVertically(float amount) {
		amount *= orbitCenter.dst(camera.position)/ camera.far;
		camera.position.add(new Vector3(0,1,0).scl(amount));
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void lookLaterally(float degree) {
		camera.up.set(0,1,0);
		camera.rotate(camera.up, degree );
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void tild(float degree) {
		
		float det = camera.direction.cpy().nor().scl(1, 0, 1).len();
		float pitch = (float) Math.toDegrees(Math.atan(camera.direction.y/ det));
		float sum = pitch + degree;
		float clampedValue = Math.min(60, Math.max(-60, sum));
		float rotation = clampedValue - pitch;
		
		if(rotation * rotation > 0.01f) {
			camera.rotate(camera.direction.cpy().crs(Vector3.Y), rotation);
			camera.update();
			EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));
		}
		

	}
	
	public void adjustFiledOfView(float amount) {
		camera.viewportWidth *= amount;
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}

	public Camera getCamera() {
		return camera;
	}

	public void setCamera(Camera camera) {
		this.camera = camera;
	}
	
	public void orbit(float amount) {
		Vector3 cc = camera.position.cpy().sub(orbitCenter);
		cc.rotate(Vector3.Y, amount/cc.len());
		Vector3 position = orbitCenter.cpy().add(cc);	
		camera.position.set(position);
		camera.direction.rotate(Vector3.Y, amount/cc.len());
		camera.up.set(Vector3.Y);
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void advance(float amount) {
		camera.position.add(camera.direction.cpy().scl(amount));
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	public void orbit(float y, float z) {
		Vector3 cc = camera.position.cpy().sub(orbitCenter);
		cc.rotate(Vector3.Y, y);
		cc.rotate(Vector3.Z, z);
		Vector3 position = orbitCenter.cpy().add(cc);
		camera.position.set(position);
		camera.up.set(0,1,0);
		camera.lookAt(orbitCenter);
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}

	public Vector3 getOrbitCenter() {
		return orbitCenter;
	}

	public void setOrbitCenter(Vector3 orbitCenter) {
		this.orbitCenter = orbitCenter;
	}
	

	
	public void handleKeyDown() {}
	
	public void update() {
		handleKeyDown();
		camera.update();
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.CAMERA_MOVED.name(), null));

	}
	
	
}
