package gdxapp.screens.ObjectEditor;

import java.awt.Dimension;
import java.awt.Point;
import java.io.IOException;
import java.util.ArrayList;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.InputMultiplexer;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.PerspectiveCamera;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelBatch;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.g3d.Renderable;
import com.badlogic.gdx.graphics.g3d.Shader;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.graphics.g3d.model.MeshPart;
import com.badlogic.gdx.graphics.g3d.utils.DefaultShaderProvider;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.g3d.utils.RenderContext;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Stage;
import com.badlogic.gdx.utils.GdxRuntimeException;
import com.badlogic.gdx.utils.viewport.ScreenViewport;

import dressing.config.Perspective;
import dressing.config.UserPreference;
import dressing.config.WorkspaceConfiguration;
import dressing.events.Event;
import dressing.events.EventDriver;
import dressing.events.EventHandler;
import dressing.mathutils.Vector4;
import dressing.model.ProjectManager;
import dressing.ui.RemoveObjectCommand;
import dressing.ui.parts.GdxPart;
import gdxapp.object3d.GeometryObject;
import gdxapp.object3d.ModelInfo;
import gdxapp.object3d.Object3D;
import gdxapp.object3d.ObjectModel;
import gdxapp.object3d.PolygonBuilder;
import gdxapp.object3d.WorldObject;
import gdxapp.object3d.WorldObjectFactory;
import gdxapp.scenes.SceneEvent;
import gdxapp.screens.room3d.CameraController;
import gdxapp.screens.room3d.MouseOnlyCameraController;
import gdxapp.screens.room3d.Room3DController;
import gdxapp.shaders.DiffuseShader;

public class ObjectEditorController implements EventHandler {

	
	private WorldObject object;
	
	private final Matrix4 transform = new Matrix4();
	
	private Stage uiStage;
	
	ObjectEditorUILayer uiLayer;

	private final ArrayList<Object3D> sceneObjects = new ArrayList<Object3D>();

	private ArrayList<ModelInstance> grid;

	private ModelBuilder modelBuilder;

	private DiffuseShader shader;
	
	private ShaderProgram program;

	private ModelBatch batch;

	private Camera camera;

	private TargetCamera cameraController;
	
	ObjectEditorInputHandler inputHandler;
	
	private static Object3D selection;
		
	public ObjectEditorController() {
		modelBuilder = new ModelBuilder();

		createGrid();
		setUpCamera();
		prepareBatch();
		prepareUILayer();
		prepareInput();
				
		subscribe(SceneEvent.OBJECT_CREATED.name(), SceneEvent.REMOVE_OBJECT_REQUEST.name(),
				SceneEvent.INSERT_3D_MODEL.name());
	}
	

	private void prepareInput() {
		inputHandler = new ObjectEditorInputHandler(uiStage, cameraController, sceneObjects);
		inputHandler.processInput();
	}

	private void prepareUILayer() {
		uiStage = new Stage(new ScreenViewport());
		uiLayer = new ObjectEditorUILayer();
		uiLayer.setFillParent(true);
		uiLayer.setDebug(true, false);
		uiStage.addActor(uiLayer);
	}


	private void createGrid() {
		int attr = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal | VertexAttributes.Usage.TextureCoordinates;
		Material largeGridMtl = new Material();
		Color lightGray = new Color(0.65f, 0.65f, 0.65f, 1.0f);
		largeGridMtl.set(ColorAttribute.createDiffuse(lightGray));
		
		Material smallGridMtl = new Material();
		Color darkGrayGray = new Color(0.4f, 0.4f, 0.4f, 1.0f);
		smallGridMtl.set(ColorAttribute.createDiffuse(darkGrayGray));
		
		Model largeGridModel = modelBuilder.createLineGrid(100, 100, 1, 1, largeGridMtl, attr);
		Model smallGridModel = modelBuilder.createLineGrid(1000, 1000, 0.1f, 0.1f, smallGridMtl, attr);
		
		Model plane = modelBuilder.createBox(100, 0.01f, 100, new Material(ColorAttribute.createDiffuse(0.1f,0.1f,0.1f,0.8f)), attr);
		ModelInstance planeInstance = new ModelInstance(plane);
		
		planeInstance.transform.setToTranslation(0.0f,-0.05f, 0.0f);
			
		this.grid = new ArrayList<ModelInstance>();
		this.grid.add(planeInstance);
		this.grid.add(new ModelInstance(largeGridModel));
		this.grid.add(new ModelInstance(smallGridModel));
	}

	private void setUpCamera() {
		camera = new PerspectiveCamera(67, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
		camera.near = 0.1f;
		camera.far = 30.0f;
		camera.up.set(0,1,0);
		camera.position.set(0,5,10);
		camera.lookAt(0, 0, 0);
		camera.update();
		cameraController = new TargetCamera(camera, new Vector3(0,0,0));
	}
	public void render() {
		cameraController.update();
		Point point = GdxPart.scrollPane.getScrollPosition();
		Dimension dims = GdxPart.scrollPane.getSize();
		uiLayer.pad(point.y, point.x, Gdx.graphics.getHeight() - point.y - dims.height, Gdx.graphics.getWidth() -point.x - dims.width);
		uiLayer.invalidateHierarchy();
		uiLayer.act(0.017f);
		batch.begin(cameraController.getCurrentCamera());
		batch.getRenderContext().setDepthTest(GL20.GL_LESS);

		//batch.render(grid);
		for (int i = 0; i < sceneObjects.size(); i++) {
			batch.render(sceneObjects.get(i));
		}
		
		if(selection != null) {
			selection.drawBoundaries(camera.combined, Vector3.X, 2.0f);
		}
		batch.end();
		batch.begin(camera);
		inputHandler.draw(batch);
		batch.end();

		uiStage.draw();
		

	}

	private void prepareBatch() {
		shader = new DiffuseShader();
		shader.init();
				
		
		//batch = new ModelBatch();
		batch = new ModelBatch(new DefaultShaderProvider() {
			@Override
			protected Shader createShader(Renderable renderable) {
				return shader;
			}

			@Override
			public Shader getShader(Renderable renderable) {
				return shader;
			}
		});
	}

	private void createUI() {
		uiStage = new Stage(new ScreenViewport());
	}
		
	public WorldObject getObject() {
		return object;
	}

	public void setObject(WorldObject object) {
		this.object = object;
		if(object.getModel() == null || object.isRequireRefrech()) {
    		Model  model =  PolygonBuilder.createModel(((GeometryObject) object).getGeometry(), object.getRealWorldPosition().y - object.getRealWorldDimension().y/2, object.getRealWorldPosition().y + object.getRealWorldDimension().y/2);
    		object.setModel(new ObjectModel(model, null));
		}
		Object3D object3D = new Object3D(object.getModel().getModel());
		calculateTransform(object3D);
		object3D.transform.mul(transform);
		object3D.calculateProperties();
		sceneObjects.add(object3D);
		cameraController.setOrbitCenter(object3D.getPosition());
	}

	private void calculateTransform(Object3D object3d) {
		Vector3 translation = object3d.getPosition();
		Quaternion rotation = new Quaternion();
		object3d.transform.getRotation(rotation, true);
		transform.setToTranslation(translation.cpy().scl(-1));
	}

	@Override
	public void handle(Event event) {
		SceneEvent sceneEvent = SceneEvent.valueOf(event.getTopic());
		switch (sceneEvent) {
		case OBJECT_CREATED:
			Object3D instance = (Object3D) event.getData();
			sceneObjects.add(instance);
			break;
		case REMOVE_OBJECT_REQUEST:
			removeObject((Object3D)event.getData());
			break;
		case INSERT_3D_MODEL:
			insertModelInstance(event);
			break;
		}
		
	}
	
	public static void setSelection(Object3D selection) {
		ObjectEditorController.selection = selection;
		EventDriver.getDriver().deliverEvent(new Event(SceneEvent.OBJECT_SELECTED.name(), selection));
	}

	private void insertModelInstance(Event event) {
		Object model = event.getData();
		if(model instanceof Model) {
			Object3D instance = new Object3D((Model)model);
			sceneObjects.add(instance);
		}
	}

	private void removeObject(Object3D object) {
		sceneObjects.remove(object);
		if(selection == object)
			selection = null;
	}

	public void addObjectsToScene() {
		for(Object3D object: sceneObjects) {
			try {
				object.transform.mul(transform.cpy().inv());
				object.calculateProperties();
				WorldObject worldObject = WorldObjectFactory.create(object);
				ProjectManager.getManager().getCurrentScene().addActor(worldObject, false);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}
}
