package test;

import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.g3d.*;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.environment.DirectionalLight;
import com.badlogic.gdx.graphics.g3d.utils.CameraInputController;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Intersector;
import com.badlogic.gdx.math.Plane;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.collision.BoundingBox;

import java.util.ArrayList;
import java.util.List;

public class SnapGridExtrudeDemo extends ApplicationAdapter {
    private PerspectiveCamera camera;
    private CameraInputController camController;

    private ShapeRenderer shapeRenderer;   // for grid
    private ModelBatch modelBatch;
    private ModelBuilder modelBuilder;

    private Plane groundPlane;
    private Vector3 tmpIntersection = new Vector3();

    private Vector3 firstPoint = null;
    private Vector3 secondPoint = null;

    private int gridSize = 20;
    private float cellSize = 1f;

    private Environment environment;
    private List<ModelInstance> instances = new ArrayList<>();
    private Model previewModel;
    private ModelInstance previewInstance;

    // Extrusion state
    private boolean extruding = false;
    private Vector3 extrudeBaseMin, extrudeBaseMax;
    private Model extrudePreviewModel;
    private ModelInstance extrudePreviewInstance;
    private float currentHeight = 0f;

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();
        modelBatch = new ModelBatch();
        modelBuilder = new ModelBuilder();

        // Camera
        camera = new PerspectiveCamera(67, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.position.set(10f, 10f, 10f);
        camera.lookAt(0, 0, 0);
        camera.near = 0.1f;
        camera.far = 100f;
        camera.update();

        camController = new CameraInputController(camera);
        Gdx.input.setInputProcessor(camController);

        groundPlane = new Plane(new Vector3(0, 1, 0), 0);

        // Lighting
        environment = new Environment();
        environment.set(new ColorAttribute(ColorAttribute.AmbientLight, 0.8f, 0.8f, 0.8f, 1f));
        environment.add(new DirectionalLight().set(Color.WHITE, new Vector3(-1f, -0.8f, -0.2f)));
    }

    @Override
    public void render() {
        camController.update();
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT);

        // --- Draw Grid ---
        shapeRenderer.setProjectionMatrix(camera.combined);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.setColor(Color.LIGHT_GRAY);
        for (int i = -gridSize; i <= gridSize; i++) {
            shapeRenderer.line(i * cellSize, 0, -gridSize * cellSize, i * cellSize, 0, gridSize * cellSize);
            shapeRenderer.line(-gridSize * cellSize, 0, i * cellSize, gridSize * cellSize, 0, i * cellSize);
        }
        shapeRenderer.end();

        // --- Draw 3D Models ---
        modelBatch.begin(camera);
        for (ModelInstance instance : instances) {
            modelBatch.render(instance, environment);
        }
        if (previewInstance != null) {
            modelBatch.render(previewInstance, environment);
        }
        if (extrudePreviewInstance != null) {
            modelBatch.render(extrudePreviewInstance, environment);
        }
        modelBatch.end();

        handleInput();
    }

    private void handleInput() {
        if (Gdx.input.justTouched() && !extruding) {
            Vector3 worldCoords = getMouseIntersection();
            if (worldCoords != null) {
                float snappedX = Math.round(worldCoords.x / cellSize) * cellSize;
                float snappedZ = Math.round(worldCoords.z / cellSize) * cellSize;
                Vector3 snapped = new Vector3(snappedX, 0, snappedZ);

                if (firstPoint == null) {
                    firstPoint = snapped;
                } else {
                    secondPoint = snapped;
                }
            }
        }

        // ENTER → finalize rectangle or extrusion
        if (Gdx.input.isKeyJustPressed(Input.Keys.ENTER)) {
            if (extruding) {
                finalizeExtrusion();
            } else if (firstPoint != null && secondPoint != null) {
                ModelInstance rect = createRectangle(firstPoint, secondPoint, Color.SKY);
                instances.add(rect);
                disposePreview();
                firstPoint = null;
                secondPoint = null;
            }
        }

        // E → start extrusion mode on last rectangle
        if (Gdx.input.isKeyJustPressed(Input.Keys.E) && !extruding) {
            if (!instances.isEmpty()) {
                startExtrusion(instances.remove(instances.size() - 1));
            }
        }

        // ESC → cancel extrusion
        if (Gdx.input.isKeyJustPressed(Input.Keys.ESCAPE) && extruding) {
            cancelExtrusion();
        }

        // SPACE → clear everything
        if (Gdx.input.isKeyJustPressed(Input.Keys.SPACE)) {
            for (ModelInstance inst : instances) inst.model.dispose();
            instances.clear();
            disposePreview();
            cancelExtrusion();
            firstPoint = null;
            secondPoint = null;
        }

        // Live preview rectangle
        if (!extruding && firstPoint != null && secondPoint != null) {
            disposePreview();
            previewModel = createRectangleModel(firstPoint, secondPoint, Color.GREEN, 0.5f);
            previewInstance = new ModelInstance(previewModel);
        }

        // Update extrusion preview
        if (extruding) {
            float dy = -Gdx.input.getDeltaY() * 0.05f; // mouse drag up/down
            if (dy != 0) {
                currentHeight = Math.max(0.1f, currentHeight + dy);
                updateExtrusionPreview();
            }
        }
    }

    private Vector3 getMouseIntersection() {
        com.badlogic.gdx.math.collision.Ray ray = camera.getPickRay(Gdx.input.getX(), Gdx.input.getY());
        if (Intersector.intersectRayPlane(ray, groundPlane, tmpIntersection)) {
            return tmpIntersection.cpy();
        }
        return null;
    }

    private ModelInstance createRectangle(Vector3 a, Vector3 b, Color color) {
        Model model = createRectangleModel(a, b, color, 1f);
        return new ModelInstance(model);
    }

    private Model createRectangleModel(Vector3 a, Vector3 b, Color color, float alpha) {
        float minX = Math.min(a.x, b.x);
        float maxX = Math.max(a.x, b.x);
        float minZ = Math.min(a.z, b.z);
        float maxZ = Math.max(a.z, b.z);

        return modelBuilder.createRect(
                minX, 0, minZ,
                maxX, 0, minZ,
                maxX, 0, maxZ,
                minX, 0, maxZ,
                0, 1, 0,
                new Material(ColorAttribute.createDiffuse(color.r, color.g, color.b, alpha)),
                VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal | VertexAttributes.Usage.ColorUnpacked
        );
    }

    private void startExtrusion(ModelInstance rect) {
        BoundingBox bounds = new BoundingBox();
        rect.calculateBoundingBox(bounds);
        extrudeBaseMin = bounds.min.cpy();
        extrudeBaseMax = bounds.max.cpy();
        currentHeight = 0.1f; // start small
        updateExtrusionPreview();
        rect.model.dispose(); // dispose rectangle model (we replace it)
        extruding = true;
    }

    private void updateExtrusionPreview() {
        if (extrudePreviewModel != null) extrudePreviewModel.dispose();

        float width = extrudeBaseMax.x - extrudeBaseMin.x;
        float depth = extrudeBaseMax.z - extrudeBaseMin.z;
        float cx = (extrudeBaseMin.x + extrudeBaseMax.x) / 2f;
        float cz = (extrudeBaseMin.z + extrudeBaseMax.z) / 2f;

        extrudePreviewModel = modelBuilder.createBox(
                width, currentHeight, depth,
                new Material(ColorAttribute.createDiffuse(Color.ORANGE)),
                VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal
        );

        extrudePreviewInstance = new ModelInstance(extrudePreviewModel);
        extrudePreviewInstance.transform.setToTranslation(cx, currentHeight / 2f, cz);
    }

    private void finalizeExtrusion() {
        instances.add(extrudePreviewInstance);
        extrudePreviewModel = null;
        extrudePreviewInstance = null;
        extruding = false;
    }

    private void cancelExtrusion() {
        if (extrudePreviewModel != null) extrudePreviewModel.dispose();
        extrudePreviewModel = null;
        extrudePreviewInstance = null;
        extruding = false;
    }

    private void disposePreview() {
        if (previewModel != null) {
            previewModel.dispose();
            previewModel = null;
            previewInstance = null;
        }
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
        modelBatch.dispose();
        if (previewModel != null) previewModel.dispose();
        if (extrudePreviewModel != null) extrudePreviewModel.dispose();
        for (ModelInstance inst : instances) {
            inst.model.dispose();
        }
    }
    
	 public static void main(String[] arg) {
	        LwjglApplicationConfiguration config = new LwjglApplicationConfiguration();
	        config.title = "CAD Assistant";
	        config.width = 1024;
	        config.height = 768;
	        config.vSyncEnabled = true;
	        new LwjglApplication(new SnapGridExtrudeDemo(), config);
	    }
}
