package test;

import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.PerspectiveCamera;
import com.badlogic.gdx.graphics.g3d.utils.CameraInputController;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Intersector;
import com.badlogic.gdx.math.Plane;
import com.badlogic.gdx.math.Vector3;

public class SnapGridDemoV0 extends ApplicationAdapter {
    private PerspectiveCamera camera;
    private CameraInputController camController;
    private ShapeRenderer shapeRenderer;

    private Plane groundPlane;
    private Vector3 tmpIntersection = new Vector3();

    private Vector3 firstPoint = null;
    private Vector3 secondPoint = null;

    private int gridSize = 20;     // grid extends [-gridSize, gridSize]
    private float cellSize = 1f;   // size of each grid cell

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();

        // Camera
        camera = new PerspectiveCamera(67, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.position.set(10f, 10f, 10f);
        camera.lookAt(0, 0, 0);
        camera.near = 0.1f;
        camera.far = 100f;
        camera.update();

        camController = new CameraInputController(camera);
        Gdx.input.setInputProcessor(camController);

        groundPlane = new Plane(new Vector3(0, 1, 0), 0); // y=0 plane
    }

    @Override
    public void render() {
        camController.update();
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT);

        shapeRenderer.setProjectionMatrix(camera.combined);

        // Draw grid
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.setColor(Color.LIGHT_GRAY);
        for (int i = -gridSize; i <= gridSize; i++) {
            shapeRenderer.line(i * cellSize, 0, -gridSize * cellSize, i * cellSize, 0, gridSize * cellSize);
            shapeRenderer.line(-gridSize * cellSize, 0, i * cellSize, gridSize * cellSize, 0, i * cellSize);
        }

        // Draw rectangle if two points selected
        if (firstPoint != null && secondPoint != null) {
            shapeRenderer.setColor(Color.RED);

            float x1 = firstPoint.x, z1 = firstPoint.z;
            float x2 = secondPoint.x, z2 = secondPoint.z;

            // Snap corners
            float minX = Math.min(x1, x2);
            float maxX = Math.max(x1, x2);
            float minZ = Math.min(z1, z2);
            float maxZ = Math.max(z1, z2);

            Vector3 p1 = new Vector3(minX, 0, minZ);
            Vector3 p2 = new Vector3(maxX, 0, minZ);
            Vector3 p3 = new Vector3(maxX, 0, maxZ);
            Vector3 p4 = new Vector3(minX, 0, maxZ);

            shapeRenderer.line(p1, p2);
            shapeRenderer.line(p2, p3);
            shapeRenderer.line(p3, p4);
            shapeRenderer.line(p4, p1);
        }

        shapeRenderer.end();

        handleInput();
    }

    private void handleInput() {
        if (Gdx.input.justTouched()) {
            Vector3 worldCoords = getMouseIntersection();
            if (worldCoords != null) {
                // Snap to nearest grid
                float snappedX = Math.round(worldCoords.x / cellSize) * cellSize;
                float snappedZ = Math.round(worldCoords.z / cellSize) * cellSize;
                Vector3 snapped = new Vector3(snappedX, 0, snappedZ);

                if (firstPoint == null) {
                    firstPoint = snapped;
                } else {
                    secondPoint = snapped;
                }
            }
        }

        // Reset with SPACE
        if (Gdx.input.isKeyJustPressed(Input.Keys.SPACE)) {
            firstPoint = null;
            secondPoint = null;
        }
    }

    private Vector3 getMouseIntersection() {
        // Create picking ray
        com.badlogic.gdx.math.collision.Ray ray = camera.getPickRay(Gdx.input.getX(), Gdx.input.getY());
        if (Intersector.intersectRayPlane(ray, groundPlane, tmpIntersection)) {
            return tmpIntersection.cpy();
        }
        return null;
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
    }
	 public static void main(String[] arg) {
	        LwjglApplicationConfiguration config = new LwjglApplicationConfiguration();
	        config.title = "CAD Assistant";
	        config.width = 1024;
	        config.height = 768;
	        config.vSyncEnabled = true;
	        new LwjglApplication(new SnapGridDemoV0(), config);
	    }
}
