package test;

import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.PerspectiveCamera;
import com.badlogic.gdx.graphics.g3d.utils.CameraInputController;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Intersector;
import com.badlogic.gdx.math.Plane;
import com.badlogic.gdx.math.Vector3;

import java.util.ArrayList;
import java.util.List;

public class SnapGridDemo extends ApplicationAdapter {
    private PerspectiveCamera camera;
    private CameraInputController camController;
    private ShapeRenderer shapeRenderer;

    private Plane groundPlane;
    private Vector3 tmpIntersection = new Vector3();

    private Vector3 firstPoint = null;
    private Vector3 secondPoint = null;

    private int gridSize = 20;
    private float cellSize = 1f;

    private List<Rectangle3D> rectangles = new ArrayList<>();

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();

        camera = new PerspectiveCamera(67, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.position.set(10f, 10f, 10f);
        camera.lookAt(0, 0, 0);
        camera.near = 0.1f;
        camera.far = 100f;
        camera.update();

        camController = new CameraInputController(camera);
        Gdx.input.setInputProcessor(camController);

        groundPlane = new Plane(new Vector3(0, 1, 0), 0); // y=0 plane
    }

    @Override
    public void render() {
        camController.update();
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT);

        shapeRenderer.setProjectionMatrix(camera.combined);

        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);

        // Draw grid
        shapeRenderer.setColor(Color.LIGHT_GRAY);
        for (int i = -gridSize; i <= gridSize; i++) {
            shapeRenderer.line(i * cellSize, 0, -gridSize * cellSize, i * cellSize, 0, gridSize * cellSize);
            shapeRenderer.line(-gridSize * cellSize, 0, i * cellSize, gridSize * cellSize, 0, i * cellSize);
        }

        // Draw finalized rectangles
        shapeRenderer.setColor(Color.RED);
        for (Rectangle3D rect : rectangles) {
            rect.draw(shapeRenderer);
        }

        // Draw preview rectangle (not finalized yet)
        if (firstPoint != null && secondPoint != null) {
            shapeRenderer.setColor(Color.GREEN);

            Rectangle3D preview = new Rectangle3D(firstPoint, secondPoint);
            preview.draw(shapeRenderer);
        }

        shapeRenderer.end();

        handleInput();
    }

    private void handleInput() {
        if (Gdx.input.justTouched()) {
            Vector3 worldCoords = getMouseIntersection();
            if (worldCoords != null) {
                float snappedX = Math.round(worldCoords.x / cellSize) * cellSize;
                float snappedZ = Math.round(worldCoords.z / cellSize) * cellSize;
                Vector3 snapped = new Vector3(snappedX, 0, snappedZ);

                if (firstPoint == null) {
                    firstPoint = snapped;
                } else {
                    secondPoint = snapped;
                }
            }
        }

        // ENTER to validate rectangle
        if (Gdx.input.isKeyJustPressed(Input.Keys.ENTER)) {
            if (firstPoint != null && secondPoint != null) {
                rectangles.add(new Rectangle3D(firstPoint, secondPoint));
                firstPoint = null;
                secondPoint = null;
            }
        }

        // SPACE to clear all
        if (Gdx.input.isKeyJustPressed(Input.Keys.SPACE)) {
            rectangles.clear();
            firstPoint = null;
            secondPoint = null;
        }
    }

    private Vector3 getMouseIntersection() {
        com.badlogic.gdx.math.collision.Ray ray = camera.getPickRay(Gdx.input.getX(), Gdx.input.getY());
        if (Intersector.intersectRayPlane(ray, groundPlane, tmpIntersection)) {
            return tmpIntersection.cpy();
        }
        return null;
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
    }

    // Helper class for rectangles on XZ plane
    static class Rectangle3D {
        Vector3 p1, p2, p3, p4;

        Rectangle3D(Vector3 a, Vector3 b) {
            float minX = Math.min(a.x, b.x);
            float maxX = Math.max(a.x, b.x);
            float minZ = Math.min(a.z, b.z);
            float maxZ = Math.max(a.z, b.z);

            p1 = new Vector3(minX, 0, minZ);
            p2 = new Vector3(maxX, 0, minZ);
            p3 = new Vector3(maxX, 0, maxZ);
            p4 = new Vector3(minX, 0, maxZ);
        }

        void draw(ShapeRenderer renderer) {
            renderer.line(p1, p2);
            renderer.line(p2, p3);
            renderer.line(p3, p4);
            renderer.line(p4, p1);
        }
    }
	 public static void main(String[] arg) {
	        LwjglApplicationConfiguration config = new LwjglApplicationConfiguration();
	        config.title = "CAD Assistant";
	        config.width = 1024;
	        config.height = 768;
	        config.vSyncEnabled = true;
	        new LwjglApplication(new SnapGridDemo(), config);
	    }
}
