package test;
// MyCADApp.java
import com.badlogic.gdx.*;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g3d.*;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.utils.MeshBuilder;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.*;

public class MyCADAppV0 extends ApplicationAdapter implements InputProcessor {
    private OrthographicCamera cam;
    private ShapeRenderer shapeRenderer;
    private BitmapFont font;
    private float gridSpacing = 1f;
    private int gridSize = 80; // 20x20 grid

    // Camera control
    private Vector3 tmpVector = new Vector3();
    private Vector2 lastMouse = new Vector2();
    private boolean isPanning = false;
    private boolean isRotating = false;
    private float cameraDistance = 20f;
    private float yaw = 45f, pitch = 30f; // Orbit angles

    @Override
    public void create() {
        // Orthographic camera (y-up, no perspective)
        cam = new OrthographicCamera();
        cam.setToOrtho(false, 20f, 20f * Gdx.graphics.getHeight() / (float)Gdx.graphics.getWidth());
        updateCamera();

        shapeRenderer = new ShapeRenderer();
        font = new BitmapFont();
        Gdx.input.setInputProcessor(this);
        Gdx.gl.glEnable(GL20.GL_DEPTH_TEST);
    }

    private void updateCamera() {
        // Position camera using spherical coordinates (orbit camera)
        float radYaw = MathUtils.degreesToRadians * yaw;
        float radPitch = MathUtils.degreesToRadians * pitch;
        float cosPitch = MathUtils.cos(radPitch);
        float sinPitch = MathUtils.sin(radPitch);
        float cosYaw = MathUtils.cos(radYaw);
        float sinYaw = MathUtils.sin(radYaw);

        cam.position.set(
            cameraDistance * cosPitch * cosYaw,
            cameraDistance * sinPitch,
            cameraDistance * cosPitch * sinYaw
        );
        cam.lookAt(0, 0, 0);
        cam.up.set(0, 1, 0);
        cam.update();
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.9f, 0.92f, 0.95f, 1); // Light blue-gray (CAD-like)
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT);

        // Render grid
        renderGrid();

        // Optional: Render sample geometry (e.g., a box)
        // renderSampleBox();
    }

    private void renderGrid() {
        shapeRenderer.setProjectionMatrix(cam.combined);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);

        // Set grid color
        shapeRenderer.setColor(0.7f, 0.7f, 0.7f, 1f);

        // Draw grid lines
        for (int i = -gridSize; i <= gridSize; i++) {
            // Vertical lines
            shapeRenderer.line(i * gridSpacing, -gridSize * gridSpacing, 
                               i * gridSpacing,  gridSize * gridSpacing);
            // Horizontal lines
            shapeRenderer.line(-gridSize * gridSpacing, i * gridSpacing,
                                gridSize * gridSpacing, i * gridSpacing);
        }

        // Highlight origin
        shapeRenderer.setColor(Color.RED);
        shapeRenderer.line(0, -0.5f, 0, 0.5f);
        shapeRenderer.line(-0.5f, 0, 0.5f, 0);

        shapeRenderer.end();
    }

//    // ===== Input Handling =====
//    @Override
//    public boolean scrolled(float amountX, float amountY) {
//        // Zoom by adjusting camera distance
//        cameraDistance += amountY * 2f;
//        cameraDistance = MathUtils.clamp(cameraDistance, 5f, 100f);
//        updateCamera();
//        return true;
//    }

    @Override
    public boolean touchDown(int screenX, int screenY, int pointer, int button) {
        lastMouse.set(screenX, screenY);
        if (button == Input.Buttons.LEFT) {
            isRotating = true;
        } else if (button == Input.Buttons.RIGHT) {
            isPanning = true;
        }
        return true;
    }

    @Override
    public boolean touchDragged(int screenX, int screenY, int pointer) {
        Vector2 current = new Vector2(screenX, screenY);
        Vector2 delta = new Vector2(current).sub(lastMouse);

        if (isRotating) {
            yaw -= delta.x * 0.5f;
            pitch += delta.y * 0.5f;
            pitch = MathUtils.clamp(pitch, -89f, 89f); // Avoid flip
            updateCamera();
        } else if (isPanning) {
            // Pan: move camera perpendicular to view direction
            tmpVector.set(cam.direction).crs(cam.up).scl(-delta.x * 0.1f);
            cam.position.add(tmpVector);
            tmpVector.set(cam.up).scl(delta.y * 0.1f);
            cam.position.add(tmpVector);
            cam.update();
        }

        lastMouse.set(current);
        return true;
    }

    @Override
    public boolean touchUp(int screenX, int screenY, int pointer, int button) {
        isRotating = false;
        isPanning = false;
        return true;
    }

    // Unused InputProcessor methods
    @Override public boolean keyDown(int keycode) { return false; }
    @Override public boolean keyUp(int keycode) { return false; }
    @Override public boolean keyTyped(char character) { return false; }
    @Override public boolean mouseMoved(int screenX, int screenY) { return false; }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
        font.dispose();
    }

	@Override
	public boolean scrolled(int amount) {
		 // In orthographic camera, "zoom" = change viewport size
	    float zoomSpeed = 0.1f; // Adjust sensitivity

	    // Smaller viewport = zoomed IN
	    // Larger viewport = zoomed OUT
	    float newViewportWidth = cam.viewportWidth + amount * zoomSpeed * cam.viewportWidth;

	    // Clamp to reasonable limits
	    newViewportWidth = MathUtils.clamp(newViewportWidth, 2f, 100f);

	    // Preserve aspect ratio
	    float aspectRatio = (float) Gdx.graphics.getHeight() / (float) Gdx.graphics.getWidth();
	    cam.setToOrtho(false, newViewportWidth, newViewportWidth * aspectRatio);
	    cam.update();

	    return true;
	}
	 public static void main(String[] arg) {
	        LwjglApplicationConfiguration config = new LwjglApplicationConfiguration();
	        config.title = "CAD Assistant";
	        config.width = 1024;
	        config.height = 768;
	        config.vSyncEnabled = true;
	        new LwjglApplication(new MyCADAppV0(), config);
	    }
}