package test;
import com.badlogic.gdx.*;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.g3d.*;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.environment.DirectionalLight;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.*;

import java.util.ArrayList;
import java.util.List;

public class MyCADApp extends ApplicationAdapter implements InputProcessor {
    private OrthographicCamera cam;
    private ModelBatch modelBatch;
    private Environment environment;
    private ShapeRenderer shapeRenderer;
    
    // Grid
    private float gridSpacing = 1f;
    private int gridSize = 20;
    
    // Drawing state
    private boolean isDrawing = false;
    private Vector2 startPos = new Vector2();
    private Vector2 currentPos = new Vector2();
    private List<ModelInstance> solids = new ArrayList<>();
    
    // Camera control
    private Vector2 lastMouse = new Vector2();
    private boolean isPanning = false;
    private float cameraDistance = 20f;
    private float yaw = 45f, pitch = 30f;
    
    // Extrusion
    private ModelInstance selectedSolid = null;
    private boolean isExtruding = false;
    private float extrusionStartHeight = 0f;

    @Override
    public void create() {
        // Camera
        cam = new OrthographicCamera();
        updateCameraViewport();
        updateCameraPosition();
        
        // Rendering
        modelBatch = new ModelBatch();
        environment = new Environment();
        environment.set(new ColorAttribute(ColorAttribute.AmbientLight, 0.6f, 0.6f, 0.6f, 1f));
        environment.add(new DirectionalLight().set(0.8f, 0.8f, 0.8f, -1f, -0.8f, -0.2f));
        
        shapeRenderer = new ShapeRenderer();
        Gdx.gl.glEnable(GL20.GL_DEPTH_TEST);
        Gdx.input.setInputProcessor(this);
    }

    private void updateCameraViewport() {
        float aspect = (float) Gdx.graphics.getHeight() / (float) Gdx.graphics.getWidth();
        cam.setToOrtho(false, cameraDistance, cameraDistance * aspect);
    }

    private void updateCameraPosition() {
        float radYaw = MathUtils.degreesToRadians * yaw;
        float radPitch = MathUtils.degreesToRadians * pitch;
        cam.position.set(
            cameraDistance * MathUtils.cos(radPitch) * MathUtils.cos(radYaw),
            cameraDistance * MathUtils.sin(radPitch),
            cameraDistance * MathUtils.cos(radPitch) * MathUtils.sin(radYaw)
        );
        cam.lookAt(0, 0, 0);
        cam.up.set(0, 1, 0);
        cam.update();
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.9f, 0.92f, 0.95f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT);

        // Render grid
        renderGrid();

        // Render preview rectangle while drawing
        if (isDrawing) {
            renderPreviewRectangle();
        }

        // Render all solids
        modelBatch.begin(cam);
        for (ModelInstance solid : solids) {
            modelBatch.render(solid, environment);
        }
        modelBatch.end();
    }

    private void renderGrid() {
        shapeRenderer.setProjectionMatrix(cam.combined);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.setColor(0.7f, 0.7f, 0.7f, 1f);

        for (int i = -gridSize; i <= gridSize; i++) {
            float pos = i * gridSpacing;
            shapeRenderer.line(pos, -gridSize * gridSpacing, pos, gridSize * gridSpacing);
            shapeRenderer.line(-gridSize * gridSpacing, pos, gridSize * gridSpacing, pos);
        }

        // Origin
        shapeRenderer.setColor(Color.RED);
        shapeRenderer.line(0, -0.5f, 0, 0.5f);
        shapeRenderer.line(-0.5f, 0, 0.5f, 0);
        shapeRenderer.end();
    }

    private void renderPreviewRectangle() {
        Vector2 snappedStart = snapToGrid(startPos);
        Vector2 snappedCurrent = snapToGrid(currentPos);
        
        float x = Math.min(snappedStart.x, snappedCurrent.x);
        float z = Math.min(snappedStart.y, snappedCurrent.y); // Note: y in 2D = z in 3D
        float width = Math.abs(snappedCurrent.x - snappedStart.x);
        float depth = Math.abs(snappedCurrent.y - snappedStart.y);

        if (width > 0.01f && depth > 0.01f) {
            shapeRenderer.setProjectionMatrix(cam.combined);
            shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
            shapeRenderer.setColor(Color.BLUE);
            // Draw rectangle in XZ plane (Y=0)
            shapeRenderer.box(x, 0, z, width, 0.01f, depth);
            shapeRenderer.end();
        }
    }

    private Vector2 snapToGrid(Vector2 worldPos) {
        return new Vector2(
            Math.round(worldPos.x / gridSpacing) * gridSpacing,
            Math.round(worldPos.y / gridSpacing) * gridSpacing
        );
    }

    private Vector2 screenToWorld(int screenX, int screenY) {
        Vector3 world = new Vector3(screenX, screenY, 0);
        cam.unproject(world);
        return new Vector2(world.x, world.z); // XZ plane
    }

    // ===== Input Handling =====
    @Override
    public boolean touchDown(int screenX, int screenY, int pointer, int button) {
        Vector2 worldPos = screenToWorld(screenX, screenY);
        lastMouse.set(screenX, screenY);

        if (button == Input.Buttons.LEFT) {
            if (!isExtruding) {
                isDrawing = true;
                startPos.set(worldPos);
                currentPos.set(worldPos);
            }
        } else if (button == Input.Buttons.RIGHT) {
            isPanning = true;
        }
        return true;
    }

    @Override
    public boolean touchDragged(int screenX, int screenY, int pointer) {
        Vector2 currentWorld = screenToWorld(screenX, screenY);
        Vector2 deltaScreen = new Vector2(screenX, screenY).sub(lastMouse);

        if (isDrawing) {
            currentPos.set(currentWorld);
        } else if (isExtruding) {
            // Extrude: use vertical mouse movement
            float deltaHeight = -deltaScreen.y * 0.05f; // Sensitivity
            float newHeight = extrusionStartHeight + deltaHeight;
            newHeight = Math.max(0.1f, newHeight); // Min height

            // Update solid height
            Matrix4 transform = selectedSolid.transform;
            Vector3 scale = new Vector3();
            transform.getScale(scale);
            scale.y = newHeight;
            transform.setToTranslation(transform.getTranslation(new Vector3()))
                    .scale(1, newHeight / extrusionStartHeight, 1);
        } else if (isPanning) {
            // Pan camera
            Vector3 right = new Vector3(cam.direction).crs(cam.up).nor().scl(-deltaScreen.x * 0.1f);
            Vector3 up = new Vector3(cam.up).scl(deltaScreen.y * 0.1f);
            cam.position.add(right).add(up);
            cam.update();
        } else {
            // Orbit (middle mouse or Ctrl+Left in real CAD, but we'll use left for drawing only)
            // For now, orbit only if not drawing/extruding
            yaw -= deltaScreen.x * 0.5f;
            pitch = MathUtils.clamp(pitch + deltaScreen.y * 0.5f, -89f, 89f);
            updateCameraPosition();
        }

        lastMouse.set(screenX, screenY);
        return true;
    }

    @Override
    public boolean touchUp(int screenX, int screenY, int pointer, int button) {
        if (button == Input.Buttons.LEFT) {
            if (isDrawing) {
                // Finalize rectangle
                Vector2 snappedStart = snapToGrid(startPos);
                Vector2 snappedEnd = snapToGrid(currentPos);
                
                float x = Math.min(snappedStart.x, snappedEnd.x);
                float z = Math.min(snappedStart.y, snappedEnd.y);
                float width = Math.abs(snappedEnd.x - snappedStart.x);
                float depth = Math.abs(snappedEnd.y - snappedStart.y);

                if (width > 0.01f && depth > 0.01f) {
                    // Create 3D box (height = 0.1 initially)
                    ModelBuilder modelBuilder = new ModelBuilder();
                    Model model = modelBuilder.createBox(
                        width, 0.1f, depth,
                        new Material(ColorAttribute.createDiffuse(Color.GRAY)),
                        VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal
                    );
                    ModelInstance instance = new ModelInstance(model);
                    instance.transform.setToTranslation(x + width/2f, 0.05f, z + depth/2f); // Centered
                    solids.add(instance);
                    
                    // Auto-select for extrusion
                    selectedSolid = instance;
                    extrusionStartHeight = 0.1f;
                    isExtruding = true;
                }
                isDrawing = false;
            }
        } else if (button == Input.Buttons.RIGHT) {
            isPanning = false;
        }
        return true;
    }

    @Override
    public boolean scrolled(int amount) {
        // Zoom by changing viewport size
        cameraDistance += amount * 2f;
        cameraDistance = MathUtils.clamp(cameraDistance, 5f, 100f);
        updateCameraViewport();
        updateCameraPosition(); // Keep orbit position consistent
        return true;
    }

    // Optional: Press 'E' to extrude selected solid
    @Override
    public boolean keyDown(int keycode) {
        if (keycode == Input.Keys.E && selectedSolid != null && !isExtruding) {
            isExtruding = true;
            Matrix4 transform = selectedSolid.transform;
            Vector3 scale = new Vector3();
            transform.getScale(scale);
            extrusionStartHeight = scale.y;
        }
        return false;
    }

    @Override
    public boolean keyUp(int keycode) {
        if (keycode == Input.Keys.E) {
            isExtruding = false;
        }
        return false;
    }

    // Unused
    @Override public boolean keyTyped(char character) { return false; }
    @Override public boolean mouseMoved(int screenX, int screenY) { return false; }

    @Override
    public void dispose() {
        modelBatch.dispose();
        shapeRenderer.dispose();
        for (ModelInstance solid : solids) {
            solid.model.dispose();
        }
    }
    
	 public static void main(String[] arg) {
	        LwjglApplicationConfiguration config = new LwjglApplicationConfiguration();
	        config.title = "CAD Assistant";
	        config.width = 1024;
	        config.height = 768;
	        config.vSyncEnabled = true;
	        new LwjglApplication(new MyCADApp(), config);
	    }
}