package test;
import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.InputAdapter;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.backends.lwjgl.LwjglAWTCanvas;

import javax.swing.*;

public class LibGdxJFrameZoomDragExample extends ApplicationAdapter {
    private ShapeRenderer shapeRenderer;
    private OrthographicCamera camera;
    private final Vector3 tmp = new Vector3();
    private final Vector3 tmp2 = new Vector3();

    // Circles
    private float circle1X = 200, circle1Y = 200, radius1 = 50;
    private float circle2X = 500, circle2Y = 300, radius2 = 70;

    // Drag state
    private boolean dragging = false;
    private int selectedCircle = -1; // -1 = none, 1 = circle1, 2 = circle2
    private float dragOffsetX = 0f, dragOffsetY = 0f;

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();
        camera = new OrthographicCamera(Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.setToOrtho(false);
        camera.update();

        Gdx.input.setInputProcessor(new InputAdapter() {
            @Override
            public boolean touchDown(int screenX, int screenY, int pointer, int button) {
                // convert screen -> world (flip Y)
                tmp.set(screenX, Gdx.graphics.getHeight() - screenY, 0);
                camera.unproject(tmp);
                float worldX = tmp.x;
                float worldY = tmp.y;

                if (isInsideCircle(worldX, worldY, circle1X, circle1Y, radius1)) {
                    selectedCircle = 1;
                    dragging = true;
                    dragOffsetX = worldX - circle1X;
                    dragOffsetY = worldY - circle1Y;
                } else if (isInsideCircle(worldX, worldY, circle2X, circle2Y, radius2)) {
                    selectedCircle = 2;
                    dragging = true;
                    dragOffsetX = worldX - circle2X;
                    dragOffsetY = worldY - circle2Y;
                }

                return true;
            }

            @Override
            public boolean touchDragged(int screenX, int screenY, int pointer) {
                if (dragging) {
                    tmp.set(screenX, Gdx.graphics.getHeight() - screenY, 0);
                    camera.unproject(tmp);
                    float worldX = tmp.x;
                    float worldY = tmp.y;

                    if (selectedCircle == 1) {
                        circle1X = worldX - dragOffsetX;
                        circle1Y = worldY - dragOffsetY;
                    } else if (selectedCircle == 2) {
                        circle2X = worldX - dragOffsetX;
                        circle2Y = worldY - dragOffsetY;
                    }
                }
                return true;
            }

            @Override
            public boolean touchUp(int screenX, int screenY, int pointer, int button) {
                dragging = false;
                selectedCircle = -1;
                return true;
            }

            // Older LibGDX versions use this signature (int amount).
            @Override
            public boolean scrolled(int amount) {
                // amount > 0 : wheel moved one notch in one direction, amount < 0 : opposite
                final float zoomFactor = 1.1f;

                // Mouse screen coordinates
                int mx = Gdx.input.getX();
                int my = Gdx.input.getY();

                // World coords under mouse BEFORE zoom
                tmp.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp);
                float beforeX = tmp.x;
                float beforeY = tmp.y;

                // Apply zoom (clamped)
                if (amount > 0) {
                    camera.zoom *= zoomFactor; // zoom out
                } else if (amount < 0) {
                    camera.zoom /= zoomFactor; // zoom in
                }
                camera.zoom = MathUtils.clamp(camera.zoom, 0.05f, 20f);
                camera.update();

                // World coords under mouse AFTER zoom
                tmp2.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp2);

                // Move camera so the point under the cursor stays fixed in screen space
                camera.position.add(beforeX - tmp2.x, beforeY - tmp2.y, 0);
                camera.update();

                return true;
            }
        });
    }

    private boolean isInsideCircle(float px, float py, float cx, float cy, float r) {
        float dx = px - cx;
        float dy = py - cy;
        return dx * dx + dy * dy <= r * r;
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.1f, 0.1f, 0.3f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        camera.update();
        shapeRenderer.setProjectionMatrix(camera.combined);

        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);

        // Circle 1: red
        shapeRenderer.setColor(1, 0, 0, 1);
        shapeRenderer.circle(circle1X, circle1Y, radius1);

        // Circle 2: green
        shapeRenderer.setColor(0, 1, 0, 1);
        shapeRenderer.circle(circle2X, circle2Y, radius2);

        shapeRenderer.end();
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("LibGDX + JFrame Drag + Zoom (mouse-centered)");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(800, 600);

            LwjglAWTCanvas canvas = new LwjglAWTCanvas(new LibGdxJFrameZoomDragExample());
            frame.add(canvas.getCanvas());

            frame.setVisible(true);
        });
    }
}
