package test;
import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.InputAdapter;
import com.badlogic.gdx.backends.lwjgl.LwjglAWTCanvas;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g2d.SpriteBatch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Vector3;

import javax.swing.*;

public class LibGdxJFrameSnapGridHint extends ApplicationAdapter {
    private ShapeRenderer shapeRenderer;
    private OrthographicCamera camera;
    private final Vector3 tmp = new Vector3();
    private final Vector3 tmp2 = new Vector3();

    // Circles
    private float circle1X = 200, circle1Y = 200, radius1 = 50;
    private float circle2X = 500, circle2Y = 300, radius2 = 70;

    // Drag state
    private boolean draggingCircle = false;
    private int selectedCircle = -1;
    private float dragOffsetX = 0f, dragOffsetY = 0f;

    // Camera pan state
    private boolean panning = false;
    private int lastPanX, lastPanY;

    // Grid
    private float gridSize = 20f;
    private boolean snapEnabled = true;

    // UI
    private SpriteBatch batch;
    private BitmapFont font;

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();
        camera = new OrthographicCamera(Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.setToOrtho(false);
        camera.update();

        batch = new SpriteBatch();
        font = new BitmapFont(); // default font

        Gdx.input.setInputProcessor(new InputAdapter() {
            @Override
            public boolean touchDown(int screenX, int screenY, int pointer, int button) {
                if (button == Input.Buttons.LEFT) {
                    tmp.set(screenX, Gdx.graphics.getHeight() - screenY, 0);
                    camera.unproject(tmp);
                    float worldX = tmp.x;
                    float worldY = tmp.y;

                    if (isInsideCircle(worldX, worldY, circle1X, circle1Y, radius1)) {
                        selectedCircle = 1;
                        draggingCircle = true;
                        dragOffsetX = worldX - circle1X;
                        dragOffsetY = worldY - circle1Y;
                    } else if (isInsideCircle(worldX, worldY, circle2X, circle2Y, radius2)) {
                        selectedCircle = 2;
                        draggingCircle = true;
                        dragOffsetX = worldX - circle2X;
                        dragOffsetY = worldY - circle2Y;
                    }
                } else if (button == Input.Buttons.RIGHT) {
                    panning = true;
                    lastPanX = screenX;
                    lastPanY = screenY;
                }
                return true;
            }

            @Override
            public boolean touchDragged(int screenX, int screenY, int pointer) {
                if (draggingCircle) {
                    tmp.set(screenX, Gdx.graphics.getHeight() - screenY, 0);
                    camera.unproject(tmp);
                    float worldX = tmp.x;
                    float worldY = tmp.y;

                    if (selectedCircle == 1) {
                        circle1X = snap(worldX - dragOffsetX);
                        circle1Y = snap(worldY - dragOffsetY);
                    } else if (selectedCircle == 2) {
                        circle2X = snap(worldX - dragOffsetX);
                        circle2Y = snap(worldY - dragOffsetY);
                    }
                } else if (panning) {
                    int dx = screenX - lastPanX;
                    int dy = screenY - lastPanY;

                    float newX = camera.position.x - dx * camera.zoom;
                    float newY = camera.position.y + dy * camera.zoom;

                    camera.position.set(snap(newX), snap(newY), 0);
                    camera.update();

                    lastPanX = screenX;
                    lastPanY = screenY;
                }
                return true;
            }

            @Override
            public boolean touchUp(int screenX, int screenY, int pointer, int button) {
                if (button == Input.Buttons.LEFT) {
                    draggingCircle = false;
                    selectedCircle = -1;
                } else if (button == Input.Buttons.RIGHT) {
                    panning = false;
                }
                return true;
            }

            @Override
            public boolean scrolled(int amount) {
                final float zoomFactor = 1.1f;
                int mx = Gdx.input.getX();
                int my = Gdx.input.getY();

                tmp.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp);
                float beforeX = tmp.x;
                float beforeY = tmp.y;

                if (amount > 0) camera.zoom *= zoomFactor;
                else if (amount < 0) camera.zoom /= zoomFactor;

                camera.zoom = MathUtils.clamp(camera.zoom, 0.05f, 20f);
                camera.update();

                tmp2.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp2);

                camera.position.add(beforeX - tmp2.x, beforeY - tmp2.y, 0);
                camera.update();

                return true;
            }
        });
    }

    private boolean isInsideCircle(float px, float py, float cx, float cy, float r) {
        float dx = px - cx;
        float dy = py - cy;
        return dx * dx + dy * dy <= r * r;
    }

    private float snap(float value) {
        if (!snapEnabled) return value;
        return Math.round(value / gridSize) * gridSize;
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.1f, 0.1f, 0.3f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        camera.update();
        shapeRenderer.setProjectionMatrix(camera.combined);

        // Draw circles
        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
        shapeRenderer.setColor(1, 0, 0, 1);
        shapeRenderer.circle(circle1X, circle1Y, radius1);
        shapeRenderer.setColor(0, 1, 0, 1);
        shapeRenderer.circle(circle2X, circle2Y, radius2);
        shapeRenderer.end();

        // Draw grid
        drawGrid();

        // Draw UI (mouse coords hint)
        batch.begin();
        int mx = Gdx.input.getX();
        int my = Gdx.input.getY();
        tmp.set(mx, Gdx.graphics.getHeight() - my, 0);
        camera.unproject(tmp);

        String coords = String.format("X: %.2f  Y: %.2f", tmp.x, tmp.y);
        font.draw(batch, coords, 10, 20); // bottom-left
        batch.end();
    }

    private void drawGrid() {
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.setColor(0.3f, 0.3f, 0.3f, 1);

        float left = camera.position.x - camera.viewportWidth * camera.zoom / 2f;
        float right = camera.position.x + camera.viewportWidth * camera.zoom / 2f;
        float bottom = camera.position.y - camera.viewportHeight * camera.zoom / 2f;
        float top = camera.position.y + camera.viewportHeight * camera.zoom / 2f;

        for (float x = snap(left); x <= right; x += gridSize) {
            shapeRenderer.line(x, bottom, x, top);
        }
        for (float y = snap(bottom); y <= top; y += gridSize) {
            shapeRenderer.line(left, y, right, y);
        }
        shapeRenderer.end();
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
        batch.dispose();
        font.dispose();
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("LibGDX + JFrame + Grid + Hint");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(800, 600);

            LwjglAWTCanvas canvas = new LwjglAWTCanvas(new LibGdxJFrameSnapGridHint());
            frame.add(canvas.getCanvas());

            frame.setVisible(true);
        });
    }
}
