package test;

import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.InputAdapter;
import com.badlogic.gdx.InputMultiplexer;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.Stage;
import com.badlogic.gdx.scenes.scene2d.utils.ClickListener;
import com.badlogic.gdx.backends.lwjgl.LwjglAWTCanvas;

import java.awt.Canvas;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;

import javax.swing.*;

public class LibGdxJFrameSnapGridExample3 extends ApplicationAdapter {
    static Canvas lastcanvass = null;
    static int lastwidth, lastheight;
    static float lastzoom = 1.0f;
    static Vector3 lastcameraposition = new Vector3();

    private ShapeRenderer shapeRenderer;
    private OrthographicCamera camera;
    private final Vector3 tmp = new Vector3();
    private final Vector3 tmp2 = new Vector3();
    private Stage stage;
    private BitmapFont font;

    // Grid settings
    private float gridSize = 20f;
    private boolean snapEnabled = true;

    // Camera pan state
    private boolean panning = false;
    private int lastPanX, lastPanY;

    // Memory monitor
    private static final Runtime runtime = Runtime.getRuntime();

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();
        font = new BitmapFont();
        font.getData().setScale(0.8f);
        font.setColor(1, 1, 1, 1);

        camera = new OrthographicCamera(Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.setToOrtho(false);
        camera.update();

        // Create circles as actors
        DraggableCircle circle1 = new DraggableCircle(200, 200, 50, 1, 0, 0); // red
        DraggableCircle circle2 = new DraggableCircle(500, 300, 70, 0, 1, 0); // green

        stage = new Stage();
        stage.addActor(circle1);
        stage.addActor(circle2);

        // Input for camera (pan, zoom)
        InputAdapter cameraInput = new InputAdapter() {
            @Override
            public boolean touchDown(int screenX, int screenY, int pointer, int button) {
                if (button == Input.Buttons.RIGHT) {
                    panning = true;
                    lastPanX = screenX;
                    lastPanY = screenY;
                    return true;
                }
                return false;
            }

            @Override
            public boolean touchDragged(int screenX, int screenY, int pointer) {
                if (panning) {
                    int dx = screenX - lastPanX;
                    int dy = screenY - lastPanY;

                    float newX = camera.position.x - dx * camera.zoom;
                    float newY = camera.position.y + dy * camera.zoom;

                    camera.position.set(snap(newX), snap(newY), 0);
                    camera.update();
                    lastcameraposition.set(camera.position);
                    lastPanX = screenX;
                    lastPanY = screenY;
                    return true;
                }
                return false;
            }

            @Override
            public boolean touchUp(int screenX, int screenY, int pointer, int button) {
                if (button == Input.Buttons.RIGHT) {
                    panning = false;
                    return true;
                }
                return false;
            }

            @Override
            public boolean scrolled(int amount) {
                final float zoomFactor = 1.1f;
                int mx = Gdx.input.getX();
                int my = Gdx.input.getY();

                tmp.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp);
                float beforeX = tmp.x;
                float beforeY = tmp.y;

                if (amount > 0) {
                    camera.zoom *= zoomFactor;
                } else if (amount < 0) {
                    camera.zoom /= zoomFactor;
                }
                camera.zoom = MathUtils.clamp(camera.zoom, 0.05f, 20f);
                camera.update();
                lastzoom = camera.zoom;

                tmp2.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp2);

                camera.position.add(beforeX - tmp2.x, beforeY - tmp2.y, 0);
                camera.update();
                lastcameraposition.set(camera.position);
                return true;
            }
        };

        // Add drag listeners to circles
        addDragListener(circle1);
        addDragListener(circle2);

        // Combine inputs: camera first, then stage (so circles get priority on left-click)
        InputMultiplexer multiplexer = new InputMultiplexer();
        multiplexer.addProcessor(cameraInput);
        multiplexer.addProcessor(stage);
        Gdx.input.setInputProcessor(multiplexer);
    }

    private void addDragListener(DraggableCircle circle) {
        circle.addListener(new ClickListener(Input.Buttons.LEFT) {
            private float dragOffsetX, dragOffsetY;

            @Override
            public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
                // Convert screen to world
                tmp.set(Gdx.input.getX(), Gdx.input.getY(), 0);
                camera.unproject(tmp);
                float worldX = tmp.x;
                float worldY = tmp.y;

                if (circle.contains(worldX, worldY)) {
                    dragOffsetX = worldX - circle.x;
                    dragOffsetY = worldY - circle.y;
                    return true;
                }
                return false;
            }

            @Override
            public void touchDragged(InputEvent event, float x, float y, int pointer) {
                tmp.set(Gdx.input.getX(), Gdx.input.getY(), 0);
                camera.unproject(tmp);
                float worldX = tmp.x;
                float worldY = tmp.y;

                float newX = snap(worldX - dragOffsetX);
                float newY = snap(worldY - dragOffsetY);
                circle.setPosition(newX, newY);
            }
        });
    }

    private boolean isInsideCircle(float px, float py, float cx, float cy, float r) {
        float dx = px - cx;
        float dy = py - cy;
        return dx * dx + dy * dy <= r * r;
    }

    private float snap(float value) {
        if (!snapEnabled) return value;
        return Math.round(value / gridSize) * gridSize;
    }

    private String getMemoryUsageString() {
        long maxMemory = runtime.maxMemory();
        long totalMemory = runtime.totalMemory();
        long freeMemory = runtime.freeMemory();
        long usedMemory = totalMemory - freeMemory;
        float usedMB = usedMemory / (1024f * 1024f);
        float maxMB = maxMemory / (1024f * 1024f);
        return String.format("Mem: %.1f / %.0f MB", usedMB, maxMB);
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.1f, 0.1f, 0.3f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        camera.update();
        shapeRenderer.setProjectionMatrix(camera.combined);

        // Render all circles
        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
        for (Actor actor : stage.getActors()) {
            if (actor instanceof DraggableCircle) {
                ((DraggableCircle) actor).render(shapeRenderer);
            }
        }
        shapeRenderer.end();

        // Draw grid
        drawGrid();

        // Draw memory monitor
        String memText = getMemoryUsageString();
//        font.draw(memText, 10, Gdx.graphics.getHeight() - 10);

        // Update stage (for input)
        stage.act();
    }

    private void drawGrid() {
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.setColor(0.3f, 0.3f, 0.3f, 1);

        float left = camera.position.x - camera.viewportWidth * camera.zoom / 2f;
        float right = camera.position.x + camera.viewportWidth * camera.zoom / 2f;
        float bottom = camera.position.y - camera.viewportHeight * camera.zoom / 2f;
        float top = camera.position.y + camera.viewportHeight * camera.zoom / 2f;

        for (float x = snap(left); x <= right; x += gridSize) {
            shapeRenderer.line(x, bottom, x, top);
        }
        for (float y = snap(bottom); y <= top; y += gridSize) {
            shapeRenderer.line(left, y, right, y);
        }

        shapeRenderer.end();
    }

    @Override
    public void resize(int width, int height) {
        if (width <= 0 || height <= 0) return;
        camera.viewportWidth = width;
        camera.viewportHeight = height;
        camera.update();
        camera.zoom = lastzoom;
        camera.position.set(lastcameraposition);
        if (stage != null) {
            stage.getViewport().update(width, height, true);
        }
    }

    @Override
    public void dispose() {
        if (font != null) font.dispose();
        if (shapeRenderer != null) shapeRenderer.dispose();
        if (stage != null) stage.dispose();
    }

    // DraggableCircle Actor
    public static class DraggableCircle extends Actor {
        public float x, y, radius;
        public float r, g, b;

        public DraggableCircle(float x, float y, float radius, float r, float g, float b) {
            this.x = x;
            this.y = y;
            this.radius = radius;
            this.r = r;
            this.g = g;
            this.b = b;
            setBounds(x - radius, y - radius, radius * 2, radius * 2);
        }

        public boolean contains(float worldX, float worldY) {
            float dx = worldX - x;
            float dy = worldY - y;
            return dx * dx + dy * dy <= radius * radius;
        }

        public void setPosition(float x, float y) {
            this.x = x;
            this.y = y;
            setBounds(x - radius, y - radius, radius * 2, radius * 2);
        }

        public void render(ShapeRenderer shapeRenderer) {
            shapeRenderer.setColor(r, g, b, 1f);
            shapeRenderer.circle(x, y, radius);
        }
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("LibGDX + JFrame Drag + Zoom + Pan + Snap Grid + Actors + Memory");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(800, 600);
            frame.setLayout(new java.awt.BorderLayout());

            frame.addComponentListener(new ComponentAdapter() {
                @Override
                public void componentResized(ComponentEvent e) {
                    int w = frame.getContentPane().getWidth();
                    int h = frame.getContentPane().getHeight();
                    if (w <= 0 || h <= 0) return;

                    if (lastwidth != w || lastheight != h) {
                        lastwidth = w;
                        lastheight = h;
                        if (lastcanvass != null) {
                            frame.remove(lastcanvass);
                        }
                        LibGdxJFrameSnapGridExample3 app = new LibGdxJFrameSnapGridExample3();
                        LwjglAWTCanvas canvas = new LwjglAWTCanvas(app);
                        lastcanvass = canvas.getCanvas();
                        frame.add(lastcanvass, java.awt.BorderLayout.CENTER);
                        frame.validate();
                    }
                }
            });

            frame.setVisible(true);
        });
    }
}