package test;
import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.InputAdapter;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.backends.lwjgl.LwjglAWTCanvas;

import javax.swing.*;

public class LibGdxJFrameSnapGridExample extends ApplicationAdapter {
    private ShapeRenderer shapeRenderer;
    private OrthographicCamera camera;
    private final Vector3 tmp = new Vector3();
    private final Vector3 tmp2 = new Vector3();

    // Circles
    private float circle1X = 200, circle1Y = 200, radius1 = 50;
    private float circle2X = 500, circle2Y = 300, radius2 = 70;

    // Drag state
    private boolean draggingCircle = false;
    private int selectedCircle = -1; 
    private float dragOffsetX = 0f, dragOffsetY = 0f;

    // Camera pan state
    private boolean panning = false;
    private int lastPanX, lastPanY;

    // Grid settings
    private float gridSize = 20f; // change for finer/coarser snapping
    private boolean snapEnabled = true;

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();
        camera = new OrthographicCamera(Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        camera.setToOrtho(false);
        camera.update();

        Gdx.input.setInputProcessor(new InputAdapter() {
            @Override
            public boolean touchDown(int screenX, int screenY, int pointer, int button) {
                if (button == Input.Buttons.LEFT) {
                    // Left button → try to grab circle
                    tmp.set(screenX, Gdx.graphics.getHeight() - screenY, 0);
                    camera.unproject(tmp);
                    float worldX = tmp.x;
                    float worldY = tmp.y;

                    if (isInsideCircle(worldX, worldY, circle1X, circle1Y, radius1)) {
                        selectedCircle = 1;
                        draggingCircle = true;
                        dragOffsetX = worldX - circle1X;
                        dragOffsetY = worldY - circle1Y;
                    } else if (isInsideCircle(worldX, worldY, circle2X, circle2Y, radius2)) {
                        selectedCircle = 2;
                        draggingCircle = true;
                        dragOffsetX = worldX - circle2X;
                        dragOffsetY = worldY - circle2Y;
                    }
                } else if (button == Input.Buttons.RIGHT) {
                    // Right button → start panning
                    panning = true;
                    lastPanX = screenX;
                    lastPanY = screenY;
                }
                return true;
            }

            @Override
            public boolean touchDragged(int screenX, int screenY, int pointer) {
                if (draggingCircle) {
                    tmp.set(screenX, Gdx.graphics.getHeight() - screenY, 0);
                    camera.unproject(tmp);
                    float worldX = tmp.x;
                    float worldY = tmp.y;

                    if (selectedCircle == 1) {
                        circle1X = snap(worldX - dragOffsetX);
                        circle1Y = snap(worldY - dragOffsetY);
                    } else if (selectedCircle == 2) {
                        circle2X = snap(worldX - dragOffsetX);
                        circle2Y = snap(worldY - dragOffsetY);
                    }
                } else if (panning) {
                    int dx = screenX - lastPanX;
                    int dy = screenY - lastPanY;

                    // Move camera opposite to drag
                    float newX = camera.position.x - dx * camera.zoom;
                    float newY = camera.position.y + dy * camera.zoom;

                    camera.position.set(snap(newX), snap(newY), 0);
                    camera.update();

                    lastPanX = screenX;
                    lastPanY = screenY;
                }
                return true;
            }

            @Override
            public boolean touchUp(int screenX, int screenY, int pointer, int button) {
                if (button == Input.Buttons.LEFT) {
                    draggingCircle = false;
                    selectedCircle = -1;
                } else if (button == Input.Buttons.RIGHT) {
                    panning = false;
                }
                return true;
            }

            @Override
            public boolean scrolled(int amount) {
                final float zoomFactor = 1.1f;

                int mx = Gdx.input.getX();
                int my = Gdx.input.getY();

                // World coords under mouse BEFORE zoom
                tmp.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp);
                float beforeX = tmp.x;
                float beforeY = tmp.y;

                // Apply zoom
                if (amount > 0) {
                    camera.zoom *= zoomFactor; // zoom out
                } else if (amount < 0) {
                    camera.zoom /= zoomFactor; // zoom in
                }
                camera.zoom = MathUtils.clamp(camera.zoom, 0.05f, 20f);
                camera.update();

                // World coords under mouse AFTER zoom
                tmp2.set(mx, Gdx.graphics.getHeight() - my, 0);
                camera.unproject(tmp2);

                // Adjust camera so mouse focus stays fixed
                camera.position.add(beforeX - tmp2.x, beforeY - tmp2.y, 0);
                camera.update();

                return true;
            }
        });
    }

    private boolean isInsideCircle(float px, float py, float cx, float cy, float r) {
        float dx = px - cx;
        float dy = py - cy;
        return dx * dx + dy * dy <= r * r;
    }

    private float snap(float value) {
        if (!snapEnabled) return value;
        return Math.round(value / gridSize) * gridSize;
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.1f, 0.1f, 0.3f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        camera.update();
        shapeRenderer.setProjectionMatrix(camera.combined);

        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);

        // Circle 1: red
        shapeRenderer.setColor(1, 0, 0, 1);
        shapeRenderer.circle(circle1X, circle1Y, radius1);

        // Circle 2: green
        shapeRenderer.setColor(0, 1, 0, 1);
        shapeRenderer.circle(circle2X, circle2Y, radius2);

        shapeRenderer.end();

        // Draw grid
        drawGrid();
    }

    private void drawGrid() {
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.setColor(0.3f, 0.3f, 0.3f, 1);

        float left = camera.position.x - camera.viewportWidth * camera.zoom / 2f;
        float right = camera.position.x + camera.viewportWidth * camera.zoom / 2f;
        float bottom = camera.position.y - camera.viewportHeight * camera.zoom / 2f;
        float top = camera.position.y + camera.viewportHeight * camera.zoom / 2f;

        // Vertical lines
        for (float x = snap(left); x <= right; x += gridSize) {
            shapeRenderer.line(x, bottom, x, top);
        }
        // Horizontal lines
        for (float y = snap(bottom); y <= top; y += gridSize) {
            shapeRenderer.line(left, y, right, y);
        }

        shapeRenderer.end();
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("LibGDX + JFrame Drag + Zoom + Pan + Snap Grid");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(800, 600);

            LwjglAWTCanvas canvas = new LwjglAWTCanvas(new LibGdxJFrameSnapGridExample());
            frame.add(canvas.getCanvas());

            frame.setVisible(true);
        });
    }
}