package test;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.backends.lwjgl.LwjglAWTCanvas;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Vector2;

public class GridCircleApp extends ApplicationAdapter {
    private static final int GRID_WIDTH = 10;
    private static final int GRID_HEIGHT = 10;
    private static final float CELL_SIZE = 50f;
    private static final float CIRCLE_RADIUS = CELL_SIZE * 0.4f;
    private static final float HIGHLIGHT_RADIUS = CELL_SIZE * 0.45f; // Slightly larger for border effect

    private ShapeRenderer shapeRenderer;
    private Vector2 circle1;
    private Vector2 circle2;
    private int selectedCircle = 0; // 0 = none, 1 = circle1, 2 = circle2

    @Override
    public void create() {
        shapeRenderer = new ShapeRenderer();
        circle1 = new Vector2(2, 2);
        circle2 = new Vector2(7, 7);
    }

    @Override
    public void render() {
        handleInput();
        draw();
    }

    private void handleInput() {
        // Handle mouse/touch selection
        if (Gdx.input.justTouched()) {
            Vector2 touchPos = screenToWorld(Gdx.input.getX(), Gdx.input.getY());
            if (isPointInCircle(touchPos, circle1)) {
                selectedCircle = 1;
            } else if (isPointInCircle(touchPos, circle2)) {
                selectedCircle = 2;
            }
        }

        // Handle movement for selected circle
        if (selectedCircle != 0) {
            Vector2 target = (selectedCircle == 1) ? circle1 : circle2;
            float speed = 5f; // Grid units per second

            if (Gdx.input.isKeyPressed(Input.Keys.UP)) target.y += speed * Gdx.graphics.getDeltaTime();
            if (Gdx.input.isKeyPressed(Input.Keys.DOWN)) target.y -= speed * Gdx.graphics.getDeltaTime();
            if (Gdx.input.isKeyPressed(Input.Keys.LEFT)) target.x -= speed * Gdx.graphics.getDeltaTime();
            if (Gdx.input.isKeyPressed(Input.Keys.RIGHT)) target.x += speed * Gdx.graphics.getDeltaTime();

            // Keep within grid bounds
            target.x = Math.max(0, Math.min(GRID_WIDTH - 1, target.x));
            target.y = Math.max(0, Math.min(GRID_HEIGHT - 1, target.y));
        }
    }

    private Vector2 screenToWorld(float screenX, float screenY) {
        // Convert screen coordinates to world coordinates
        float worldX = screenX;
        float worldY = Gdx.graphics.getHeight() - screenY; // Flip Y-axis
        return new Vector2(worldX, worldY);
    }

    private boolean isPointInCircle(Vector2 point, Vector2 circleCenter) {
        float worldX = (circleCenter.x + 0.5f) * CELL_SIZE;
        float worldY = (circleCenter.y + 0.5f) * CELL_SIZE;
        float distance = point.dst(worldX, worldY);
        return distance <= CIRCLE_RADIUS;
    }

    private void draw() {
        Gdx.gl.glClearColor(0.1f, 0.1f, 0.1f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);

        // Draw grid
        shapeRenderer.setColor(Color.LIGHT_GRAY);
        for (int x = 0; x <= GRID_WIDTH; x++) {
            shapeRenderer.rectLine(x * CELL_SIZE, 0, x * CELL_SIZE, GRID_HEIGHT * CELL_SIZE, 1f);
        }
        for (int y = 0; y <= GRID_HEIGHT; y++) {
            shapeRenderer.rectLine(0, y * CELL_SIZE, GRID_WIDTH * CELL_SIZE, y * CELL_SIZE, 1f);
        }

        // Draw circles with selection highlights
        drawCircle(circle1, Color.BLUE, selectedCircle == 1);
        drawCircle(circle2, Color.RED, selectedCircle == 2);

        shapeRenderer.end();
    }

    private void drawCircle(Vector2 position, Color baseColor, boolean isSelected) {
        float worldX = (position.x + 0.5f) * CELL_SIZE;
        float worldY = (position.y + 0.5f) * CELL_SIZE;

        // Draw highlight border if selected
        if (isSelected) {
            shapeRenderer.setColor(Color.WHITE);
            shapeRenderer.circle(worldX, worldY, HIGHLIGHT_RADIUS);
        }

        // Draw main circle
        shapeRenderer.setColor(baseColor);
        shapeRenderer.circle(worldX, worldY, CIRCLE_RADIUS);
    }

    @Override
    public void dispose() {
        shapeRenderer.dispose();
    }
    public static void main(String[] args) {
    SwingUtilities.invokeLater(() -> {
        JFrame frame = new JFrame("LibGDX + JFrame + Grid + Hint");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setSize(800, 600);

        LwjglAWTCanvas canvas = new LwjglAWTCanvas(new GridCircleApp());
        frame.add(canvas.getCanvas());

//        frame.addComponentListener(new ComponentAdapter() {
//            @Override
//            public void componentResized(ComponentEvent e) {
//                int newWidth = frame.getContentPane().getWidth();
//                int newHeight = frame.getContentPane().getHeight();
//
//                canvas.getCanvas().setSize(newWidth, newHeight); // adjust canvas size
//                canvas.getApplicationListener().resize(newWidth, newHeight); // notify LibGDX
//            }
//        });
        frame.setVisible(true);
    });
    }
}